/**
 * MCP Server for Vibe Extension
 * 
 * Main server module that manages WebSocket connections and routes requests.
 * Uses Native Messaging to communicate with a Node.js process that runs the actual WebSocket server.
 */

import {
  McpServerSettings,
  McpServerState,
  JsonRpcRequest,
  JsonRpcResponse,
  NativeMessage,
  NativeCommand,
  ConnectedClient,
} from './types';
import { McpProtocolHandler, getProtocolHandler } from './protocol-handler';
import { getToolRegistry } from './tool-adapter';

// Storage key for server settings
const STORAGE_KEY = 'mcp_server_settings';

/**
 * MCP Server Manager
 * 
 * Manages the MCP server lifecycle and communication with the native WebSocket server.
 */
export class McpServer {
  private protocolHandler: McpProtocolHandler;
  private state: McpServerState = {
    running: false,
    port: 19989,
    host: '127.0.0.1',
    clients: [],
  };
  private nativePort: chrome.runtime.Port | null = null;
  private stateListeners: Set<(state: McpServerState) => void> = new Set();
  
  constructor() {
    this.protocolHandler = getProtocolHandler();
  }
  
  /**
   * Start the MCP server
   */
  async start(settings?: Partial<McpServerSettings>): Promise<void> {
    if (this.state.running) {
      console.log('[MCP Server] Already running');
      return;
    }
    
    const savedSettings = await this.getSettings();
    const mergedSettings: McpServerSettings = {
      enabled: true,
      port: settings?.port ?? savedSettings?.port ?? 19989,
      host: settings?.host ?? savedSettings?.host ?? '127.0.0.1',
      token: settings?.token ?? savedSettings?.token,
      exposedTools: settings?.exposedTools ?? savedSettings?.exposedTools,
    };
    
    console.log('[MCP Server] Starting on', mergedSettings.host, ':', mergedSettings.port);
    
    // For now, implement a simple in-extension WebSocket-like message handler
    // In a full implementation, this would use Native Messaging to a Node.js process
    // that runs the actual WebSocket server
    
    this.state = {
      running: true,
      port: mergedSettings.port,
      host: mergedSettings.host,
      clients: [],
      startedAt: Date.now(),
    };
    
    // Save settings
    await this.saveSettings(mergedSettings);
    
    this.notifyStateChange();
    console.log('[MCP Server] Started');
  }
  
  /**
   * Stop the MCP server
   */
  async stop(): Promise<void> {
    if (!this.state.running) {
      console.log('[MCP Server] Already stopped');
      return;
    }
    
    console.log('[MCP Server] Stopping...');
    
    // Disconnect all clients
    for (const client of this.state.clients) {
      this.protocolHandler.onClientDisconnect(client.id);
    }
    
    // Disconnect native port if connected
    if (this.nativePort) {
      this.nativePort.disconnect();
      this.nativePort = null;
    }
    
    this.state = {
      running: false,
      port: this.state.port,
      host: this.state.host,
      clients: [],
    };
    
    this.notifyStateChange();
    console.log('[MCP Server] Stopped');
  }
  
  /**
   * Handle a message from a client (for testing or internal use)
   */
  async handleMessage(request: JsonRpcRequest, clientId: string): Promise<JsonRpcResponse> {
    // Update client activity
    const clientIndex = this.state.clients.findIndex(c => c.id === clientId);
    if (clientIndex >= 0) {
      this.state.clients[clientIndex].lastActivity = Date.now();
    }
    
    return this.protocolHandler.handleRequest(request, clientId);
  }
  
  /**
   * Simulate a client connection (for testing)
   */
  connectClient(clientId: string): void {
    if (!this.state.running) {
      throw new Error('Server not running');
    }
    
    const client: ConnectedClient = {
      id: clientId,
      connectedAt: Date.now(),
      lastActivity: Date.now(),
    };
    
    this.state.clients.push(client);
    this.notifyStateChange();
    console.log('[MCP Server] Client connected:', clientId);
  }
  
  /**
   * Simulate a client disconnection (for testing)
   */
  disconnectClient(clientId: string): void {
    const index = this.state.clients.findIndex(c => c.id === clientId);
    if (index >= 0) {
      this.state.clients.splice(index, 1);
      this.protocolHandler.onClientDisconnect(clientId);
      this.notifyStateChange();
      console.log('[MCP Server] Client disconnected:', clientId);
    }
  }
  
  /**
   * Get current server state
   */
  getState(): McpServerState {
    return { ...this.state };
  }
  
  /**
   * Subscribe to state changes
   */
  onStateChange(listener: (state: McpServerState) => void): () => void {
    this.stateListeners.add(listener);
    return () => this.stateListeners.delete(listener);
  }
  
  /**
   * Get tool registry
   */
  getToolRegistry() {
    return getToolRegistry();
  }
  
  /**
   * Get protocol handler
   */
  getProtocolHandler() {
    return this.protocolHandler;
  }
  
  /**
   * Get saved settings
   */
  async getSettings(): Promise<McpServerSettings | null> {
    try {
      const result = await chrome.storage.local.get(STORAGE_KEY);
      return result[STORAGE_KEY] || null;
    } catch {
      return null;
    }
  }
  
  /**
   * Save settings
   */
  async saveSettings(settings: McpServerSettings): Promise<void> {
    await chrome.storage.local.set({ [STORAGE_KEY]: settings });
  }
  
  /**
   * Notify state change listeners
   */
  private notifyStateChange(): void {
    const state = this.getState();
    for (const listener of this.stateListeners) {
      try {
        listener(state);
      } catch (error) {
        console.error('[MCP Server] State listener error:', error);
      }
    }
  }
}

// Singleton instance
let serverInstance: McpServer | null = null;

/**
 * Get the MCP server singleton
 */
export function getMcpServer(): McpServer {
  if (!serverInstance) {
    serverInstance = new McpServer();
  }
  return serverInstance;
}

// Re-export types and utilities
export { getToolRegistry, registerBrowserTools } from './tool-adapter';
export { getProtocolHandler } from './protocol-handler';
export * from './types';

// Re-export external client
export { McpExternalClient, getMcpExternalClient } from '../external-client';
