/**
 * MCP Manager - Manages multiple MCP server connections
 * 
 * Handles connecting to servers, loading tools, and converting them to BrowserTool format.
 */

import { z } from 'zod';
import { BrowserTool } from '../../ai_tools_interface.js';
import { McpClient, McpAuthError } from './client';
import { McpServerConfig, McpServerStatus, McpToolDefinition } from './types';
import { getEnabledMcpServers, getMcpConfig } from './config';

/**
 * MCP Manager singleton for managing all MCP connections
 */
export class McpManager {
  private clients: Map<string, McpClient> = new Map();
  private status: Map<string, McpServerStatus> = new Map();
  private tools: Map<string, BrowserTool[]> = new Map();
  private initialized: boolean = false;

  /**
   * Initialize all enabled MCP servers
   */
  async initialize(): Promise<void> {
    if (this.initialized) {
      return;
    }

    console.log('[MCP Manager] Initializing...');
    
    const servers = await getEnabledMcpServers();
    
    await Promise.all(
      servers.map(async (server) => {
        await this.connectServer(server);
      })
    );

    this.initialized = true;
    console.log('[MCP Manager] Initialized with', this.clients.size, 'servers');
  }

  /**
   * Connect to a single MCP server
   */
  async connectServer(config: McpServerConfig): Promise<McpServerStatus> {
    const client = new McpClient(config);
    this.clients.set(config.name, client);
    this.status.set(config.name, { status: 'connecting' });

    const status = await client.connect();
    this.status.set(config.name, status);

    if (status.status === 'connected') {
      // Load tools from the server
      try {
        const serverTools = await client.listTools();
        const browserTools = this.convertToolsToBrowserTools(config.name, client, serverTools);
        this.tools.set(config.name, browserTools);
        console.log(`[MCP Manager] Loaded ${browserTools.length} tools from ${config.name}`);
      } catch (error) {
        console.error(`[MCP Manager] Failed to load tools from ${config.name}:`, error);
      }
    }

    return status;
  }

  /**
   * Disconnect from a server
   */
  async disconnectServer(name: string): Promise<void> {
    const client = this.clients.get(name);
    if (client) {
      await client.disconnect();
      this.clients.delete(name);
      this.status.set(name, { status: 'disabled' });
      this.tools.delete(name);
    }
  }

  /**
   * Refresh tools from all connected servers
   */
  async refreshTools(): Promise<void> {
    for (const [name, client] of this.clients) {
      if (client.isConnected) {
        try {
          const serverTools = await client.listTools();
          const browserTools = this.convertToolsToBrowserTools(name, client, serverTools);
          this.tools.set(name, browserTools);
        } catch (error) {
          console.error(`[MCP Manager] Failed to refresh tools from ${name}:`, error);
        }
      }
    }
  }

  /**
   * Get all MCP tools as BrowserTools
   */
  getAllTools(): BrowserTool[] {
    const allTools: BrowserTool[] = [];
    for (const tools of this.tools.values()) {
      allTools.push(...tools);
    }
    return allTools;
  }

  /**
   * Get status of all servers
   */
  getStatus(): Record<string, McpServerStatus> {
    const result: Record<string, McpServerStatus> = {};
    for (const [name, status] of this.status) {
      result[name] = status;
    }
    return result;
  }

  /**
   * Convert MCP tool definitions to BrowserTool instances
   */
  private convertToolsToBrowserTools(
    serverName: string, 
    client: McpClient, 
    tools: McpToolDefinition[]
  ): BrowserTool[] {
    return tools.map(tool => {
      // Create unique tool name with server prefix
      const sanitizedServerName = serverName.replace(/[^a-zA-Z0-9_-]/g, '_');
      const sanitizedToolName = tool.name.replace(/[^a-zA-Z0-9_-]/g, '_');
      const uniqueName = `mcp_${sanitizedServerName}_${sanitizedToolName}`;

      // Convert JSON Schema to Zod schema
      const zodSchema = this.jsonSchemaToZod(tool.inputSchema);

      return new McpBrowserTool(
        uniqueName,
        tool.description || `MCP tool: ${tool.name}`,
        zodSchema,
        client,
        tool.name // Original tool name for MCP call
      );
    });
  }

  /**
   * Convert JSON Schema to Zod schema
   * Simple conversion for common types
   */
  private jsonSchemaToZod(schema: McpToolDefinition['inputSchema']): z.ZodObject<any> {
    const properties: Record<string, z.ZodTypeAny> = {};
    
    if (schema.properties) {
      for (const [key, prop] of Object.entries(schema.properties)) {
        properties[key] = this.jsonPropertyToZod(prop as JsonSchemaProperty, key);
      }
    }

    const zodObject = z.object(properties);
    
    // Make non-required fields optional
    if (schema.required && schema.required.length > 0) {
      const requiredSet = new Set(schema.required);
      const partialProps: Record<string, z.ZodTypeAny> = {};
      
      for (const [key, zodProp] of Object.entries(properties)) {
        if (requiredSet.has(key)) {
          partialProps[key] = zodProp;
        } else {
          partialProps[key] = zodProp.optional();
        }
      }
      
      return z.object(partialProps);
    }

    return zodObject;
  }

  /**
   * Convert a single JSON Schema property to Zod
   */
  private jsonPropertyToZod(prop: JsonSchemaProperty, name: string): z.ZodTypeAny {
    const description = prop.description || name;

    switch (prop.type) {
      case 'string':
        if (prop.enum) {
          return z.enum(prop.enum as [string, ...string[]]).describe(description);
        }
        return z.string().describe(description);
      
      case 'number':
      case 'integer':
        return z.number().describe(description);
      
      case 'boolean':
        return z.boolean().describe(description);
      
      case 'array':
        if (prop.items) {
          const itemSchema = this.jsonPropertyToZod(prop.items as JsonSchemaProperty, 'item');
          return z.array(itemSchema).describe(description);
        }
        return z.array(z.any()).describe(description);
      
      case 'object':
        if (prop.properties) {
          const nestedProps: Record<string, z.ZodTypeAny> = {};
          for (const [key, nested] of Object.entries(prop.properties)) {
            nestedProps[key] = this.jsonPropertyToZod(nested as JsonSchemaProperty, key);
          }
          return z.object(nestedProps).describe(description);
        }
        return z.record(z.any()).describe(description);
      
      default:
        return z.any().describe(description);
    }
  }

  /**
   * Shutdown all connections
   */
  async shutdown(): Promise<void> {
    for (const [name, client] of this.clients) {
      await client.disconnect();
    }
    this.clients.clear();
    this.status.clear();
    this.tools.clear();
    this.initialized = false;
  }
}

interface JsonSchemaProperty {
  type?: string;
  description?: string;
  enum?: string[];
  items?: unknown;
  properties?: Record<string, unknown>;
  required?: string[];
}

/**
 * BrowserTool wrapper for MCP tools
 */
class McpBrowserTool extends BrowserTool {
  private client: McpClient;
  private originalName: string;

  constructor(
    name: string,
    description: string,
    schema: z.ZodObject<any>,
    client: McpClient,
    originalName: string
  ) {
    super(name, description, schema);
    this.client = client;
    this.originalName = originalName;
  }

  async call(args: Record<string, unknown>): Promise<string> {
    try {
      const result = await this.client.callTool(this.originalName, args);
      
      // Format the result
      if (result.isError) {
        const errorText = result.content
          .filter(c => c.type === 'text')
          .map(c => c.text)
          .join('\n');
        throw new Error(errorText || 'MCP tool execution failed');
      }

      // Combine all text content
      const textContent = result.content
        .filter(c => c.type === 'text')
        .map(c => c.text)
        .join('\n');

      // Handle image content
      const imageContent = result.content
        .filter(c => c.type === 'image')
        .map(c => `[Image: ${c.mimeType}]`)
        .join('\n');

      return [textContent, imageContent].filter(Boolean).join('\n') || 'Tool executed successfully';
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      throw new Error(`MCP tool ${this.originalName} failed: ${errorMessage}`);
    }
  }
}

// Singleton instance
let mcpManagerInstance: McpManager | null = null;

/**
 * Get the MCP Manager singleton
 */
export function getMcpManager(): McpManager {
  if (!mcpManagerInstance) {
    mcpManagerInstance = new McpManager();
  }
  return mcpManagerInstance;
}

/**
 * Initialize MCP and get all tools
 * This is the main entry point for getting MCP tools
 */
export async function getMcpTools(): Promise<BrowserTool[]> {
  const manager = getMcpManager();
  await manager.initialize();
  return manager.getAllTools();
}
