/**
 * MCP External Client
 * 
 * WebSocket client that connects to an external MCP server (like vibe-mcp)
 * to allow AI agents to control the browser through the Vibe extension.
 * 
 * Note: Using port 19889 to avoid conflict with Playwriter MCP (uses 19988/19989)
 */

import { getToolRegistry } from './server/tool-adapter';

const DEFAULT_MCP_PORT = 19889; // Avoid conflict with Playwriter MCP (19988/19989)
const RECONNECT_INTERVAL_MS = 30000; // Reconnect every 30 seconds

/**
 * Message from MCP server to extension
 */
interface ServerMessage {
  type: 'list_tools' | 'call_tool' | 'get_snapshot' | 'ping';
  requestId: string;
  data?: {
    name?: string;
    arguments?: Record<string, unknown>;
  };
}

/**
 * Message from extension to MCP server
 */
interface ExtensionMessage {
  type: 'connected' | 'disconnected' | 'tool_result' | 'tools_list' | 'error' | 'snapshot';
  requestId?: string;
  data?: unknown;
  error?: string;
}

/**
 * MCP External Connection State
 */
interface McpExternalState {
  connected: boolean;
  port: number;
  reconnectAttempts: number;
  lastConnectedAt?: number;
  lastError?: string;
}

/**
 * MCP External Client
 * 
 * Connects to an external MCP server and handles tool execution requests.
 */
export class McpExternalClient {
  private ws: WebSocket | null = null;
  private state: McpExternalState = {
    connected: false,
    port: DEFAULT_MCP_PORT,
    reconnectAttempts: 0,
  };
  private reconnectTimer: ReturnType<typeof setTimeout> | null = null;
  private stateListeners: Set<(state: McpExternalState) => void> = new Set();
  private toolExecutor: ((name: string, args: Record<string, unknown>) => Promise<string>) | null = null;

  /**
   * Connect to the MCP server
   */
  async connect(port: number = DEFAULT_MCP_PORT): Promise<void> {
    if (this.ws && this.state.connected) {
      console.log('[MCP External] Already connected');
      return;
    }

    this.state.port = port;
    return this.doConnect();
  }

  /**
   * Perform the actual connection
   */
  private doConnect(): Promise<void> {
    return new Promise((resolve, reject) => {
      const url = `ws://127.0.0.1:${this.state.port}`;
      console.log(`[MCP External] Connecting to ${url}...`);

      try {
        this.ws = new WebSocket(url);

        this.ws.onopen = () => {
          console.log('[MCP External] Connected');
          this.state.connected = true;
          this.state.reconnectAttempts = 0;
          this.state.lastConnectedAt = Date.now();
          this.state.lastError = undefined;
          this.notifyStateChange();

          // Send connected message
          this.send({ type: 'connected' });
          resolve();
        };

        this.ws.onmessage = (event) => {
          this.handleMessage(event.data);
        };

        this.ws.onclose = () => {
          console.log('[MCP External] Disconnected');
          this.ws = null;
          this.state.connected = false;
          this.notifyStateChange();

          // Always attempt reconnect when enabled (every 30s)
          this.scheduleReconnect();
        };

        this.ws.onerror = (error) => {
          console.error('[MCP External] WebSocket error:', error);
          this.state.lastError = 'Connection error';
          this.notifyStateChange();
          reject(new Error('WebSocket connection failed'));
        };

      } catch (error) {
        const message = error instanceof Error ? error.message : String(error);
        this.state.lastError = message;
        this.notifyStateChange();
        reject(error);
      }
    });
  }

  /**
   * Schedule a reconnection attempt (every 30 seconds)
   */
  private scheduleReconnect(): void {
    if (this.reconnectTimer) {
      clearTimeout(this.reconnectTimer);
    }

    this.state.reconnectAttempts++;
    
    console.log(`[MCP External] Will reconnect in ${RECONNECT_INTERVAL_MS / 1000}s (attempt ${this.state.reconnectAttempts})...`);
    
    this.reconnectTimer = setTimeout(() => {
      this.doConnect().catch((error) => {
        console.log('[MCP External] Reconnect failed, will retry:', error.message || error);
      });
    }, RECONNECT_INTERVAL_MS);
  }

  /**
   * Disconnect from the MCP server
   */
  disconnect(): void {
    if (this.reconnectTimer) {
      clearTimeout(this.reconnectTimer);
      this.reconnectTimer = null;
    }

    if (this.ws) {
      this.ws.close();
      this.ws = null;
    }

    this.state.connected = false;
    this.state.reconnectAttempts = 0;
    this.notifyStateChange();
    console.log('[MCP External] Disconnected');
  }

  /**
   * Handle incoming message from MCP server
   */
  private async handleMessage(data: string): Promise<void> {
    try {
      const message: ServerMessage = JSON.parse(data);
      console.log('[MCP External] Received:', message.type);

      switch (message.type) {
        case 'list_tools':
          await this.handleListTools(message.requestId);
          break;

        case 'call_tool':
          if (message.data?.name) {
            await this.handleCallTool(
              message.requestId,
              message.data.name,
              message.data.arguments || {}
            );
          }
          break;

        case 'get_snapshot':
          await this.handleGetSnapshot(message.requestId);
          break;

        case 'ping':
          this.send({ type: 'connected', requestId: message.requestId });
          break;

        default:
          console.warn('[MCP External] Unknown message type:', message.type);
      }
    } catch (error) {
      console.error('[MCP External] Failed to handle message:', error);
    }
  }

  /**
   * Handle list tools request
   */
  private async handleListTools(requestId: string): Promise<void> {
    try {
      const registry = getToolRegistry();
      const tools = registry.getDefinitions();
      
      this.send({
        type: 'tools_list',
        requestId,
        data: tools,
      });
    } catch (error) {
      const message = error instanceof Error ? error.message : String(error);
      this.send({
        type: 'error',
        requestId,
        error: message,
      });
    }
  }

  /**
   * Handle call tool request
   */
  private async handleCallTool(
    requestId: string,
    name: string,
    args: Record<string, unknown>
  ): Promise<void> {
    try {
      const registry = getToolRegistry();
      const result = await registry.execute(name, args);
      
      this.send({
        type: 'tool_result',
        requestId,
        data: result,
      });
    } catch (error) {
      const message = error instanceof Error ? error.message : String(error);
      this.send({
        type: 'error',
        requestId,
        error: message,
      });
    }
  }

  /**
   * Handle get snapshot request
   */
  private async handleGetSnapshot(requestId: string): Promise<void> {
    try {
      // Get current tab info
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      
      if (!tab?.id) {
        throw new Error('No active tab');
      }

      // Execute script to get page info
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        func: () => {
          // Get accessibility tree snapshot (simplified)
          const getAccessibilityTree = (element: Element, depth = 0): string => {
            if (depth > 5) return '';
            
            const role = element.getAttribute('role') || element.tagName.toLowerCase();
            const name = element.getAttribute('aria-label') || 
                        element.getAttribute('title') || 
                        (element as HTMLElement).innerText?.slice(0, 50);
            
            let result = `${'  '.repeat(depth)}${role}`;
            if (name) result += `: "${name.trim()}"`;
            result += '\n';
            
            for (const child of element.children) {
              result += getAccessibilityTree(child, depth + 1);
            }
            
            return result;
          };

          return {
            title: document.title,
            snapshot: getAccessibilityTree(document.body),
          };
        },
      });

      const pageInfo = results[0]?.result || { title: '', snapshot: '' };

      this.send({
        type: 'snapshot',
        requestId,
        data: {
          url: tab.url || '',
          title: pageInfo.title || tab.title || '',
          snapshot: pageInfo.snapshot || '',
        },
      });
    } catch (error) {
      const message = error instanceof Error ? error.message : String(error);
      this.send({
        type: 'error',
        requestId,
        error: message,
      });
    }
  }

  /**
   * Send message to MCP server
   */
  private send(message: ExtensionMessage): void {
    if (this.ws && this.state.connected) {
      this.ws.send(JSON.stringify(message));
    }
  }

  /**
   * Set tool executor function
   */
  setToolExecutor(executor: (name: string, args: Record<string, unknown>) => Promise<string>): void {
    this.toolExecutor = executor;
  }

  /**
   * Get current connection state
   */
  getState(): McpExternalState {
    return { ...this.state };
  }

  /**
   * Check if connected
   */
  isConnected(): boolean {
    return this.state.connected;
  }

  /**
   * Subscribe to state changes
   */
  onStateChange(listener: (state: McpExternalState) => void): () => void {
    this.stateListeners.add(listener);
    return () => this.stateListeners.delete(listener);
  }

  /**
   * Notify state change listeners
   */
  private notifyStateChange(): void {
    const state = this.getState();
    for (const listener of this.stateListeners) {
      try {
        listener(state);
      } catch (error) {
        console.error('[MCP External] State listener error:', error);
      }
    }
  }
}

// Singleton instance
let clientInstance: McpExternalClient | null = null;

/**
 * Get the MCP external client singleton
 */
export function getMcpExternalClient(): McpExternalClient {
  if (!clientInstance) {
    clientInstance = new McpExternalClient();
  }
  return clientInstance;
}
