/**
 * MCP Configuration Storage for Chrome Extension
 * 
 * Manages MCP server configurations and OAuth tokens in chrome.storage.local
 */

import { 
  McpConfig, 
  McpConfigSchema, 
  McpServerConfig, 
  McpAuthEntry,
  McpOAuthTokens 
} from './types';

const MCP_CONFIG_KEY = 'vibe_mcp_config';
const MCP_AUTH_KEY = 'vibe_mcp_auth';

/**
 * Get MCP configuration from storage
 */
export async function getMcpConfig(): Promise<McpConfig> {
  try {
    const result = await chrome.storage.local.get(MCP_CONFIG_KEY);
    if (result[MCP_CONFIG_KEY]) {
      return McpConfigSchema.parse(result[MCP_CONFIG_KEY]);
    }
    return { servers: {} };
  } catch (error) {
    console.error('[MCP] Failed to get config:', error);
    return { servers: {} };
  }
}

/**
 * Save MCP configuration to storage
 */
export async function saveMcpConfig(config: McpConfig): Promise<void> {
  await chrome.storage.local.set({ [MCP_CONFIG_KEY]: config });
}

/**
 * Add or update an MCP server configuration
 */
export async function addMcpServer(server: McpServerConfig): Promise<void> {
  const config = await getMcpConfig();
  config.servers[server.name] = server;
  await saveMcpConfig(config);
}

/**
 * Remove an MCP server configuration
 */
export async function removeMcpServer(name: string): Promise<void> {
  const config = await getMcpConfig();
  delete config.servers[name];
  await saveMcpConfig(config);
  
  // Also remove auth tokens
  await removeMcpAuth(name);
}

/**
 * Enable or disable an MCP server
 */
export async function setMcpServerEnabled(name: string, enabled: boolean): Promise<void> {
  const config = await getMcpConfig();
  if (config.servers[name]) {
    config.servers[name].enabled = enabled;
    await saveMcpConfig(config);
  }
}

/**
 * Get all enabled MCP server configurations
 */
export async function getEnabledMcpServers(): Promise<McpServerConfig[]> {
  const config = await getMcpConfig();
  return Object.values(config.servers).filter(s => s.enabled);
}

// ============================================================================
// OAuth Token Storage
// ============================================================================

/**
 * Get all MCP auth entries
 */
async function getAllMcpAuth(): Promise<Record<string, McpAuthEntry>> {
  try {
    const result = await chrome.storage.local.get(MCP_AUTH_KEY);
    return result[MCP_AUTH_KEY] || {};
  } catch (error) {
    console.error('[MCP] Failed to get auth:', error);
    return {};
  }
}

/**
 * Get OAuth tokens for an MCP server
 */
export async function getMcpAuth(serverName: string): Promise<McpAuthEntry | undefined> {
  const auth = await getAllMcpAuth();
  return auth[serverName];
}

/**
 * Save OAuth tokens for an MCP server
 */
export async function saveMcpAuth(
  serverName: string, 
  tokens: McpOAuthTokens,
  serverUrl?: string
): Promise<void> {
  const auth = await getAllMcpAuth();
  auth[serverName] = {
    tokens,
    serverUrl,
  };
  await chrome.storage.local.set({ [MCP_AUTH_KEY]: auth });
}

/**
 * Remove OAuth tokens for an MCP server
 */
export async function removeMcpAuth(serverName: string): Promise<void> {
  const auth = await getAllMcpAuth();
  delete auth[serverName];
  await chrome.storage.local.set({ [MCP_AUTH_KEY]: auth });
}

/**
 * Check if OAuth tokens are expired
 */
export async function isTokenExpired(serverName: string): Promise<boolean | null> {
  const entry = await getMcpAuth(serverName);
  if (!entry?.tokens) return null;
  if (!entry.tokens.expiresAt) return false;
  return entry.tokens.expiresAt < Date.now() / 1000;
}
