/**
 * MCP Client for Chrome Extension
 * 
 * Lightweight MCP client implementation for browser environments.
 * Supports SSE (Server-Sent Events) transport for remote MCP servers.
 * 
 * Based on the MCP specification: https://spec.modelcontextprotocol.io/
 */

import { 
  McpServerConfig, 
  McpServerStatus, 
  McpToolDefinition, 
  McpToolResult 
} from './types';
import { getMcpAuth, saveMcpAuth } from './config';

const MCP_PROTOCOL_VERSION = '2024-11-05';

interface JsonRpcRequest {
  jsonrpc: '2.0';
  id: number | string;
  method: string;
  params?: Record<string, unknown>;
}

interface JsonRpcResponse {
  jsonrpc: '2.0';
  id: number | string;
  result?: unknown;
  error?: {
    code: number;
    message: string;
    data?: unknown;
  };
}

interface McpInitializeResult {
  protocolVersion: string;
  capabilities: {
    tools?: { listChanged?: boolean };
    resources?: { subscribe?: boolean; listChanged?: boolean };
    prompts?: { listChanged?: boolean };
  };
  serverInfo: {
    name: string;
    version: string;
  };
}

interface McpListToolsResult {
  tools: McpToolDefinition[];
}

interface McpCallToolResult {
  content: Array<{
    type: 'text' | 'image' | 'resource';
    text?: string;
    data?: string;
    mimeType?: string;
  }>;
  isError?: boolean;
}

/**
 * MCP Client for connecting to remote MCP servers
 */
export class McpClient {
  private serverConfig: McpServerConfig;
  private requestId: number = 0;
  private initialized: boolean = false;
  private serverInfo: McpInitializeResult['serverInfo'] | null = null;
  private capabilities: McpInitializeResult['capabilities'] = {};
  private sessionId: string | null = null;

  constructor(config: McpServerConfig) {
    this.serverConfig = config;
  }

  /**
   * Get the server name
   */
  get name(): string {
    return this.serverConfig.name;
  }

  /**
   * Check if connected and initialized
   */
  get isConnected(): boolean {
    return this.initialized;
  }

  /**
   * Make a JSON-RPC request to the MCP server
   */
  private async request<T>(method: string, params?: Record<string, unknown>): Promise<T> {
    const id = ++this.requestId;
    const request: JsonRpcRequest = {
      jsonrpc: '2.0',
      id,
      method,
      params,
    };

    return this.sendRequest<T>(request, id);
  }

  /**
   * Send a notification (no response expected)
   */
  private async notify(method: string, params?: Record<string, unknown>): Promise<void> {
    const request = {
      jsonrpc: '2.0' as const,
      method,
      params,
    };

    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
      'Accept': 'application/json, text/event-stream',
      ...this.serverConfig.headers,
    };

    if (this.sessionId) {
      headers['Mcp-Session-Id'] = this.sessionId;
    }

    const auth = await getMcpAuth(this.serverConfig.name);
    if (auth?.tokens?.accessToken) {
      headers['Authorization'] = `Bearer ${auth.tokens.accessToken}`;
    }

    // Fire and forget - notifications don't expect a response
    await fetch(this.serverConfig.url, {
      method: 'POST',
      headers,
      body: JSON.stringify(request),
    });
  }

  /**
   * Send a request and wait for response
   */
  private async sendRequest<T>(request: JsonRpcRequest, id: number | string): Promise<T> {
    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
      'Accept': 'application/json, text/event-stream',
      ...this.serverConfig.headers,
    };

    // Add session ID if we have one (required for stateful MCP servers)
    if (this.sessionId) {
      headers['Mcp-Session-Id'] = this.sessionId;
    }

    // Add authorization header if we have tokens
    const auth = await getMcpAuth(this.serverConfig.name);
    if (auth?.tokens?.accessToken) {
      headers['Authorization'] = `Bearer ${auth.tokens.accessToken}`;
    }

    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), this.serverConfig.timeout || 30000);

    try {
      const response = await fetch(this.serverConfig.url, {
        method: 'POST',
        headers,
        body: JSON.stringify(request),
        signal: controller.signal,
      });

      clearTimeout(timeoutId);

      if (response.status === 401) {
        throw new McpAuthError('Authentication required');
      }

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      // Capture session ID from response header (required for stateful MCP servers)
      const newSessionId = response.headers.get('mcp-session-id');
      if (newSessionId) {
        this.sessionId = newSessionId;
      }

      const contentType = response.headers.get('content-type') || '';
      
      // Handle SSE response
      if (contentType.includes('text/event-stream')) {
        return await this.handleSSEResponse<T>(response, id);
      }

      // Handle regular JSON response
      const json = await response.json() as JsonRpcResponse;
      
      if (json.error) {
        throw new Error(`MCP Error ${json.error.code}: ${json.error.message}`);
      }

      return json.result as T;
    } catch (error) {
      clearTimeout(timeoutId);
      if ((error as Error).name === 'AbortError') {
        throw new Error(`Request timeout after ${this.serverConfig.timeout}ms`);
      }
      throw error;
    }
  }

  /**
   * Handle Server-Sent Events response
   */
  private async handleSSEResponse<T>(response: Response, expectedId: number | string): Promise<T> {
    const reader = response.body?.getReader();
    if (!reader) {
      throw new Error('No response body');
    }

    const decoder = new TextDecoder();
    let buffer = '';

    try {
      while (true) {
        const { done, value } = await reader.read();
        
        if (done) {
          throw new Error('SSE stream ended without response');
        }

        buffer += decoder.decode(value, { stream: true });
        const lines = buffer.split('\n');
        buffer = lines.pop() || '';

        for (const line of lines) {
          if (line.startsWith('data: ')) {
            const data = line.slice(6);
            try {
              const json = JSON.parse(data) as JsonRpcResponse;
              if (json.id === expectedId) {
                if (json.error) {
                  throw new Error(`MCP Error ${json.error.code}: ${json.error.message}`);
                }
                return json.result as T;
              }
            } catch (parseError) {
              // Continue if JSON parse fails - might be partial data
              if (!(parseError instanceof SyntaxError)) {
                throw parseError;
              }
            }
          }
        }
      }
    } finally {
      reader.releaseLock();
    }
  }

  /**
   * Connect to the MCP server and initialize
   */
  async connect(): Promise<McpServerStatus> {
    if (!this.serverConfig.enabled) {
      return { status: 'disabled' };
    }

    try {
      console.log(`[MCP] Connecting to ${this.serverConfig.name}...`);

      const result = await this.request<McpInitializeResult>('initialize', {
        protocolVersion: MCP_PROTOCOL_VERSION,
        capabilities: {
          roots: { listChanged: true },
        },
        clientInfo: {
          name: 'Vibe AI Browser',
          version: '1.0.0',
        },
      });

      this.serverInfo = result.serverInfo;
      this.capabilities = result.capabilities;
      this.initialized = true;

      // Send initialized notification (no response expected)
      await this.notify('notifications/initialized');

      console.log(`[MCP] Connected to ${this.serverConfig.name}:`, {
        server: result.serverInfo,
        capabilities: result.capabilities,
      });

      return { status: 'connected' };
    } catch (error) {
      console.error(`[MCP] Failed to connect to ${this.serverConfig.name}:`, error);
      
      if (error instanceof McpAuthError) {
        return { status: 'needs_auth' };
      }

      return { 
        status: 'failed', 
        error: error instanceof Error ? error.message : String(error) 
      };
    }
  }

  /**
   * List available tools from the server
   */
  async listTools(): Promise<McpToolDefinition[]> {
    if (!this.initialized) {
      throw new Error('MCP client not initialized');
    }

    const result = await this.request<McpListToolsResult>('tools/list', {});
    return result.tools;
  }

  /**
   * Call a tool on the server
   */
  async callTool(name: string, args: Record<string, unknown>): Promise<McpToolResult> {
    if (!this.initialized) {
      throw new Error('MCP client not initialized');
    }

    const result = await this.request<McpCallToolResult>('tools/call', {
      name,
      arguments: args,
    });

    return result;
  }

  /**
   * Disconnect from the server
   */
  async disconnect(): Promise<void> {
    this.initialized = false;
    this.serverInfo = null;
    this.capabilities = {};
    this.sessionId = null;
  }
}

/**
 * Custom error for authentication issues
 */
export class McpAuthError extends Error {
  constructor(message: string) {
    super(message);
    this.name = 'McpAuthError';
  }
}
