// Shared utilities for browser tools

// Note: content.js is automatically injected by manifest.json on all web pages
// No runtime injection or verification is needed

// Restricted URL patterns where content scripts cannot be injected
const RESTRICTED_URL_PATTERNS = [
  'chrome-extension://',
  'chrome://',
  'edge://',
  'brave://',
  'about:',
  'data:',
  'file:',
  'devtools://',
  'view-source:',
  'chrome-search://'
];

// Check if a URL is restricted (content scripts cannot be injected)
export function isRestrictedUrl(url) {
  if (!url) return false;
  return RESTRICTED_URL_PATTERNS.some(pattern => url.startsWith(pattern));
}

// Utility function to get the current active tab
export async function getCurrentActiveTab() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (!tab) {
    throw new Error('No active tab found');
  }
  return tab;
}

// Utility function to validate tab exists and get valid tab ID
export async function getValidTabId(tabId) {
  try {
    // If no tabId provided, get active tab
    if (!tabId) {
      const activeTab = await getCurrentActiveTab();
      return activeTab.id;
    }

    // Try to validate the provided tab ID
    try {
      await chrome.tabs.get(tabId);
      return tabId; // Tab exists, return it
    } catch (tabError) {
      console.warn(`⚠️ Tab ${tabId} not found, falling back to active tab`);

      // Fallback to active tab
      const activeTab = await getCurrentActiveTab();
      console.log(`Using active tab ${activeTab.id} instead of invalid tab ${tabId}`);
      return activeTab.id;
    }
  } catch (error) {
    const errorMessage = error instanceof Error ? error.message : String(error);
    throw new Error(`Failed to get valid tab: ${errorMessage}`);
  }
}

// Check if content script can communicate on this tab
export async function ensureContentScriptCanCommunicate(tabId) {
  const tab = await chrome.tabs.get(tabId);
  if (isRestrictedUrl(tab?.url || "")) {
    throw new Error('Cannot interact with browser internal pages (chrome://, about://, etc.)');
  }
  return true;
}

// Simple utility to wait for page to be fully loaded
export async function waitForPageReady(targetTabId, maxWaitMs = 3000) {
  const startTime = Date.now();

  // Wait for tab status to be 'complete'
  // When tab.status === 'complete':
  // 1. DOM is fully parsed
  // 2. All resources (images, stylesheets) are loaded
  // Note: Our manifest uses "run_at": "document_end" which injects after DOM parsing.
  // However, tab.status === 'complete' doesn't guarantee content script is responding.
  // NavigationTools.js pings the content script separately to verify readiness.
  while (Date.now() - startTime < maxWaitMs) {
    try {
      const tab = await chrome.tabs.get(targetTabId);

      // Check if on restricted page
      if (isRestrictedUrl(tab?.url || "")) {
        // On restricted pages, just return immediately
        return false;
      }

      // Check if page is fully loaded
      if (tab.status === 'complete') {
        // Page is loaded, but content script may still be initializing
        return true;
      }

      // Page still loading, wait a bit
      await new Promise(resolve => setTimeout(resolve, 200));
    } catch (error) {
      console.warn(`Failed to get tab ${targetTabId} status:`, error);
      return false;
    }
  }

  console.warn(`Tab ${targetTabId} didn't reach complete status within ${maxWaitMs}ms`);
  return false;
}


// Parse role-based locators from string format
export function parseRoleLocator(selector) {
  // Match patterns like "role:button[name=Submit]" or "role:textbox"
  const roleMatch = selector.match(/role:(\w+)(?:\[(.+)\])?/);
  if (!roleMatch) return null;

  const [, role, attributes] = roleMatch;
  const options = { role };

  if (attributes) {
    // Parse attributes like name=Submit or multiple attributes
    const attrPairs = attributes.split(',').map(s => s.trim());
    for (const pair of attrPairs) {
      const [key, value] = pair.split('=').map(s => s.trim());
      if (key && value) {
        // Remove quotes if present
        options[key] = value.replace(/^["']|["']$/g, '');
      }
    }
  }

  return options;
}

// Convert various selector formats to a normalized locator object
export function normalizeLocator(selector) {
  if (!selector) return null;

  // Already an object
  if (typeof selector === 'object') {
    return selector;
  }

  // String selector
  if (typeof selector === 'string') {
    // Text selector
    if (selector.startsWith('text:')) {
      return { text: selector.substring(5) };
    }

    // Role selector
    if (selector.startsWith('role:')) {
      return parseRoleLocator(selector);
    }

    // Placeholder selector
    if (selector.startsWith('placeholder:')) {
      return { placeholder: selector.substring(12) };
    }

    // Label selector
    if (selector.startsWith('label:')) {
      return { label: selector.substring(6) };
    }

    // Default to CSS selector
    return { selector };
  }

  return null;
}

// Format error message with helpful suggestions
export function formatErrorMessage(error, action, selector) {
  const selectorDesc = typeof selector === 'object' ? JSON.stringify(selector) : selector;
  const errorMsg = error instanceof Error ? error.message : String(error);

  // Element not found
  if (errorMsg.includes('No element') || errorMsg.includes('not found')) {
    return `${action} failed: No elements found matching ${selectorDesc}. Try using find_element tool first to verify the selector.`;
  }

  // Multiple elements found
  if (errorMsg.includes('multiple elements') || errorMsg.includes('strict mode')) {
    const countMatch = errorMsg.match(/(\d+) elements?/);
    const count = countMatch ? countMatch[1] : 'multiple';
    return `${action} failed: ${count} elements found matching ${selectorDesc}. Use a more specific selector or provide an index.`;
  }

  // Element not visible
  if (errorMsg.includes('not visible') || errorMsg.includes('hidden')) {
    return `${action} failed: Element ${selectorDesc} is not visible. Wait for it to appear or check if it's hidden behind another element.`;
  }

  // Element not interactable
  if (errorMsg.includes('intercepts pointer events') || errorMsg.includes('not clickable')) {
    return `${action} failed: Element ${selectorDesc} is blocked by an overlay. Try waiting for the overlay to disappear or use hover to reveal it.`;
  }

  // Timeout
  if (errorMsg.includes('Timeout') || errorMsg.includes('timed out')) {
    return `${action} failed: Timeout waiting for ${selectorDesc}. Ensure the page is loaded and element exists.`;
  }

  // Default
  return `${action} failed: ${errorMsg}`;
}