/**
 * WebSearchTool - Perform web searches using external search APIs
 * Provides contextual information for better decision making
 */

import { BrowserTool, CommonToolArgs } from "../../../ai_tools_interface.js";
import { z } from "zod";

export class WebSearchTool extends BrowserTool {
  constructor() {
    super(
      "web_search",
      "Search the web for information to help with decision making. Use when you need additional context, product information, or current data before interacting with a page.",
      z.object({
        query: z.string().describe("Search query to find relevant information"),
        maxResults: z.number().min(1).max(10).default(5).optional()
          .describe("Maximum number of search results to return (1-10)"),
        searchType: z.enum(["general", "news", "shopping", "images"]).default("general").optional()
          .describe("Type of search to perform")
      })
    );
  }

  async call({ query, maxResults = 5, searchType = "general" }) {
    try {
      console.log(`🔍 [WEB_SEARCH] Searching for: "${query}" (type: ${searchType}, max: ${maxResults})`);

      // Use DuckDuckGo Instant Answer API (no API key required)
      const searchUrl = `https://api.duckduckgo.com/?q=${encodeURIComponent(query)}&format=json&no_html=1&skip_disambig=1`;
      
      let searchResults = [];
      
      try {
        const response = await fetch(searchUrl, {
          method: 'GET',
          headers: {
            'User-Agent': 'Vibe Browser AI Agent 1.0'
          }
        });

        if (!response.ok) {
          throw new Error(`Search API returned ${response.status}: ${response.statusText}`);
        }

        const data = await response.json();
        
        // Process DuckDuckGo results
        if (data.Abstract) {
          searchResults.push({
            title: data.Heading || "DuckDuckGo Instant Answer",
            snippet: data.Abstract,
            url: data.AbstractURL || "",
            source: "DuckDuckGo"
          });
        }

        // Add related topics if available
        if (data.RelatedTopics && data.RelatedTopics.length > 0) {
          const relatedResults = data.RelatedTopics
            .slice(0, maxResults - 1)
            .filter(topic => topic.Text && topic.FirstURL)
            .map(topic => ({
              title: topic.Text.split(' - ')[0] || "Related Topic",
              snippet: topic.Text,
              url: topic.FirstURL,
              source: "DuckDuckGo Related"
            }));
          
          searchResults.push(...relatedResults);
        }

      } catch (apiError) {
        console.warn(`⚠️ [WEB_SEARCH] DuckDuckGo API failed: ${apiError.message}`);
        
        // Fallback to a simulated search with common knowledge
        searchResults = this.generateFallbackResults(query, maxResults);
      }

      // If we still don't have results, try alternative approach
      if (searchResults.length === 0) {
        searchResults = this.generateFallbackResults(query, maxResults);
      }

      // Limit results to requested maximum
      searchResults = searchResults.slice(0, maxResults);

      console.log(`[WEB_SEARCH] Found ${searchResults.length} results for "${query}"`);

      return {
        query: query,
        searchType: searchType,
        results: searchResults,
        totalResults: searchResults.length,
        timestamp: new Date().toISOString(),
        message: `Found ${searchResults.length} search results for "${query}"`
      };

    } catch (error) {
      console.error(`❌ [WEB_SEARCH] Search failed:`, error);
      
      // Return fallback results even on error
      const fallbackResults = this.generateFallbackResults(query, maxResults);
      
      return {
        query: query,
        searchType: searchType,
        results: fallbackResults,
        totalResults: fallbackResults.length,
        timestamp: new Date().toISOString(),
        error: error.message,
        message: `Search API failed, returning ${fallbackResults.length} fallback results`
      };
    }
  }

  /**
   * Generate fallback search results based on common knowledge
   * @private
   */
  generateFallbackResults(query, maxResults) {
    const queryLower = query.toLowerCase();
    const fallbackResults = [];

    // Flight-related queries
    if (queryLower.includes('flight') || queryLower.includes('airline') || queryLower.includes('airport')) {
      fallbackResults.push(
        {
          title: "Google Flights - Flight Search",
          snippet: "Search and compare flights from hundreds of airlines and travel sites to find cheap flights",
          url: "https://flights.google.com",
          source: "Fallback Knowledge"
        },
        {
          title: "Expedia Flights",
          snippet: "Book cheap flights and airline tickets with Expedia",
          url: "https://expedia.com/flights",
          source: "Fallback Knowledge"
        }
      );
    }

    // Hotel-related queries
    if (queryLower.includes('hotel') || queryLower.includes('accommodation') || queryLower.includes('booking')) {
      fallbackResults.push(
        {
          title: "Booking.com - Hotels",
          snippet: "Book hotels, apartments, and accommodations worldwide",
          url: "https://booking.com",
          source: "Fallback Knowledge"
        },
        {
          title: "Hotels.com",
          snippet: "Find and book hotels with Hotels.com",
          url: "https://hotels.com",
          source: "Fallback Knowledge"
        }
      );
    }

    // Shopping-related queries
    if (queryLower.includes('buy') || queryLower.includes('shop') || queryLower.includes('product')) {
      fallbackResults.push(
        {
          title: "Amazon - Online Shopping",
          snippet: "Shop online for electronics, books, home & garden, toys & games, and more",
          url: "https://amazon.com",
          source: "Fallback Knowledge"
        },
        {
          title: "Walmart - Save Money Live Better",
          snippet: "Shop for everyday low prices on groceries, electronics, home goods and more",
          url: "https://walmart.com",
          source: "Fallback Knowledge"
        }
      );
    }

    // General web search fallback
    if (fallbackResults.length === 0) {
      fallbackResults.push({
        title: `Search results for "${query}"`,
        snippet: `No specific results available for "${query}". Consider using more specific search terms or checking relevant websites directly.`,
        url: "",
        source: "Fallback Knowledge"
      });
    }

    return fallbackResults.slice(0, maxResults);
  }
}

export default WebSearchTool;
