// UI Pattern Handlers for common web application patterns
import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { getValidTabId } from "./utils.js";

// Pattern definitions for common sites and UI patterns
const UI_PATTERNS = {
  googleFlights: {
    name: "Google Flights",
    detect: (url) => url.includes('flights.google.com'),
    handlers: {
      fillDepartureAirport: {
        selectors: [
          '[aria-label*="Where from"]',
          '[aria-label*="Departure"]',
          '[placeholder*="From"]',
          'input[jsname="yrriRe"]', // Google's internal names
          '.fsapp-origin-input input'
        ],
        strategy: async (tool, value) => {
          // Click to activate
          await tool.click();
          await tool.wait(500);
          
          // Clear existing
          await tool.clearField();
          await tool.wait(300);
          
          // Type airport code
          await tool.typeSlowly(value, 100);
          await tool.wait(1000);
          
          // Select from dropdown
          await tool.selectFromAutocomplete(value);
        }
      },
      
      fillArrivalAirport: {
        selectors: [
          '[aria-label*="Where to"]',
          '[aria-label*="Destination"]', 
          '[placeholder*="To"]',
          'input[jsname="pIWJjc"]',
          '.fsapp-destination-input input'
        ],
        strategy: async (tool, value) => {
          await tool.click();
          await tool.wait(500);
          await tool.clearField();
          await tool.wait(300);
          await tool.typeSlowly(value, 100);
          await tool.wait(1000);
          await tool.selectFromAutocomplete(value);
        }
      },
      
      selectDate: {
        strategy: async (tool, date) => {
          // Click date field
          const dateSelectors = [
            '[aria-label*="Departure date"]',
            '[jsname="huwV7e"]',
            '.fsapp-date-input'
          ];
          
          await tool.clickFirstAvailable(dateSelectors);
          await tool.wait(500);
          
          // Navigate calendar
          await tool.navigateCalendarToDate(date);
          
          // Click the date
          await tool.clickCalendarDate(date);
        }
      },
      
      selectOneWay: {
        selectors: [
          '[aria-label="One way"]',
          'input[value="one-way"]',
          '[jsname="fnKX3e"]'
        ],
        strategy: async (tool) => {
          await tool.click();
        }
      },
      
      searchFlights: {
        selectors: [
          '[aria-label*="Search"]',
          'button[jsname="vLv7Lb"]',
          '.fsapp-search-button'
        ],
        strategy: async (tool) => {
          await tool.click();
          await tool.waitForNavigation();
        }
      }
    }
  },
  
  genericAutocomplete: {
    name: "Generic Autocomplete",
    detect: (element) => {
      return element.getAttribute('aria-autocomplete') || 
             element.getAttribute('aria-haspopup') === 'listbox';
    },
    handlers: {
      fill: {
        strategy: async (tool, element, value) => {
          // Focus element
          await tool.focus(element);
          await tool.wait(200);
          
          // Clear and type
          await tool.clearField(element);
          await tool.typeSlowly(value, 80);
          await tool.wait(500);
          
          // Look for dropdown
          const dropdown = await tool.findNearby({
            role: 'listbox',
            near: element
          });
          
          if (dropdown) {
            // Find matching option
            const option = await tool.findInDropdown(dropdown, value);
            if (option) {
              await tool.click(option);
            } else {
              // Press Enter if no exact match
              await tool.pressKey('Enter');
            }
          }
        }
      }
    }
  },
  
  datePicker: {
    name: "Date Picker",
    detect: (element) => {
      return element.type === 'date' || 
             element.getAttribute('aria-label')?.toLowerCase().includes('date') ||
             element.className?.includes('date');
    },
    handlers: {
      selectDate: {
        strategy: async (tool, element, date) => {
          // Try native date input first
          if (element.type === 'date') {
            const formattedDate = tool.formatDateForInput(date);
            await tool.setValue(element, formattedDate);
            return;
          }
          
          // Click to open picker
          await tool.click(element);
          await tool.wait(500);
          
          // Find calendar widget
          const calendar = await tool.findNearby({
            role: 'dialog',
            className: 'calendar',
            near: element
          });
          
          if (calendar) {
            await tool.navigateCalendarToDate(calendar, date);
            await tool.clickCalendarDate(calendar, date);
          }
        }
      }
    }
  }
};

// Main Pattern Handler Tool
export class UIPatternTool extends BrowserTool {
  constructor() {
    super(
      "handle_ui_pattern",
      "Intelligently handle common UI patterns like autocomplete, date pickers, and site-specific forms",
      z.object({
        tabId: z.number().optional(),
        action: z.enum([
          "fill_departure",
          "fill_arrival",
          "select_date",
          "select_one_way",
          "search_flights",
          "fill_autocomplete",
          "handle_date_picker"
        ]),
        value: z.string().optional(),
        options: z.object({
          patternHint: z.string().optional().describe("Hint about which pattern to use"),
          fallbackToGeneric: z.boolean().default(true).optional()
        }).optional()
      })
    );
  }

  async call({ tabId, action, value, options = {} }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      // Get current URL
      const tab = await chrome.tabs.get(validTabId);
      const url = tab.url;
      
      // Detect pattern
      const pattern = this.detectPattern(url, options.patternHint);
      
      if (pattern) {
        console.log(`Using ${pattern.name} pattern for action: ${action}`);
        return await this.executePatternAction(validTabId, pattern, action, value);
      }
      
      // Fallback to generic handling
      if (options.fallbackToGeneric) {
        console.log(`No specific pattern found, using generic handling for: ${action}`);
        return await this.executeGenericAction(validTabId, action, value);
      }
      
      throw new Error(`No pattern handler found for action: ${action}`);
      
    } catch (error) {
      console.error('[UIPatternTool] Error:', error);
      throw error;
    }
  }
  
  detectPattern(url, hint) {
    // Check URL-based patterns
    for (const [key, pattern] of Object.entries(UI_PATTERNS)) {
      if (pattern.detect && typeof pattern.detect === 'function') {
        if (pattern.detect(url)) {
          return pattern;
        }
      }
    }
    
    // Check hint
    if (hint) {
      const hintLower = hint.toLowerCase();
      if (hintLower.includes('flight') || hintLower.includes('google')) {
        return UI_PATTERNS.googleFlights;
      }
    }
    
    return null;
  }
  
  async executePatternAction(tabId, pattern, action, value) {
    // Map action to pattern handler
    const handlerMap = {
      'fill_departure': 'fillDepartureAirport',
      'fill_arrival': 'fillArrivalAirport',
      'select_date': 'selectDate',
      'select_one_way': 'selectOneWay',
      'search_flights': 'searchFlights'
    };
    
    const handlerName = handlerMap[action];
    const handler = pattern.handlers[handlerName];
    
    if (!handler) {
      throw new Error(`Pattern ${pattern.name} doesn't support action: ${action}`);
    }
    
    // Execute pattern-specific strategy
    const response = await chrome.tabs.sendMessage(tabId, {
      type: 'EXECUTE_PATTERN',
      data: {
        pattern: pattern.name,
        handler: handlerName,
        selectors: handler.selectors,
        value
      }
    });
    
    if (response && response.success) {
      return `Successfully executed ${action} using ${pattern.name} pattern`;
    }
    
    throw new Error(response?.error || 'Pattern execution failed');
  }
  
  async executeGenericAction(tabId, action, value) {
    const response = await chrome.tabs.sendMessage(tabId, {
      type: 'EXECUTE_GENERIC_ACTION',
      data: { action, value }
    });
    
    if (response && response.success) {
      return `Successfully executed ${action}`;
    }
    
    throw new Error(response?.error || 'Generic action failed');
  }
}

// Flight Booking Orchestrator - High-level tool for complete flight booking
export class FlightBookingTool extends BrowserTool {
  constructor() {
    super(
      "book_flight",
      "Complete flight booking workflow with intelligent handling of common booking sites",
      z.object({
        tabId: z.number().optional(),
        booking: z.object({
          from: z.string().describe("Departure airport code (e.g., SFO)"),
          to: z.string().describe("Arrival airport code (e.g., JFK)"),
          date: z.string().describe("Departure date (e.g., Oct 1, 2024-10-01)"),
          returnDate: z.string().optional().describe("Return date for round trip"),
          tripType: z.enum(["one_way", "round_trip"]).default("one_way"),
          passengers: z.number().default(1).optional(),
          class: z.enum(["economy", "business", "first"]).default("economy").optional()
        }),
        preferences: z.object({
          nonstop: z.boolean().default(false).optional(),
          maxPrice: z.number().optional(),
          airlines: z.array(z.string()).optional(),
          sortBy: z.enum(["price", "duration", "departure_time"]).default("price").optional()
        }).optional()
      })
    );
  }

  async call({ tabId, booking, preferences = {} }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      // Navigate to flights.google.com if not already there
      const tab = await chrome.tabs.get(validTabId);
      if (!tab.url.includes('flights.google.com')) {
        await chrome.tabs.update(validTabId, { url: 'https://flights.google.com' });
        await this.waitForPageLoad(validTabId);
      }
      
      // Execute booking flow
      const steps = [
        { name: 'Handle cookies', action: this.handleCookieBanner },
        { name: 'Select trip type', action: () => this.selectTripType(validTabId, booking.tripType) },
        { name: 'Fill departure', action: () => this.fillAirport(validTabId, 'departure', booking.from) },
        { name: 'Fill arrival', action: () => this.fillAirport(validTabId, 'arrival', booking.to) },
        { name: 'Select date', action: () => this.selectDate(validTabId, booking.date) },
        { name: 'Search flights', action: () => this.searchFlights(validTabId) },
        { name: 'Filter results', action: () => this.filterResults(validTabId, preferences) },
        { name: 'Select flight', action: () => this.selectFlight(validTabId, preferences) }
      ];
      
      const results = [];
      
      for (const step of steps) {
        console.log(`Executing: ${step.name}`);
        try {
          const result = await step.action();
          results.push({ step: step.name, success: true, result });
        } catch (error) {
          results.push({ step: step.name, success: false, error: error.message });
          
          // Try to recover from certain errors
          if (step.name.includes('Fill') || step.name.includes('Select')) {
            console.log(`Retrying ${step.name} with alternative approach...`);
            const retryResult = await this.retryWithAlternative(validTabId, step);
            if (retryResult.success) {
              results[results.length - 1] = retryResult;
              continue;
            }
          }
          
          break; // Stop on unrecoverable error
        }
      }
      
      // Generate summary
      return this.generateBookingSummary(results);
      
    } catch (error) {
      console.error('[FlightBookingTool] Error:', error);
      throw error;
    }
  }
  
  // Helper methods would go here...
  async waitForPageLoad(tabId) {
    return new Promise(resolve => {
      chrome.tabs.onUpdated.addListener(function listener(id, info) {
        if (id === tabId && info.status === 'complete') {
          chrome.tabs.onUpdated.removeListener(listener);
          resolve();
        }
      });
    });
  }
  
  generateBookingSummary(results) {
    const successful = results.filter(r => r.success);
    const failed = results.filter(r => !r.success);
    
    let summary = `Flight booking progress: ${successful.length}/${results.length} steps completed\n\n`;
    
    if (successful.length > 0) {
      summary += "Completed steps:\n";
      successful.forEach(r => {
        summary += `✓ ${r.step}\n`;
      });
    }
    
    if (failed.length > 0) {
      summary += "\nFailed steps:\n";
      failed.forEach(r => {
        summary += `✗ ${r.step}: ${r.error}\n`;
      });
    }
    
    return summary;
  }
}

// Export all tools
export default { UIPatternTool, FlightBookingTool };