/**
 * World-class provider-agnostic tool call normalization system
 * Handles compatibility across different LLM providers (OpenAI, Anthropic, Gemini, etc.)
 */

export class ToolCallNormalizer {
  /**
   * Normalize an array of tool calls for a specific provider
   * @param {Array} toolCalls - Raw tool calls from LLM
   * @param {string} provider - Target provider ('openai', 'anthropic', 'gemini')
   * @returns {Array} Normalized tool calls
   */
  static normalizeToolCalls(toolCalls, provider = 'openai') {
    if (!Array.isArray(toolCalls) || toolCalls.length === 0) {
      return [];
    }
    
    console.log(`🔧 [TOOL_NORMALIZER] Normalizing ${toolCalls.length} tool calls for provider: ${provider}`);
    
    return toolCalls.map((tc, index) => {
      try {
        return this.normalizeToolCall(tc, provider);
      } catch (error) {
        console.error(`❌ [TOOL_NORMALIZER] Failed to normalize tool call ${index}:`, error);
        // Return a safe fallback instead of breaking the entire chain
        return this.createFallbackToolCall(tc, error);
      }
    }).filter(Boolean); // Remove any null/undefined results
  }
  
  /**
   * Normalize a single tool call
   * @param {Object} toolCall - Raw tool call object
   * @param {string} provider - Target provider
   * @returns {Object} Normalized tool call
   */
  static normalizeToolCall(toolCall, provider) {
    if (!toolCall || typeof toolCall !== 'object') {
      throw new Error('Invalid tool call: must be an object');
    }
    
    const normalized = { ...toolCall };
    const originalType = toolCall.type;
    
    // Normalize type field based on provider expectations
    normalized.type = this.getExpectedType(provider, toolCall);
    
    // Ensure required fields exist
    this.ensureRequiredFields(normalized);
    
    // Provider-specific transformations
    this.applyProviderSpecificTransforms(normalized, provider);
    
    // Validate the final result
    this.validateToolCall(normalized);
    
    if (originalType !== normalized.type) {
      console.log(`🔄 [TOOL_NORMALIZER] Transformed type: ${originalType} → ${normalized.type} (${provider})`);
    }
    
    return normalized;
  }
  
  /**
   * Get the expected type for a provider
   * @param {string} provider - Target provider
   * @param {Object} toolCall - Original tool call for context
   * @returns {string} Expected type
   */
  static getExpectedType(provider, toolCall) {
    const providerTypes = {
      openai: 'function',
      anthropic: 'tool_use', 
      gemini: 'function',
      google: 'function',
      cohere: 'function',
      langchain_internal: 'function'
    };
    
    const normalizedProvider = provider.toLowerCase();
    
    if (providerTypes[normalizedProvider]) {
      return providerTypes[normalizedProvider];
    }
    
    // Smart detection for unknown providers
    return this.detectCorrectType(toolCall, provider);
  }
  
  /**
   * Detect correct type based on tool call structure
   * @param {Object} toolCall - Tool call object
   * @param {string} provider - Provider name for context
   * @returns {string} Detected type
   */
  static detectCorrectType(toolCall, provider) {
    // Check structure patterns
    if (toolCall.function && typeof toolCall.function === 'object') {
      return 'function';
    }
    
    if (toolCall.tool_use && typeof toolCall.tool_use === 'object') {
      return 'tool_use';
    }
    
    // Check naming conventions
    if (toolCall.name && toolCall.args) {
      return 'function'; // Most common pattern
    }
    
    console.warn(`⚠️ [TOOL_NORMALIZER] Could not detect type for provider ${provider}, using 'function' as fallback`);
    return 'function';
  }
  
  /**
   * Ensure required fields exist on the tool call
   * @param {Object} toolCall - Tool call to validate
   */
  static ensureRequiredFields(toolCall) {
    // Generate ID if missing
    if (!toolCall.id) {
      toolCall.id = `tool_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    }
    
    // Ensure name exists
    if (!toolCall.name) {
      if (toolCall.function?.name) {
        toolCall.name = toolCall.function.name;
      } else {
        throw new Error('Tool call missing required name field');
      }
    }
    
    // Ensure args exist
    if (!toolCall.args && !toolCall.arguments) {
      if (toolCall.function?.arguments) {
        toolCall.args = typeof toolCall.function.arguments === 'string' 
          ? JSON.parse(toolCall.function.arguments)
          : toolCall.function.arguments;
      } else {
        toolCall.args = {};
      }
    }
    
    // Normalize args vs arguments
    if (toolCall.arguments && !toolCall.args) {
      toolCall.args = typeof toolCall.arguments === 'string'
        ? JSON.parse(toolCall.arguments)
        : toolCall.arguments;
    }
  }
  
  /**
   * Apply provider-specific transformations
   * @param {Object} toolCall - Tool call to transform
   * @param {string} provider - Target provider
   */
  static applyProviderSpecificTransforms(toolCall, provider) {
    switch (provider.toLowerCase()) {
      case 'openai':
        // OpenAI expects function structure
        if (!toolCall.function) {
          toolCall.function = {
            name: toolCall.name,
            arguments: typeof toolCall.args === 'string' ? toolCall.args : JSON.stringify(toolCall.args)
          };
        }
        break;
        
      case 'anthropic':
        // Anthropic has different structure expectations
        if (toolCall.function) {
          toolCall.input = typeof toolCall.function.arguments === 'string'
            ? JSON.parse(toolCall.function.arguments)
            : toolCall.function.arguments;
        } else if (toolCall.args) {
          toolCall.input = toolCall.args;
        }
        break;
        
      case 'gemini':
      case 'google':
        // Google/Gemini similar to OpenAI but may have differences
        this.applyProviderSpecificTransforms(toolCall, 'openai');
        break;
    }
  }
  
  /**
   * Validate tool call structure
   * @param {Object} toolCall - Tool call to validate
   * @throws {Error} If validation fails
   */
  static validateToolCall(toolCall) {
    const required = ['id', 'name', 'type'];
    const missing = required.filter(field => !toolCall[field]);
    
    if (missing.length > 0) {
      throw new Error(`Invalid tool call: missing required fields: ${missing.join(', ')}`);
    }
    
    // Validate type is a string
    if (typeof toolCall.type !== 'string') {
      throw new Error(`Invalid tool call: type must be a string, got ${typeof toolCall.type}`);
    }
    
    // Validate name is a string
    if (typeof toolCall.name !== 'string') {
      throw new Error(`Invalid tool call: name must be a string, got ${typeof toolCall.name}`);
    }
    
    // Validate args if present
    if (toolCall.args && typeof toolCall.args !== 'object') {
      throw new Error(`Invalid tool call: args must be an object, got ${typeof toolCall.args}`);
    }
  }
  
  /**
   * Create a fallback tool call when normalization fails
   * @param {Object} originalToolCall - Original tool call that failed
   * @param {Error} error - The error that occurred
   * @returns {Object|null} Fallback tool call or null if cannot create
   */
  static createFallbackToolCall(originalToolCall, error) {
    try {
      return {
        id: originalToolCall.id || `fallback_${Date.now()}`,
        name: originalToolCall.name || 'unknown_tool',
        type: 'function',
        args: originalToolCall.args || {},
        _fallback: true,
        _error: error.message
      };
    } catch (fallbackError) {
      console.error(`❌ [TOOL_NORMALIZER] Could not create fallback tool call:`, fallbackError);
      return null;
    }
  }
  
  /**
   * Get normalization statistics for debugging
   * @param {Array} originalToolCalls - Original tool calls
   * @param {Array} normalizedToolCalls - Normalized tool calls
   * @returns {Object} Statistics object
   */
  static getStats(originalToolCalls, normalizedToolCalls) {
    const originalTypes = originalToolCalls.map(tc => tc.type).filter(Boolean);
    const normalizedTypes = normalizedToolCalls.map(tc => tc.type).filter(Boolean);
    const fallbackCount = normalizedToolCalls.filter(tc => tc._fallback).length;
    
    return {
      originalCount: originalToolCalls.length,
      normalizedCount: normalizedToolCalls.length,
      fallbackCount,
      typeTransformations: this.getTypeTransformations(originalTypes, normalizedTypes),
      successRate: ((normalizedToolCalls.length - fallbackCount) / originalToolCalls.length * 100).toFixed(1) + '%'
    };
  }
  
  /**
   * Track type transformations for analysis
   * @param {Array} originalTypes - Original types
   * @param {Array} normalizedTypes - Normalized types  
   * @returns {Object} Transformation map
   */
  static getTypeTransformations(originalTypes, normalizedTypes) {
    const transformations = {};
    
    for (let i = 0; i < Math.min(originalTypes.length, normalizedTypes.length); i++) {
      const original = originalTypes[i];
      const normalized = normalizedTypes[i];
      
      if (original !== normalized) {
        const key = `${original} → ${normalized}`;
        transformations[key] = (transformations[key] || 0) + 1;
      }
    }
    
    return transformations;
  }
}

export default ToolCallNormalizer;