// Tab management browser automation tools

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { waitForPageReady } from "./utils.js";

// Tool to list all opened tabs
export class ListTabsTool extends BrowserTool {
  constructor() {
    super(
      "list_tabs",
      "List all currently opened browser tabs with their IDs, URLs, and titles. Use this to check existing tabs before creating new ones.",
      z.object({
        includeDetails: z.boolean().default(true).nullable().optional().describe("Include tab details like URL and title (default: true)")
      })
    );
  }
  
  async call({ includeDetails = true }) {
    try {
      // Get all tabs
      const tabs = await chrome.tabs.query({});
      
      if (!tabs || tabs.length === 0) {
        return "No tabs found";
      }
      
      // Sort tabs by ID for consistent ordering
      tabs.sort((a, b) => a.id - b.id);
      
      // Format tab information
      const tabInfo = tabs.map(tab => {
        if (includeDetails) {
          const status = tab.active ? ' [ACTIVE]' : '';
          return `Tab ${tab.id}${status}: "${tab.title}" - ${tab.url}`;
        } else {
          return `Tab ${tab.id}`;
        }
      });
      
      const summary = `Found ${tabs.length} tab(s):\n${tabInfo.join('\n')}`;
      
      console.log(`📑 Listed ${tabs.length} tabs`);
      return summary;
      
    } catch (error) {
      console.error(`❌ Failed to list tabs:`, error);
      throw new Error(`Failed to list tabs: ${error.message}`);
    }
  }
}

// Tool to switch to a specific tab by ID
export class SwitchToTabTool extends BrowserTool {
  constructor() {
    super(
      "switch_to_tab",
      "Switch to a specific browser tab by its ID. Use list_tabs first to get available tab IDs.",
      z.object({
        tabId: z.number().describe("The ID of the tab to switch to")
      })
    );
  }

  async call({ tabId }) {
    try {
      console.log(`🔄 Switching to tab ${tabId}`);

      // Update the tab to make it active
      const updatedTab = await chrome.tabs.update(tabId, {
        active: true
      });

      // Also focus the window containing the tab
      await chrome.windows.update(updatedTab.windowId, {
        focused: true
      });

      console.log(`Successfully switched to tab ${tabId}: ${updatedTab.url}`);
      return `Successfully switched to tab ${tabId}: "${updatedTab.title}" - ${updatedTab.url}`;

    } catch (error) {
      console.error(`❌ Failed to switch to tab ${tabId}:`, error);

      // Check if tab exists
      try {
        await chrome.tabs.get(tabId);
        throw new Error(`Failed to switch to tab ${tabId}: ${error.message}`);
      } catch (getError) {
        throw new Error(`Tab ${tabId} not found. Use list_tabs to see available tabs.`);
      }
    }
  }
}

// Tool to set the working tab for page content extraction
// Unlike switch_to_tab, this doesn't change browser focus - just updates internal state
export class SetWorkingTabTool extends BrowserTool {
  constructor() {
    super(
      "set_working_tab",
      "Set which tab to read page content from WITHOUT changing browser focus. Use this to work on multiple tabs in background. Subagents should use this to switch context between tabs.",
      z.object({
        tabId: z.number().describe("The ID of the tab to set as working tab")
      })
    );
  }

  async call({ tabId }) {
    try {
      // Validate the tab exists
      const tab = await chrome.tabs.get(tabId);
      console.log(`📌 Setting working tab to ${tabId}: ${tab.url}`);

      // Just return success - ReactGraph.ts will update currentTabId based on this
      return `Working tab set to ${tabId}: "${tab.title}" - ${tab.url}`;
    } catch (error) {
      throw new Error(`Tab ${tabId} not found. Use list_tabs to see available tabs.`);
    }
  }
}

// Tool to create a new tab and optionally navigate to a URL
export class CreateNewTabTool extends BrowserTool {
  constructor() {
    super(
      "create_new_tab",
      "Create a new browser tab and optionally navigate to a URL",
      z.object({
        url: z.string().url().nullable().optional().describe("Optional URL to navigate to in the new tab"),
        waitForReady: z.boolean().default(true).nullable().optional().describe("Wait for page to be ready before returning (default: true)")
      })
    );
  }

  
  async call({ url, waitForReady = true }) {
    try {
      console.log(`🆕 Creating new tab${url ? ` and navigating to ${url}` : ''}`);
      
      // Create a new tab
      const newTab = await chrome.tabs.create({
        url: url || 'about:blank',
        active: true
      });
      
      // If we have a URL and should wait for readiness
      if (url && waitForReady && url !== 'about:blank') {
        console.log('⏳ Waiting for new tab to be ready...');
        
        // First wait for tab to complete loading (basic navigation)
        await this.waitForTabComplete(newTab.id);
        
        // Then wait for DOM and dynamic content to be ready
        await waitForPageReady(newTab.id);
        
        return `Successfully created new tab (ID: ${newTab.id}), navigated to ${url}, and page is ready`;
      } else if (url) {
        return `Successfully created new tab (ID: ${newTab.id}) and navigated to ${url}`;
      } else {
        return `Successfully created new blank tab (ID: ${newTab.id})`;
      }
    } catch (error) {
      throw new Error(`Failed to create new tab: ${error.message}`);
    }
  }

  /**
   * Wait for tab to complete basic loading
   * Uses DOMContentLoaded for faster acceptance - don't wait for all resources
   * @private
   */
  waitForTabComplete(tabId, timeoutMs = 60000) {
    return new Promise((resolve, reject) => {
      let timeoutId;
      let hasResolved = false;
      let listener;
      let domContentLoadedListener;

      const cleanup = () => {
        if (listener) {
          chrome.tabs.onUpdated.removeListener(listener);
        }
        if (domContentLoadedListener) {
          chrome.webNavigation.onDOMContentLoaded.removeListener(domContentLoadedListener);
        }
        if (timeoutId) {
          clearTimeout(timeoutId);
        }
      };

      const resolveOnce = (tab) => {
        if (!hasResolved) {
          hasResolved = true;
          cleanup();
          resolve(tab);
        }
      };

      const rejectOnce = (error) => {
        if (!hasResolved) {
          hasResolved = true;
          cleanup();
          reject(error);
        }
      };

      // Listen for DOMContentLoaded - accept page as soon as DOM is ready
      // This is MUCH faster than waiting for all resources (tab.status === 'complete')
      domContentLoadedListener = async (details) => {
        if (details.tabId === tabId && details.frameId === 0) {
          console.log(`✅ DOM ready for new tab ${tabId}: ${details.url}`);
          try {
            const tab = await chrome.tabs.get(tabId);
            if (tab && !tab.url?.includes('chrome-error://')) {
              resolveOnce(tab);
            }
          } catch (e) {
            console.warn(`Failed to get tab after DOMContentLoaded: ${e.message}`);
          }
        }
      };
      chrome.webNavigation.onDOMContentLoaded.addListener(domContentLoadedListener);

      listener = (changedTabId, changeInfo, tab) => {
        if (changedTabId === tabId) {
          // Check for completion (fallback if DOMContentLoaded didn't fire)
          if (changeInfo.status === 'complete') {
            console.log(`New tab ${tabId} completed loading: ${tab.url}`);
            resolveOnce(tab);
          }
          // Check for navigation errors
          else if (tab.url && tab.url.includes('chrome-error://')) {
            console.error(`❌ New tab ${tabId} navigation error: ${tab.url}`);
            rejectOnce(new Error(`Navigation error: ${tab.url}`));
          }
        }
      };

      // Set up timeout
      timeoutId = setTimeout(() => {
        console.error(`❌ New tab navigation timeout after ${timeoutMs}ms for tab ${tabId}`);
        rejectOnce(new Error(`Navigation timeout after ${timeoutMs}ms`));
      }, timeoutMs);

      chrome.tabs.onUpdated.addListener(listener);

      // Also check if tab is already complete
      chrome.tabs.get(tabId, (tab) => {
        if (chrome.runtime.lastError) {
          rejectOnce(new Error(`Tab ${tabId} not found: ${chrome.runtime.lastError.message}`));
          return;
        }

        if (tab && tab.status === 'complete') {
          console.log(`New tab ${tabId} already complete: ${tab.url}`);
          resolveOnce(tab);
        } else if (tab && tab.url && tab.url.includes('chrome-error://')) {
          rejectOnce(new Error(`Tab already in error state: ${tab.url}`));
        } else {
          console.log(`⏳ New tab ${tabId} still loading: ${tab.url} (status: ${tab.status})`);
        }
      });
    });
  }
}