/**
 * SubAgentTool - Simple subagent that inherits parent ReactAgent config
 *
 * Architecture:
 * 1. SubAgentTool receives parent config via inherit()
 * 2. Creates new ReactAgent with SAME config (llm, tools, options)
 * 3. Subagent decides itself if it needs new tab
 * 4. Returns result to parent
 */

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { ReactAgent } from "../../agent/ReactGraph.js";

const SUBAGENT_MAX_ITERATIONS = 15;

export class SubAgentTool extends BrowserTool {
  // Parent config - set via inherit()
  parentConfig = null;

  constructor() {
    super(
      "subagent",
      `Spawn a sub-agent to perform a task with isolated context.

**WHEN TO USE:**
- Break down complex tasks into smaller independent subtasks
- Run multiple lookups in parallel on ALREADY OPENED tabs
- Isolate context for specific operations

**PARAMETERS:**
- systemPrompt: Complete task instructions for the sub-agent
- tabId (optional): Tab ID to work on. If provided, subagent works on that specific tab (NO navigation needed).

**EXAMPLE 1 - Basic (creates own tab):**
{ "name": "subagent", "args": { "systemPrompt": "Navigate to amazon.com and find X" } }

**EXAMPLE 2 - With tabId (works on existing tab):**
{ "name": "subagent", "args": { "systemPrompt": "Extract price from current page", "tabId": 123456 } }

**BATCH PATTERN - Check multiple pages in parallel:**
1. Parent opens tabs: navigate_to_url(url1, newTab:true) → tabId: 111
                      navigate_to_url(url2, newTab:true) → tabId: 222
2. Parent delegates to subagents with specific tabIds:
   parallel([
     subagent({ systemPrompt: "Extract data", tabId: 111 }),
     subagent({ systemPrompt: "Extract data", tabId: 222 })
   ])
3. Each subagent works on its assigned tab - no redundant navigation!`,
      z.object({
        systemPrompt: z.string().describe("Complete task instructions for the sub-agent"),
        tabId: z.number().nullable().optional().describe("Tab ID to work on. Subagent will use this tab instead of creating a new one.")
      })
    );
  }

  /**
   * Inherit parent ReactAgent config
   * Called by ReactGraph to pass its config to SubAgentTool
   * @param {object} config - Parent config { llm, langchainTools, options, modelName }
   */
  inherit(config) {
    this.parentConfig = config;
    console.log(`[SubAgent] Inherited parent config: ${config.modelName}`);
  }

  async call(args) {
    const { systemPrompt, tabId } = args;
    const startTime = Date.now();

    console.log(`[SubAgent] Starting with prompt: ${systemPrompt.substring(0, 100)}...${tabId ? ` (tabId: ${tabId})` : ''}`);

    if (!this.parentConfig) {
      console.error('[SubAgent] No parent config - inherit() was not called');
      return {
        result: null,
        success: false,
        error: "SubAgentTool not initialized. Parent agent must call inherit()."
      };
    }

    try {
      const { llm, langchainTools, options, modelName } = this.parentConfig;

      // Create new ReactAgent with tab context
      // If tabId is provided, subagent works on that specific tab
      // If tabId is null/undefined, subagent creates its own tab when navigating
      const parentOnEvent = options.onEvent;
      const subAgent = new ReactAgent({
        llm,
        langchainTools,  // REUSE parent's tools - tool calls get tabId injected by ReactGraph
        options: {
          onEvent: (type, event) => {
            console.log(`[SubAgent] ${type}`);
            // Propagate subagent events to parent with subagent metadata
            if (parentOnEvent) {
              parentOnEvent(type, {
                ...event,
                isSubagent: true,
                subagentTabId: tabId,
                subagentArgs: args // Pass full args for matching
              });
            }
          },
          isAborted: options.isAborted || (() => null),
          getPageContentTool: options.getPageContentTool,
          getTabs: options.getTabs,
          takeScreenshotCallback: options.takeScreenshotCallback,
          takeScreenshot: options.takeScreenshot,
          maxTokens: options.maxTokens || 50000,
          modelName,
          initialTabId: tabId || null  // KEY: Use provided tabId, or null to create new tab
        },
        mode: 'agent'
      });

      // Run with enhanced prompt to prevent doom loops
      const enhancedPrompt = `${systemPrompt}

IMPORTANT: You are a SUB-AGENT. Complete this task yourself. Return a concise result.`;

      const result = await subAgent.run(enhancedPrompt, { maxIterations: SUBAGENT_MAX_ITERATIONS });

      const duration = Date.now() - startTime;
      console.log(`[SubAgent] Completed in ${duration}ms`);

      return {
        result: result.output,
        success: result.completed,
        iterations: result.iterations,
        duration
      };

    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      console.error('[SubAgent] Error:', errorMessage);
      return {
        result: null,
        success: false,
        error: errorMessage,
        duration: Date.now() - startTime
      };
    }
  }
}

export default SubAgentTool;
