// Smart Wait and Retry System for robust element interactions
import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { getValidTabId, formatErrorMessage } from "./utils.js";

export class SmartWaitForElementTool extends BrowserTool {
  constructor() {
    super(
      "smart_wait_for_element",
      "Wait for element with intelligent retry strategies and condition checking",
      z.object({
        tabId: z.number().optional(),
        locator: z.union([
          z.string().describe("Simple selector or text"),
          z.object({
            role: z.string().optional(),
            text: z.string().optional(),
            placeholder: z.string().optional(),
            selector: z.string().optional()
          })
        ]),
        conditions: z.array(z.enum([
          "visible",
          "enabled",
          "stable",
          "clickable",
          "has_value",
          "text_changed",
          "attribute_changed"
        ])).default(["visible", "enabled"]),
        options: z.object({
          timeout: z.number().default(10000).optional(),
          retryInterval: z.number().default(500).optional(),
          stableTime: z.number().default(500).optional().describe("Time element must remain stable"),
          expectedText: z.string().optional().describe("Wait until element has this text"),
          expectedAttribute: z.object({
            name: z.string(),
            value: z.string()
          }).optional()
        }).optional()
      })
    );
  }

  async call({ tabId, locator, conditions, options = {} }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      const {
        timeout = 10000,
        retryInterval = 500,
        stableTime = 500
      } = options;
      
      const startTime = Date.now();
      let lastError;
      let element;
      let previousState = {};
      
      while (Date.now() - startTime < timeout) {
        try {
          // Find element using smart strategies
          element = await this.findElement(validTabId, locator);
          
          if (element) {
            // Check all conditions
            const allConditionsMet = await this.checkConditions(
              validTabId, 
              element, 
              conditions, 
              options,
              previousState
            );
            
            if (allConditionsMet) {
              // Wait for stability if required
              if (conditions.includes("stable")) {
                await new Promise(resolve => setTimeout(resolve, stableTime));
                
                // Re-check after stability period
                const stillValid = await this.checkConditions(
                  validTabId, 
                  element, 
                  conditions, 
                  options,
                  previousState
                );
                
                if (stillValid) {
                  return this.formatSuccessResponse(element, conditions);
                }
              } else {
                return this.formatSuccessResponse(element, conditions);
              }
            }
            
            // Update previous state for comparison
            previousState = await this.getElementState(validTabId, element);
          }
          
        } catch (error) {
          lastError = error;
          console.log(`Wait attempt failed: ${error.message}`);
        }
        
        // Intelligent retry with backoff
        const elapsed = Date.now() - startTime;
        const backoffTime = this.calculateBackoff(elapsed, timeout, retryInterval);
        
        // Try alternative strategies if taking too long
        if (elapsed > timeout / 2 && !element) {
          console.log("Attempting alternative search strategies...");
          element = await this.tryAlternativeStrategies(validTabId, locator);
        }
        
        await new Promise(resolve => setTimeout(resolve, backoffTime));
      }
      
      // Timeout - provide detailed error
      throw new Error(await this.generateTimeoutError(validTabId, locator, lastError, conditions));
      
    } catch (error) {
      console.error('[SmartWaitForElement] Error:', error);
      throw new Error(formatErrorMessage(error, 'Smart wait for element', locator));
    }
  }
  
  async findElement(tabId, locator) {
    const normalizedLocator = typeof locator === 'string' ? 
      { text: locator, selector: locator } : locator;
    
    const response = await chrome.tabs.sendMessage(tabId, {
      type: 'SMART_WAIT_FIND',
      data: { locator: normalizedLocator }
    });
    
    if (response && response.element) {
      return response.element;
    }
    
    return null;
  }
  
  async checkConditions(tabId, element, conditions, options, previousState) {
    const response = await chrome.tabs.sendMessage(tabId, {
      type: 'CHECK_CONDITIONS',
      data: { 
        element,
        conditions,
        options,
        previousState
      }
    });
    
    return response && response.allMet;
  }
  
  async getElementState(tabId, element) {
    const response = await chrome.tabs.sendMessage(tabId, {
      type: 'GET_ELEMENT_STATE',
      data: { element }
    });
    
    return response && response.state;
  }
  
  calculateBackoff(elapsed, timeout, baseInterval) {
    // Progressive backoff: start fast, slow down over time
    const progress = elapsed / timeout;
    
    if (progress < 0.25) {
      return baseInterval; // Fast checking at start
    } else if (progress < 0.5) {
      return baseInterval * 1.5;
    } else if (progress < 0.75) {
      return baseInterval * 2;
    } else {
      return baseInterval * 3; // Slower near timeout
    }
  }
  
  async tryAlternativeStrategies(tabId, locator) {
    const alternatives = await chrome.tabs.sendMessage(tabId, {
      type: 'TRY_ALTERNATIVES',
      data: { locator }
    });
    
    return alternatives && alternatives.element;
  }
  
  formatSuccessResponse(element, conditions) {
    const conditionText = conditions.join(', ');
    return `Element found and all conditions met (${conditionText}): ${element.description}`;
  }
  
  async generateTimeoutError(tabId, locator, lastError, conditions) {
    const diagnostic = await chrome.tabs.sendMessage(tabId, {
      type: 'WAIT_TIMEOUT_DIAGNOSTIC',
      data: { locator, conditions }
    });
    
    const parts = [
      `Timeout waiting for element: ${JSON.stringify(locator)}`,
      `Conditions required: ${conditions.join(', ')}`,
      ''
    ];
    
    if (diagnostic) {
      if (diagnostic.elementFound) {
        parts.push('Element was found but conditions not met:');
        diagnostic.failedConditions.forEach(cond => {
          parts.push(`  - ${cond.condition}: ${cond.reason}`);
        });
      } else {
        parts.push('Element not found. Possible reasons:');
        parts.push('  - Element loads dynamically and needs more time');
        parts.push('  - Selector/locator is incorrect');
        parts.push('  - Element is in iframe or shadow DOM');
        
        if (diagnostic.suggestions.length > 0) {
          parts.push('', 'Suggestions:');
          diagnostic.suggestions.forEach(sugg => {
            parts.push(`  - ${sugg}`);
          });
        }
      }
    }
    
    if (lastError) {
      parts.push('', `Last error: ${lastError.message}`);
    }
    
    return parts.join('\n');
  }
}

// Smart Fill Tool with retry logic
export class SmartFillTool extends BrowserTool {
  constructor() {
    super(
      "smart_fill",
      "Fill form field with intelligent retry and activation strategies",
      z.object({
        tabId: z.number().optional(),
        target: z.union([
          z.string(),
          z.object({
            role: z.string().optional(),
            placeholder: z.string().optional(),
            label: z.string().optional(),
            selector: z.string().optional()
          })
        ]),
        value: z.string(),
        options: z.object({
          clearFirst: z.boolean().default(true).optional(),
          clickFirst: z.boolean().default(true).optional(),
          typeSlowly: z.boolean().default(false).optional(),
          pressEnter: z.boolean().default(false).optional(),
          waitForAutocomplete: z.boolean().default(false).optional(),
          retryStrategies: z.array(z.enum([
            "click_then_fill",
            "focus_then_fill",
            "clear_then_fill",
            "type_character_by_character",
            "dispatch_events",
            "set_value_directly"
          ])).optional()
        }).optional()
      })
    );
  }

  async call({ tabId, target, value, options = {} }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      const strategies = options.retryStrategies || [
        "click_then_fill",
        "focus_then_fill",
        "clear_then_fill",
        "type_character_by_character",
        "dispatch_events",
        "set_value_directly"
      ];
      
      let lastError;
      
      for (const strategy of strategies) {
        try {
          console.log(`Trying fill strategy: ${strategy}`);
          
          const response = await chrome.tabs.sendMessage(validTabId, {
            type: 'SMART_FILL',
            data: {
              target,
              value,
              strategy,
              options
            }
          });
          
          if (response && response.success) {
            // Verify the value was actually set
            const verification = await this.verifyFill(validTabId, target, value);
            if (verification.success) {
              return `Successfully filled field using ${strategy}: "${value}"`;
            } else {
              console.log(`Fill verification failed: ${verification.reason}`);
              continue;
            }
          }
          
        } catch (error) {
          lastError = error;
          console.log(`Strategy ${strategy} failed: ${error.message}`);
        }
      }
      
      // All strategies failed
      throw new Error(`Failed to fill field after trying ${strategies.length} strategies. Last error: ${lastError?.message}`);
      
    } catch (error) {
      console.error('[SmartFill] Error:', error);
      throw new Error(formatErrorMessage(error, 'Smart fill', target));
    }
  }
  
  async verifyFill(tabId, target, expectedValue) {
    const response = await chrome.tabs.sendMessage(tabId, {
      type: 'VERIFY_FILL',
      data: { target, expectedValue }
    });
    
    return response || { success: false, reason: 'No response' };
  }
}

// Export for use
export default { SmartWaitForElementTool, SmartFillTool };