// Smart Element Finder with multiple fallback strategies
import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { getValidTabId, formatErrorMessage } from "./utils.js";

// Enhanced locator schema with more options
const SmartLocatorSchema = z.object({
  // Primary strategies
  role: z.string().optional().describe("ARIA role (button, textbox, link, etc)"),
  text: z.string().optional().describe("Visible text content"),
  label: z.string().optional().describe("Associated label or aria-label"),
  placeholder: z.string().optional().describe("Placeholder text"),
  name: z.string().optional().describe("Name attribute"),
  id: z.string().optional().describe("Element ID"),
  
  // Context strategies
  near: z.string().optional().describe("Find element near this text"),
  after: z.string().optional().describe("Find element after this text"),
  before: z.string().optional().describe("Find element before this text"),
  under: z.string().optional().describe("Find element under this heading"),
  
  // Fallback strategies
  contains: z.string().optional().describe("Element contains this text"),
  selector: z.string().optional().describe("CSS selector fallback"),
  xpath: z.string().optional().describe("XPath fallback"),
  
  // Options
  exact: z.boolean().default(false).optional(),
  timeout: z.number().default(5000).optional(),
  retryStrategies: z.boolean().default(true).optional()
});

export class SmartFindElementTool extends BrowserTool {
  constructor() {
    super(
      "smart_find_element",
      "Advanced element finder with multiple strategies and intelligent fallbacks",
      z.object({
        tabId: z.number().optional(),
        locator: SmartLocatorSchema,
        options: z.object({
          waitForStable: z.boolean().default(true).optional(),
          returnAll: z.boolean().default(false).optional(),
          includeHidden: z.boolean().default(false).optional(),
          maxResults: z.number().default(10).optional()
        }).optional()
      })
    );
  }

  async call({ tabId, locator, options = {} }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      // Try strategies in order of reliability
      const strategies = [
        () => this.findByRole(validTabId, locator),
        () => this.findByLabel(validTabId, locator),
        () => this.findByText(validTabId, locator),
        () => this.findByPlaceholder(validTabId, locator),
        () => this.findByProximity(validTabId, locator),
        () => this.findBySelector(validTabId, locator),
        () => this.findWithIntelligentFallback(validTabId, locator)
      ];
      
      let lastError;
      let allResults = [];
      
      for (const strategy of strategies) {
        try {
          const result = await strategy();
          if (result && result.elements && result.elements.length > 0) {
            // Merge results if collecting all
            if (options.returnAll) {
              allResults = allResults.concat(result.elements);
            } else {
              return this.formatResponse(result, locator);
            }
          }
        } catch (error) {
          lastError = error;
          console.log(`Strategy failed: ${error.message}`);
        }
      }
      
      // If collecting all results
      if (options.returnAll && allResults.length > 0) {
        return this.formatResponse({ elements: allResults, count: allResults.length }, locator);
      }
      
      // If all strategies failed, provide detailed diagnostic
      return await this.provideDiagnostic(validTabId, locator, lastError);
      
    } catch (error) {
      console.error('[SmartFindElement] Error:', error);
      throw new Error(formatErrorMessage(error, 'Smart find element', locator));
    }
  }
  
  async findByRole(tabId, locator) {
    if (!locator.role) return null;
    
    const command = {
      type: 'SMART_FIND',
      data: {
        strategy: 'role',
        role: locator.role,
        text: locator.text,
        name: locator.name || locator.label,
        exact: locator.exact
      }
    };
    
    return await chrome.tabs.sendMessage(tabId, command);
  }
  
  async findByLabel(tabId, locator) {
    if (!locator.label) return null;
    
    const command = {
      type: 'SMART_FIND',
      data: {
        strategy: 'label',
        label: locator.label,
        exact: locator.exact
      }
    };
    
    return await chrome.tabs.sendMessage(tabId, command);
  }
  
  async findByText(tabId, locator) {
    if (!locator.text) return null;
    
    const command = {
      type: 'SMART_FIND',
      data: {
        strategy: 'text',
        text: locator.text,
        exact: locator.exact
      }
    };
    
    return await chrome.tabs.sendMessage(tabId, command);
  }
  
  async findByPlaceholder(tabId, locator) {
    if (!locator.placeholder) return null;
    
    const command = {
      type: 'SMART_FIND',
      data: {
        strategy: 'placeholder',
        placeholder: locator.placeholder,
        exact: locator.exact
      }
    };
    
    return await chrome.tabs.sendMessage(tabId, command);
  }
  
  async findByProximity(tabId, locator) {
    if (!locator.near && !locator.after && !locator.before && !locator.under) return null;
    
    const command = {
      type: 'SMART_FIND',
      data: {
        strategy: 'proximity',
        near: locator.near,
        after: locator.after,
        before: locator.before,
        under: locator.under,
        targetRole: locator.role,
        targetText: locator.text
      }
    };
    
    return await chrome.tabs.sendMessage(tabId, command);
  }
  
  async findBySelector(tabId, locator) {
    if (!locator.selector) return null;
    
    const command = {
      type: 'SMART_FIND',
      data: {
        strategy: 'selector',
        selector: locator.selector
      }
    };
    
    return await chrome.tabs.sendMessage(tabId, command);
  }
  
  async findWithIntelligentFallback(tabId, locator) {
    // This strategy tries to be smart about common patterns
    const command = {
      type: 'SMART_FIND',
      data: {
        strategy: 'intelligent',
        hints: locator
      }
    };
    
    return await chrome.tabs.sendMessage(tabId, command);
  }
  
  async provideDiagnostic(tabId, locator, error) {
    // Get page context for debugging
    const diagnostic = await chrome.tabs.sendMessage(tabId, {
      type: 'GET_DIAGNOSTIC',
      data: { locator }
    });
    
    const report = [
      `Could not find element matching: ${JSON.stringify(locator)}`,
      '',
      'Diagnostic Information:',
      `- Page URL: ${diagnostic.url}`,
      `- Total interactive elements: ${diagnostic.interactiveCount}`,
      `- Form fields found: ${diagnostic.formFieldCount}`,
      `- Buttons found: ${diagnostic.buttonCount}`,
      '',
      'Suggestions:'
    ];
    
    // Add specific suggestions based on what we were looking for
    if (locator.role === 'textbox' || locator.placeholder) {
      report.push('- Try using find_element with role:"combobox" for autocomplete fields');
      report.push('- Input might be inside a shadow DOM or iframe');
      report.push('- Field might require clicking before it becomes active');
    }
    
    if (locator.text) {
      report.push(`- Elements with similar text: ${diagnostic.similarTextElements.join(', ')}`);
      report.push('- Text might be dynamically loaded - try wait_for_element first');
    }
    
    if (diagnostic.possibleMatches && diagnostic.possibleMatches.length > 0) {
      report.push('', 'Possible matches found:');
      diagnostic.possibleMatches.forEach((match, i) => {
        report.push(`  ${i + 1}. ${match.description}`);
      });
    }
    
    return report.join('\n');
  }
  
  formatResponse(result, locator) {
    if (!result.elements || result.elements.length === 0) {
      return `No elements found matching: ${JSON.stringify(locator)}`;
    }
    
    const elements = result.elements.slice(0, 5); // Limit display
    const descriptions = elements.map((el, i) => {
      const tag = el.tagName.toLowerCase();
      const text = el.text || el.value || el.placeholder || '';
      const role = el.role ? `[role="${el.role}"]` : '';
      const id = el.id ? `#${el.id}` : '';
      
      return `[${i}] <${tag}${id}${role}> "${text.substring(0, 50)}${text.length > 50 ? '...' : ''}"`;
    });
    
    return `Found ${result.count} element(s): ${descriptions}`;
  }
}

// Export for use in browser tools
export default SmartFindElementTool;