// Session History Tool - Retrieves conversation history from previous sessions
// Allows the agent to access context from past conversations

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

/**
 * Tool to retrieve conversation history from previous chat sessions.
 * Useful when user references past conversations or needs context continuity.
 */
export class GetSessionHistoryTool extends BrowserTool {
  constructor() {
    super(
      "get_session_history",
      "Retrieve conversation history from a previous chat session. Use when user references past conversations (e.g., 'in our last chat', 'what we discussed before', 'continue from where we left off'). Returns formatted message history.",
      z.object({
        sessionIndex: z.number().min(0).max(10).default(0).optional()
          .describe("Which previous session to retrieve (0 = most recent previous session, 1 = second most recent, etc.)"),
        maxMessages: z.number().min(5).max(100).default(50).optional()
          .describe("Maximum number of messages to retrieve"),
        excludeToolMessages: z.boolean().default(false).optional()
          .describe("If true, exclude tool call/result messages for a cleaner summary")
      })
    );
    
    // Store reference to current session ID (set by agent when tool is initialized)
    this.currentSessionId = null;
  }

  /**
   * Set the current session ID to exclude from results
   * @param {string} sessionId - Current session ID
   */
  setCurrentSessionId(sessionId) {
    this.currentSessionId = sessionId;
  }

  /**
   * Inherit configuration from parent agent (called by ReactGraph)
   * @param {object} config - Configuration from parent
   */
  inherit(config) {
    if (config.sessionId) {
      this.setCurrentSessionId(config.sessionId);
    }
  }

  /**
   * Format messages into readable text
   * @param {Array} messages - Array of message objects
   * @param {boolean} excludeToolMessages - Whether to exclude tool messages
   * @param {number} maxMessages - Maximum messages to include
   * @returns {string} Formatted message history
   */
  formatMessages(messages, excludeToolMessages = false, maxMessages = 50) {
    const MAX_CHARS = 32000; // ~8k tokens
    let formattedContent = '';
    let totalChars = 0;
    let messageCount = 0;
    let truncated = false;
    
    for (const msg of messages) {
      if (messageCount >= maxMessages) {
        truncated = true;
        break;
      }
      
      let role = '';
      if (msg.type === 'HumanMessage') role = 'User';
      else if (msg.type === 'AIMessage') role = 'Assistant';
      else if (msg.type === 'SystemMessage') {
        // Skip system messages (page content) - they're not conversational
        continue;
      }
      else if (msg.type === 'ToolMessage') {
        if (excludeToolMessages) continue;
        role = `Tool (${msg.name || 'unknown'})`;
      }
      else continue; // Skip unknown message types
      
      // Extract content
      const content = typeof msg.content === 'string' ? msg.content : JSON.stringify(msg.content);
      if (!content || content.trim() === '') continue;
      
      // Truncate very long individual messages
      const truncatedContent = content.length > 2000 ? content.slice(0, 2000) + '...(truncated)' : content;
      const line = `${role}: ${truncatedContent}\n\n`;
      
      if (totalChars + line.length > MAX_CHARS) {
        truncated = true;
        break;
      }
      
      formattedContent += line;
      totalChars += line.length;
      messageCount++;
    }
    
    if (truncated) {
      formattedContent = '(Earlier messages truncated)\n\n' + formattedContent;
    }
    
    return formattedContent.trim();
  }

  async call(args) {
    const { sessionIndex = 0, maxMessages = 50, excludeToolMessages = false } = args;
    
    try {
      // Get all sessions from storage
      const allData = await chrome.storage.local.get(null);
      
      // Find all sessions and sort by lastUpdated
      const storedSessions = [];
      
      for (const [key, value] of Object.entries(allData)) {
        if (key.startsWith('session_') && value?.sessionId && value?.state) {
          // Skip the current session
          if (value.sessionId === this.currentSessionId) continue;
          storedSessions.push({
            sessionId: value.sessionId,
            lastUpdated: value.lastUpdated || 0,
            state: value.state
          });
        }
      }
      
      // Sort by most recent first
      storedSessions.sort((a, b) => b.lastUpdated - a.lastUpdated);
      
      if (storedSessions.length === 0) {
        return "No previous chat sessions found.";
      }
      
      if (sessionIndex >= storedSessions.length) {
        return `Session index ${sessionIndex} not found. Only ${storedSessions.length} previous session(s) available (use index 0-${storedSessions.length - 1}).`;
      }
      
      const targetSession = storedSessions[sessionIndex];
      const messages = targetSession.state.messages || [];
      
      if (messages.length === 0) {
        return `Previous session (index ${sessionIndex}) has no messages.`;
      }
      
      const formattedHistory = this.formatMessages(messages, excludeToolMessages, maxMessages);
      
      // Add metadata header
      const sessionDate = new Date(targetSession.lastUpdated).toLocaleString();
      const header = `=== Previous Session (${sessionIndex === 0 ? 'most recent' : `#${sessionIndex + 1}`}) ===\nDate: ${sessionDate}\nMessages: ${messages.length}\n\n`;
      
      return header + formattedHistory;
      
    } catch (error) {
      console.error('[GetSessionHistoryTool] Error:', error);
      return `Failed to retrieve session history: ${error.message}`;
    }
  }
}

export default GetSessionHistoryTool;
