/**
 * Password Manager Tool for Chrome Extension
 * Uses chrome.passwordsPrivate API to access Chrome's password store
 */

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { passwordManager } from "../../password-manager.js";

export class PasswordManagerTool extends BrowserTool {
  constructor() {
    super(
      "secrets_manager",
      "Access saved passwords from Chrome's password manager. Operations: 'list' returns all service names, 'read <service>' returns login and password for that service",
      z.object({
        tabId: z.number().optional().describe("Tab ID (not used by this tool)"),
        operation: z.enum(['list', 'read']).describe("Operation to perform: 'list' or 'read'"),
        service: z.string().optional().describe("Service name (required for 'read' operation)"),
        reasoning: z.string().describe("Why you need to access these credentials")
      })
    );
  }

  async call({ tabId, operation, service, reasoning }) {
    try {
      // Check if password manager is available
      const isAvailable = await passwordManager.checkAvailability();
      if (!isAvailable) {
        return "Chrome password manager is not available. Please check permissions.";
      }

      // Authenticate user if needed
      try {
        await passwordManager.authenticate();
      } catch (authError) {
        return "Authentication required to access passwords. Please authenticate and try again.";
      }

      if (operation === 'list') {
        // Get all saved passwords
        const passwords = await passwordManager.getSavedPasswords();
        
        if (passwords.length === 0) {
          return "No saved passwords found in Chrome password manager";
        }

        // Extract unique service names
        const services = new Set();
        passwords.forEach(entry => {
          entry.affiliatedDomains.forEach(domain => {
            // Extract service name from domain
            const serviceName = this._extractServiceName(domain.name || domain.url);
            if (serviceName) {
              services.add(serviceName);
            }
          });
        });

        return `Available services: ${Array.from(services).join(', ')}`;
      }

      if (operation === 'read') {
        if (!service) {
          return "Error: Service name is required for 'read' operation";
        }

        try {
          // Get credentials for the specific service
          const credentials = await passwordManager.getCredentialsForService(service);
          
          // Return credentials in the expected format
          return JSON.stringify({
            service: service,
            login: credentials.login,
            password: credentials.password
          });
        } catch (error) {
          return `No credentials found for service: ${service}. Please save the credentials in Chrome's password manager (chrome://settings/passwords).`;
        }
      }

      return "Invalid operation. Use 'list' or 'read <service>'";
      
    } catch (error) {
      console.error('PasswordManagerTool error:', error);
      return `Password manager error: ${error.message}`;
    }
  }

  /**
   * Extract service name from URL or domain name
   */
  _extractServiceName(urlOrName) {
    if (!urlOrName) return null;
    
    // Remove protocol
    let domain = urlOrName.replace(/^https?:\/\//, '');
    
    // Remove path
    domain = domain.split('/')[0];
    
    // Remove www
    domain = domain.replace(/^www\./, '');
    
    // Extract main domain name
    const parts = domain.split('.');
    if (parts.length >= 2) {
      // Return the main domain name (e.g., 'linkedin' from 'linkedin.com')
      return parts[parts.length - 2];
    }
    
    return domain;
  }

}

// Export for use in extension tools
export default PasswordManagerTool;