/**
 * ParallelTool - Execute multiple tools in parallel
 *
 * This meta-tool allows explicit parallel execution of multiple tools.
 * Instead of relying on LLM to return multiple tool_calls (non-deterministic),
 * this provides a structured way to request parallel execution.
 *
 * Example usage by LLM:
 * {
 *   "name": "parallel",
 *   "args": {
 *     "tool_uses": [
 *       { "name": "subagent", "args": { "systemPrompt": "Task 1: Navigate to URL_1. Do X." } },
 *       { "name": "subagent", "args": { "systemPrompt": "Task 2: Navigate to URL_2. Do X." } }
 *     ]
 *   }
 * }
 *
 * ReactGraph will detect this and execute all tool_uses in parallel.
 */

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

// Maximum tools that can be executed in parallel
const MAX_PARALLEL_TOOLS = 10;

export class ParallelTool extends BrowserTool {
  constructor() {
    super(
      "parallel",
      `Execute multiple tools in parallel for maximum efficiency.

**WHEN TO USE:**
- When you need to check/analyze multiple already-opened tabs
- When you need to run different independent tools simultaneously
- When tasks don't depend on each other's results

**ARCHITECTURE:**
- All tool_uses execute CONCURRENTLY (not sequentially)
- Results are returned in order once ALL complete
- Faster than calling tools one-by-one

**BEST PATTERN - Assign subagents to specific tabs:**
First open tabs with navigate_to_url(newTab: true), note the tabIds, then:
{
  "tool_uses": [
    { "name": "subagent", "args": { "systemPrompt": "Extract price and specs from current page", "tabId": 111 } },
    { "name": "subagent", "args": { "systemPrompt": "Extract price and specs from current page", "tabId": 222 } },
    { "name": "subagent", "args": { "systemPrompt": "Extract price and specs from current page", "tabId": 333 } }
  ]
}

**KEY: With tabId, subagent works on existing tab - NO redundant navigation!**

**IMPORTANT:**
- Maximum ${MAX_PARALLEL_TOOLS} tools per call
- Each tool_use must specify "name" and "args"
- Results are aggregated and returned together`,
      z.object({
        tool_uses: z.array(z.object({
          name: z.string().describe("Name of the tool to execute"),
          args: z.record(z.any()).describe("Arguments to pass to the tool")
        })).min(1).max(MAX_PARALLEL_TOOLS).describe("Array of tools to execute in parallel")
      })
    );
  }

  /**
   * This tool is a META-TOOL - it doesn't execute directly.
   * ReactGraph.toolsNode detects this tool and expands it into parallel execution.
   * This call() method is a fallback that should never be reached.
   */
  async call(args) {
    // This should never be called directly - ReactGraph handles expansion
    console.warn('[ParallelTool] Direct call() invoked - this should be handled by ReactGraph');
    return {
      error: "multi_tool_use.parallel is a meta-tool. It should be expanded by the agent runtime.",
      hint: "Ensure ReactGraph.toolsNode handles this tool specially."
    };
  }
}

export default ParallelTool;
