// Navigation-related browser automation tools

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { getCurrentActiveTab, getValidTabId, waitForPageReady, isRestrictedUrl } from "./utils.js";


const DEFAULT_TIMEOUT_MS = 60000;

// HTTP status code descriptions
function getHttpStatusText(statusCode) {
  const statusTexts = {
    400: 'Bad Request',
    401: 'Unauthorized',
    403: 'Forbidden',
    404: 'Not Found',
    405: 'Method Not Allowed',
    408: 'Request Timeout',
    429: 'Too Many Requests',
    500: 'Internal Server Error',
    502: 'Bad Gateway',
    503: 'Service Unavailable',
    504: 'Gateway Timeout'
  };
  return statusTexts[statusCode] || 'Error';
}
// Tool to navigate to a URL
export class NavigateToURLTool extends BrowserTool {
  constructor() {
    super(
      "navigate_to_url",
      "Navigate the browser to a specific URL and wait for the page to load",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID to navigate (optional - will use active tab if not specified)"),
        url: z.string().url().describe("URL to navigate to"),
        timeoutMs: z.number().min(5000).max(120000).default(45000).nullable().optional()
          .describe("Navigation timeout in milliseconds (5s-2min, default 45s)")
      })
    );
  }

  // Helper function to wait for tab to complete loading
  // Note: MV3 keeps service worker alive while event handler promise is pending,
  // so no keepalive hack is needed
  waitForTabComplete(tabId, timeoutMs = DEFAULT_TIMEOUT_MS) {
    return new Promise((resolve, reject) => {
      let timeoutId;
      let pollIntervalId;
      let hasResolved = false;
      let listener;
      let errorListener;
      let httpStatusListener;
      let domContentLoadedListener;
      let navigationError = null;
      let httpStatusCode = null;

      const cleanup = () => {
        if (listener) {
          chrome.tabs.onUpdated.removeListener(listener);
        }
        if (errorListener) {
          chrome.webNavigation.onErrorOccurred.removeListener(errorListener);
        }
        if (httpStatusListener) {
          chrome.webRequest.onCompleted.removeListener(httpStatusListener);
        }
        if (domContentLoadedListener) {
          chrome.webNavigation.onDOMContentLoaded.removeListener(domContentLoadedListener);
        }
        if (timeoutId) {
          clearTimeout(timeoutId);
        }
        if (pollIntervalId) {
          clearInterval(pollIntervalId);
        }
      };

      const resolveOnce = (tab) => {
        if (!hasResolved) {
          hasResolved = true;
          cleanup();
          // Attach HTTP status code to the tab object for the caller
          tab.httpStatusCode = httpStatusCode;
          resolve(tab);
        }
      };

      const rejectOnce = (error) => {
        if (!hasResolved) {
          hasResolved = true;
          cleanup();
          reject(error);
        }
      };

      listener = (changedTabId, changeInfo, tab) => {
        if (changedTabId === tabId) {
          // Check for completion
          if (changeInfo.status === 'complete') {
            console.log(`Tab ${tabId} completed loading: ${tab.url}`);
            // Check if we captured a navigation error
            if (navigationError) {
              console.error(`❌ Tab ${tabId} completed but had navigation error: ${navigationError}`);
              rejectOnce(new Error(`Navigation failed: ${navigationError}`));
              return;
            }
            resolveOnce(tab);
          }
          // Check for navigation errors
          else if (tab.url && tab.url.includes('chrome-error://')) {
            console.error(`❌ Tab ${tabId} navigation error: ${tab.url}`);
            rejectOnce(new Error(`Navigation error: ${tab.url}`));
          }
          // Check for URL changes that might indicate navigation issues
          else if (changeInfo.url && changeInfo.url.startsWith('chrome://')) {
            console.warn(`⚠️ Tab ${tabId} navigated to chrome:// URL: ${changeInfo.url}`);
            rejectOnce(new Error(`Navigation redirected to internal page: ${changeInfo.url}`));
          }
        }
      };

      // Set up configurable timeout
      timeoutId = setTimeout(() => {
        console.error(`❌ Navigation timeout after ${timeoutMs}ms for tab ${tabId}`);
        rejectOnce(new Error(`Navigation timeout after ${timeoutMs}ms`));
      }, timeoutMs);

      chrome.tabs.onUpdated.addListener(listener);

      // Listen for navigation errors (DNS failures, connection refused, etc.)
      errorListener = (details) => {
        if (details.tabId === tabId && details.frameId === 0) {
          // Main frame navigation error
          navigationError = details.error;
          console.error(`❌ Navigation error for tab ${tabId}: ${details.error}`);
          // Don't reject immediately - let the tab complete handler check the error
          // This ensures we capture the error even if tab shows as "complete"
        }
      };
      chrome.webNavigation.onErrorOccurred.addListener(errorListener);

      // Listen for DOMContentLoaded - accept page as soon as DOM is ready
      // This is MUCH faster than waiting for all resources (tab.status === 'complete')
      // Amazon, Best Buy etc. have tracking pixels that never finish loading
      domContentLoadedListener = async (details) => {
        if (details.tabId === tabId && details.frameId === 0) {
          console.log(`✅ DOM ready for tab ${tabId}: ${details.url}`);
          try {
            const tab = await chrome.tabs.get(tabId);
            if (tab && !tab.url?.includes('chrome-error://')) {
              tab.httpStatusCode = httpStatusCode;
              resolveOnce(tab);
            }
          } catch (e) {
            console.warn(`Failed to get tab after DOMContentLoaded: ${e.message}`);
          }
        }
      };
      chrome.webNavigation.onDOMContentLoaded.addListener(domContentLoadedListener);

      // Listen for HTTP status codes (404, 503, etc.)
      httpStatusListener = (details) => {
        if (details.tabId === tabId && details.type === 'main_frame') {
          httpStatusCode = details.statusCode;
          if (details.statusCode >= 400) {
            console.warn(`⚠️ HTTP ${details.statusCode} for tab ${tabId}: ${details.url}`);
          }
        }
      };
      chrome.webRequest.onCompleted.addListener(
        httpStatusListener,
        { urls: ['<all_urls>'], tabId: tabId }
      );

      // Polling fallback: check tab status every 500ms
      // In headless mode, also accept 'loading' with valid URL as success
      // because getPageState in ReactGraph.ts handles waiting for content
      let loadingStartTime = null;
      const LOADING_TIMEOUT_ACCEPT = 15000; // Accept 'loading' status after 15s if URL is correct

      pollIntervalId = setInterval(async () => {
        if (hasResolved) {
          return;
        }
        try {
          const tab = await chrome.tabs.get(tabId);
          console.log(`🔍 Poll: Tab ${tabId} status=${tab.status}, url=${tab.url?.substring(0, 50)}`);

          // Accept complete OR loading with valid URL (headless mode workaround)
          const hasValidUrl = tab.url && !tab.url.startsWith('about:') && !tab.url.startsWith('chrome://');
          if (tab.status === 'complete' && hasValidUrl) {
            // Check if we captured a navigation error
            if (navigationError) {
              console.error(`❌ Tab ${tabId} completed but had navigation error (polling): ${navigationError}`);
              rejectOnce(new Error(`Navigation failed: ${navigationError}`));
              return;
            }
            console.log(`✅ Tab ${tabId} ready (status=${tab.status}): ${tab.url}`);
            resolveOnce(tab);
          } else if (tab.status === 'loading' && hasValidUrl) {
            // Track when we first saw the target URL in loading state
            if (!loadingStartTime) {
              loadingStartTime = Date.now();
            }
            // If page has been loading for >15s with correct URL, accept it
            // (some pages like Wikipedia take forever due to tracking/analytics)
            const loadingDuration = Date.now() - loadingStartTime;
            if (loadingDuration > LOADING_TIMEOUT_ACCEPT) {
              console.log(`✅ Tab ${tabId} accepting after ${Math.round(loadingDuration / 1000)}s loading: ${tab.url}`);
              resolveOnce(tab);
            }
          } else if (tab.url && tab.url.includes('chrome-error://')) {
            rejectOnce(new Error(`Tab error detected (polling): ${tab.url}`));
          }
        } catch (e) {
          rejectOnce(new Error(`Tab polling failed: ${e.message}`));
        }
      }, 500);

      // Also check if tab is already complete
      chrome.tabs.get(tabId, (tab) => {
        if (chrome.runtime.lastError) {
          rejectOnce(new Error(`Tab ${tabId} not found: ${chrome.runtime.lastError.message}`));
          return;
        }

        if (tab) {
          if (tab.status === 'complete' && tab.url && !tab.url.startsWith('about:')) {
            console.log(`Tab ${tabId} already complete: ${tab.url}`);
            resolveOnce(tab);
          } else if (tab.url && tab.url.includes('chrome-error://')) {
            rejectOnce(new Error(`Tab already in error state: ${tab.url}`));
          } else {
            console.log(`⏳ Tab ${tabId} still loading: ${tab.url} (status: ${tab.status})`);
          }
        }
      });
    });
  }


  async call({ tabId, url, timeoutMs = DEFAULT_TIMEOUT_MS }) {
    try {
      let targetTabId = tabId;
      let targetTab = null;

      // Helper to check if URL is a restricted/extension page that can't be navigated
      const isExtensionOrRestrictedUrl = (tabUrl) => {
        if (!tabUrl) return false;
        return tabUrl.startsWith('chrome-extension://') ||
               tabUrl.startsWith('chrome://') ||
               tabUrl.startsWith('about:') ||
               tabUrl.startsWith('edge://') ||
               tabUrl.startsWith('brave://');
      };

      // First validate that the tab exists or get/create appropriate tab
      // tabId behavior:
      //   - number: use that specific tab
      //   - null: create new tab (for subagent isolation)
      //   - undefined: fall back to active tab (default behavior)
      try {
        if (typeof tabId === 'number') {
          // Use specified tab
          targetTab = await chrome.tabs.get(tabId);
          targetTabId = tabId;
        } else if (tabId === null) {
          // Explicitly create new tab (subagent isolation mode)
          console.log(`📄 Creating new tab for isolated navigation to ${url}`);
          const newTab = await chrome.tabs.create({ url, active: false });
          console.log(`📄 Created isolated tab ${newTab.id}`);

          // Wait for the new tab to load
          const completedTab = await this.waitForTabComplete(newTab.id, timeoutMs);
          await waitForPageReady(newTab.id);

          // Wait for content script with timeout
          const contentScriptTimeout = Math.max(timeoutMs - 15000, 30000);
          const startTime = Date.now();
          let contentScriptReady = false;

          while (!contentScriptReady && (Date.now() - startTime) < contentScriptTimeout) {
            try {
              const pingResponse = await chrome.tabs.sendMessage(newTab.id, { type: 'ping' });
              if (pingResponse && pingResponse.status === 'pong') {
                contentScriptReady = true;
              }
            } catch (e) {
              await new Promise(resolve => setTimeout(resolve, 500));
            }
          }

          console.log(`✅ Isolated tab ${newTab.id} navigated to ${completedTab.url}`);
          // Return result with tabId so caller can track it
          return JSON.stringify({
            success: true,
            message: `Created new tab and navigated to ${url}`,
            url: completedTab.url,
            tabId: newTab.id
          });
        } else {
          // tabId is undefined - fall back to active tab (default behavior)
          const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
          let activeTab = tabs[0];

          if (!activeTab) {
            throw new Error('No active tab found');
          }
          targetTab = activeTab;
          targetTabId = activeTab.id;
        }
      } catch (tabError) {
        if (tabError.message?.includes('No tab with id')) {
          // Tab doesn't exist, fallback to active web tab
          const activeTabs = await chrome.tabs.query({ active: true, currentWindow: true });
          const activeTab = activeTabs[0];

          if (activeTab) {
            console.log(`⚠️ Tab ${tabId} not found, using active tab ${activeTab.id}`);
            targetTab = activeTab;
            targetTabId = activeTab.id;
          } else {
            // Create new tab as last resort
            const newTab = await chrome.tabs.create({ url: 'about:blank' });
            console.log(`📄 Created new tab ${newTab.id} as fallback`);
            targetTab = newTab;
            targetTabId = newTab.id;
          }
        } else {
          throw tabError;
        }
      }

      // Check if target tab is an extension/restricted page - if so, create a new tab
      if (isExtensionOrRestrictedUrl(targetTab.url)) {
        console.log(`⚠️ Target tab ${targetTabId} is extension/restricted page (${targetTab.url}), creating new tab`);

        // Try to find an existing web tab first
        const allTabs = await chrome.tabs.query({ currentWindow: true });
        const webTab = allTabs.find(t => t.url && !isExtensionOrRestrictedUrl(t.url));

        if (webTab) {
          console.log(`📄 Found existing web tab ${webTab.id}: ${webTab.url}`);
          targetTab = webTab;
          targetTabId = webTab.id;
        } else {
          // No web tabs exist - create a new one
          const newTab = await chrome.tabs.create({ url, active: true });
          console.log(`📄 Created new web tab ${newTab.id} for navigation`);

          // Wait for the new tab to load
          const completedTab = await this.waitForTabComplete(newTab.id, timeoutMs);
          await waitForPageReady(newTab.id);

          // Wait for content script
          const contentScriptTimeout = Math.max(timeoutMs - 15000, 30000);
          const startTime = Date.now();
          let contentScriptReady = false;

          while (!contentScriptReady && (Date.now() - startTime) < contentScriptTimeout) {
            try {
              const pingResponse = await chrome.tabs.sendMessage(newTab.id, { type: 'ping' });
              if (pingResponse && pingResponse.status === 'pong') {
                contentScriptReady = true;
              }
            } catch (e) {
              await new Promise(resolve => setTimeout(resolve, 500));
            }
          }

          console.log(`✅ New tab created and navigated to ${completedTab.url}`);
          return `Created new tab and navigated to ${url}. Current URL: ${completedTab.url}`;
        }
      }

      console.log(`🧭 Navigating tab ${targetTabId} to ${url}`);

      // Get the current URL before navigation for comparison
      const beforeUrl = targetTab.url;

      // IMPORTANT: Set up listeners BEFORE starting navigation to catch DOMContentLoaded
      // This fixes the race condition where DOMContentLoaded fires before listener is registered
      console.log(`⏳ Waiting for page to load (timeout: ${timeoutMs}ms)...`);
      const waitPromise = this.waitForTabComplete(targetTabId, timeoutMs);

      // Navigate to the URL AFTER listeners are set up
      await chrome.tabs.update(targetTabId, { url });

      // Now wait for completion
      const completedTab = await waitPromise;

      // Verify navigation actually occurred
      if (completedTab.url === beforeUrl && completedTab.url !== url) {
        throw new Error(`Navigation failed: URL did not change from ${beforeUrl}. Expected: ${url}, Actual: ${completedTab.url}`);
      }

      // Additional verification: check if we ended up at the expected URL
      if (!completedTab.url || completedTab.url.includes('chrome-error://')) {
        throw new Error(`Navigation failed: Page loaded with error. URL: ${completedTab.url || 'unknown'}`);
      }

      // Wait for DOM and dynamic content to be ready
      await waitForPageReady(targetTabId);

      // Wait for content script to be ready after navigation
      // Ping until content script responds or remaining timeout expires
      const contentScriptTimeout = Math.max(timeoutMs - 15000, 30000); // At least 30s for content script
      const pingInterval = 500;
      const startTime = Date.now();
      let contentScriptReady = false;
      let attempts = 0;

      console.log(`⏳ Waiting for content script (timeout: ${contentScriptTimeout}ms)...`);

      while (!contentScriptReady && (Date.now() - startTime) < contentScriptTimeout) {
        attempts++;
        try {
          const pingResponse = await chrome.tabs.sendMessage(targetTabId, { type: 'ping' });
          if (pingResponse && pingResponse.status === 'pong') {
            contentScriptReady = true;
            const elapsed = Date.now() - startTime;
            console.log(`✅ Content script ready after ${attempts} attempts (${elapsed}ms)`);
          }
        } catch (e) {
          // Content script not ready yet - this is expected
          if (attempts === 1) {
            console.log('   Waiting for content script to inject...');
          } else if (attempts % 20 === 0) {
            const elapsed = Date.now() - startTime;
            console.log(`   Still waiting for content script... (${elapsed}ms, ${attempts} attempts)`);
          }
          await new Promise(resolve => setTimeout(resolve, pingInterval));
        }
      }

      if (!contentScriptReady) {
        const elapsed = Date.now() - startTime;
        console.warn(`⚠️ Content script not ready after ${elapsed}ms - page may be inaccessible`);

        // If content script failed and we have an HTTP error status code, report it
        if (completedTab.httpStatusCode && completedTab.httpStatusCode >= 400) {
          const statusText = getHttpStatusText(completedTab.httpStatusCode);
          throw new Error(`HTTP ${completedTab.httpStatusCode} ${statusText}: Server returned error status with no readable content`);
        }

        // Content script not ready but no HTTP error - page may still be loading
        // Return partial success - don't block the agent but indicate content may not be extractable yet
        console.log(`Navigation completed to ${completedTab.url} but content script not ready - page may still be loading`);
        return `Navigated to ${url}. Page loaded but content may still be loading. Current URL: ${completedTab.url}. Note: If page content is not accessible, wait a moment and try reading the page again.`;
      }

      // Include HTTP status code in success message if it was an error (4xx/5xx)
      if (completedTab.httpStatusCode && completedTab.httpStatusCode >= 400) {
        const statusText = getHttpStatusText(completedTab.httpStatusCode);
        console.log(`Successfully navigated to ${completedTab.url} (HTTP ${completedTab.httpStatusCode} ${statusText})`);
        return `Navigated to ${url}. Server returned HTTP ${completedTab.httpStatusCode} ${statusText}. Current URL: ${completedTab.url}`;
      }

      console.log(`Successfully navigated to ${completedTab.url} and page is ready`);
      return `Successfully navigated to ${url}, page finished loading, and DOM is ready. Current URL: ${completedTab.url}`;
    } catch (error) {
      console.error(`❌ Navigation failed:`, error);
      throw new Error(`Navigation failed: ${error.message}`);
    }
  }
}

// Tool to navigate browser history (back/forward/refresh)
export class NavigateTool extends BrowserTool {
  constructor() {
    super(
      "navigate",
      "Navigate browser history or refresh the page. Use 'back' to go to previous page, 'forward' to go to next page, or 'refresh' to reload current page.",
      z.object({
        action: z.enum(["back", "forward", "refresh"]).describe("Navigation action: 'back' (previous page), 'forward' (next page), or 'refresh' (reload current page)"),
        tabId: z.number().nullable().optional().describe("Tab ID to navigate (optional - uses active tab if not specified)")
      })
    );
  }

  async call({ action, tabId }) {
    try {
      // Get target tab ID
      let targetTabId = tabId;
      if (!targetTabId) {
        const activeTab = await getCurrentActiveTab();
        targetTabId = activeTab.id;
      }

      // Get current tab info for logging
      const currentTab = await chrome.tabs.get(targetTabId);
      const beforeUrl = currentTab.url;

      console.log(`🧭 Navigate ${action} on tab ${targetTabId} (current: ${beforeUrl})`);

      let result;
      switch (action) {
        case "back":
          result = await this.goBack(targetTabId, beforeUrl);
          break;
        case "forward":
          result = await this.goForward(targetTabId, beforeUrl);
          break;
        case "refresh":
          result = await this.refresh(targetTabId, beforeUrl);
          break;
        default:
          throw new Error(`Unknown action: ${action}`);
      }

      return result;
    } catch (error) {
      console.error(`❌ Navigate ${action} failed:`, error);
      throw new Error(`Navigate ${action} failed: ${error.message}`);
    }
  }

  async goBack(tabId, beforeUrl) {
    // Use chrome.tabs.goBack() - available in Chrome 72+
    await chrome.tabs.goBack(tabId);

    // Wait for navigation to complete
    await this.waitForNavigation(tabId);

    const tab = await chrome.tabs.get(tabId);
    if (tab.url === beforeUrl) {
      return "No previous page in history to go back to.";
    }

    console.log(`✅ Navigated back from ${beforeUrl} to ${tab.url}`);
    return `Navigated back from ${beforeUrl} to ${tab.url}`;
  }

  async goForward(tabId, beforeUrl) {
    // Use chrome.tabs.goForward() - available in Chrome 72+
    await chrome.tabs.goForward(tabId);

    // Wait for navigation to complete
    await this.waitForNavigation(tabId);

    const tab = await chrome.tabs.get(tabId);
    if (tab.url === beforeUrl) {
      return "No forward page in history to navigate to.";
    }

    console.log(`✅ Navigated forward from ${beforeUrl} to ${tab.url}`);
    return `Navigated forward from ${beforeUrl} to ${tab.url}`;
  }

  async refresh(tabId, beforeUrl) {
    // Use chrome.tabs.reload()
    await chrome.tabs.reload(tabId);

    // Wait for page to reload
    await this.waitForNavigation(tabId);

    console.log(`✅ Refreshed page: ${beforeUrl}`);
    return `Refreshed page: ${beforeUrl}`;
  }

  // Wait for tab to complete loading after navigation action
  waitForNavigation(tabId, timeoutMs = 30000) {
    return new Promise((resolve, reject) => {
      let timeoutId;
      let listener;

      const cleanup = () => {
        if (listener) {
          chrome.tabs.onUpdated.removeListener(listener);
        }
        if (timeoutId) {
          clearTimeout(timeoutId);
        }
      };

      listener = (changedTabId, changeInfo, tab) => {
        if (changedTabId === tabId && changeInfo.status === 'complete') {
          cleanup();
          resolve(tab);
        }
      };

      timeoutId = setTimeout(() => {
        cleanup();
        // Resolve anyway after timeout - page may be slow but navigation likely occurred
        chrome.tabs.get(tabId).then(resolve).catch(reject);
      }, timeoutMs);

      chrome.tabs.onUpdated.addListener(listener);

      // Check if already complete
      chrome.tabs.get(tabId, (tab) => {
        if (chrome.runtime.lastError) {
          cleanup();
          reject(new Error(chrome.runtime.lastError.message));
          return;
        }
        if (tab.status === 'complete') {
          // Give a small delay for the navigation to actually start
          setTimeout(() => {
            chrome.tabs.get(tabId, (newTab) => {
              if (newTab && newTab.status === 'complete') {
                cleanup();
                resolve(newTab);
              }
            });
          }, 100);
        }
      });
    });
  }
}