/**
 * Moondream Visual Tools for Extension (OSS Agent-style)
 * Atomic tools that find AND execute in one call
 *
 * Based on OSS Agent createMoondreamVisualClickTool / createMoondreamVisualTypeTool
 */

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { getValidTabId } from "./utils.js";

const MOONDREAM_API_URL = "https://api.moondream.ai/v1/point";

/**
 * Visual Click Tool - Find and click element using Moondream (ATOMIC)
 *
 * Replicates OSS Agent approach: finds AND clicks in single tool call
 */
export class MoondreamVisualClickTool extends BrowserTool {
  constructor(apiKey) {
    super(
      "visual_click",
      "Click on any element by describing what it looks like. Pass a clear description like 'blue submit button', 'search icon', 'first checkbox', 'close button in modal', etc. Use when index-based clicking fails.",
      z.object({
        tabId: z.number().optional().describe("Tab ID (optional - will use active tab)"),
        instruction: z.string().describe("Clear visual description of what to click (e.g., 'Submit button', 'search icon', 'Google logo')")
      })
    );

    this.apiKey = apiKey;
  }

  async call({ tabId, instruction }) {
    if (!this.apiKey) {
      return JSON.stringify({
        ok: false,
        error: "Moondream API key not configured. Set MOONDREAM_API_KEY environment variable."
      });
    }

    try {
      const validTabId = await getValidTabId(tabId);

      console.log(`🎯 [visual_click] Clicking: "${instruction}"`);

      // Step 1: Get viewport size
      const viewport = await chrome.tabs.sendMessage(validTabId, {
        type: 'GET_VIEWPORT_SIZE'
      });

      if (!viewport || viewport.error) {
        throw new Error(`Failed to get viewport size: ${viewport?.error}`);
      }

      const { width: viewportWidth, height: viewportHeight } = viewport;

      // Step 2: Take screenshot
      const screenshot = await this._takeScreenshot(validTabId);

      // Step 3: Call Moondream API
      const coordinates = await this._callMoondreamPoint(screenshot, instruction, viewportWidth, viewportHeight);

      if (!coordinates) {
        return JSON.stringify({
          ok: false,
          error: `No "${instruction}" found on the page`
        });
      }

      console.log(`[visual_click] Found at (${coordinates.x}, ${coordinates.y}), clicking...`);

      // Step 4: Click at coordinates (ATOMIC - do it immediately)
      await this._clickAtCoordinates(validTabId, coordinates.x, coordinates.y);

      await new Promise(resolve => setTimeout(resolve, 1000));

      return JSON.stringify({
        ok: true,
        output: {
          coordinates: coordinates,
          description: `Clicked "${instruction}" at (${coordinates.x}, ${coordinates.y})`
        }
      });

    } catch (error) {
      console.error('[visual_click] Error:', error);
      return JSON.stringify({
        ok: false,
        error: `Visual click failed: ${error.message}`
      });
    }
  }

  async _takeScreenshot(tabId) {
    return new Promise((resolve, reject) => {
      chrome.tabs.captureVisibleTab(null, {
        format: 'png',
        quality: 80
      }, (dataUrl) => {
        if (chrome.runtime.lastError) {
          reject(new Error(chrome.runtime.lastError.message));
        } else {
          resolve(dataUrl);
        }
      });
    });
  }

  async _callMoondreamPoint(screenshot, instruction, viewportWidth, viewportHeight) {
    const response = await fetch(MOONDREAM_API_URL, {
      method: "POST",
      headers: {
        "X-Moondream-Auth": this.apiKey,
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        image_url: screenshot,
        object: instruction
      })
    });

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      const errorMessage = errorData.error?.message || `API error: ${response.status}`;
      throw new Error(`Moondream API error: ${errorMessage}`);
    }

    const data = await response.json();

    if (!data.points || data.points.length === 0) {
      console.warn(`[visual_click] No "${instruction}" found`);
      return null;
    }

    const point = data.points[0];
    const x = Math.round(point.x * viewportWidth);
    const y = Math.round(point.y * viewportHeight);

    return { x, y };
  }

  async _clickAtCoordinates(tabId, x, y) {
    return chrome.tabs.sendMessage(tabId, {
      type: 'CLICK',
      data: {
        target: { x, y },
        button: 'left'
      }
    });
  }
}

/**
 * Visual Type Tool - Find input and type text using Moondream (ATOMIC)
 *
 * Replicates OSS Agent approach: finds AND types in single tool call
 */
export class MoondreamVisualTypeTool extends BrowserTool {
  constructor(apiKey) {
    super(
      "visual_type",
      "Type into an input field by describing what it looks like. Pass a clear description like 'email input', 'search box', 'password field'. Use when index-based filling fails.",
      z.object({
        tabId: z.number().optional().describe("Tab ID (optional - will use active tab)"),
        instruction: z.string().describe("Clear visual description of input field (e.g., 'email input', 'search box')"),
        text: z.string().describe("Text to type into the field")
      })
    );

    this.apiKey = apiKey;
  }

  async call({ tabId, instruction, text }) {
    if (!this.apiKey) {
      return JSON.stringify({
        ok: false,
        error: "Moondream API key not configured. Set MOONDREAM_API_KEY environment variable."
      });
    }

    try {
      const validTabId = await getValidTabId(tabId);

      console.log(`⌨️ [visual_type] Typing "${text}" into: "${instruction}"`);

      // Step 1: Get viewport size
      const viewport = await chrome.tabs.sendMessage(validTabId, {
        type: 'GET_VIEWPORT_SIZE'
      });

      if (!viewport || viewport.error) {
        throw new Error(`Failed to get viewport size: ${viewport?.error}`);
      }

      const { width: viewportWidth, height: viewportHeight } = viewport;

      // Step 2: Take screenshot
      const screenshot = await this._takeScreenshot(validTabId);

      // Step 3: Call Moondream API
      const coordinates = await this._callMoondreamPoint(screenshot, instruction, viewportWidth, viewportHeight);

      if (!coordinates) {
        return JSON.stringify({
          ok: false,
          error: `No "${instruction}" found on the page`
        });
      }

      console.log(`[visual_type] Found at (${coordinates.x}, ${coordinates.y}), typing...`);

      // Step 4: Click to focus
      await this._clickAtCoordinates(validTabId, coordinates.x, coordinates.y);
      await new Promise(resolve => setTimeout(resolve, 200));

      // Step 5: Type text (ATOMIC - do it immediately)
      await this._typeText(validTabId, coordinates, text);
      await new Promise(resolve => setTimeout(resolve, 500));

      return JSON.stringify({
        ok: true,
        output: {
          coordinates: coordinates,
          description: `Typed "${text}" into "${instruction}" at (${coordinates.x}, ${coordinates.y})`
        }
      });

    } catch (error) {
      console.error('[visual_type] Error:', error);
      return JSON.stringify({
        ok: false,
        error: `Visual type failed: ${error.message}`
      });
    }
  }

  async _takeScreenshot(tabId) {
    return new Promise((resolve, reject) => {
      chrome.tabs.captureVisibleTab(null, {
        format: 'png',
        quality: 80
      }, (dataUrl) => {
        if (chrome.runtime.lastError) {
          reject(new Error(chrome.runtime.lastError.message));
        } else {
          resolve(dataUrl);
        }
      });
    });
  }

  async _callMoondreamPoint(screenshot, instruction, viewportWidth, viewportHeight) {
    const response = await fetch(MOONDREAM_API_URL, {
      method: "POST",
      headers: {
        "X-Moondream-Auth": this.apiKey,
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        image_url: screenshot,
        object: instruction
      })
    });

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      const errorMessage = errorData.error?.message || `API error: ${response.status}`;
      throw new Error(`Moondream API error: ${errorMessage}`);
    }

    const data = await response.json();

    if (!data.points || data.points.length === 0) {
      console.warn(`[visual_type] No "${instruction}" found`);
      return null;
    }

    const point = data.points[0];
    const x = Math.round(point.x * viewportWidth);
    const y = Math.round(point.y * viewportHeight);

    return { x, y };
  }

  async _clickAtCoordinates(tabId, x, y) {
    return chrome.tabs.sendMessage(tabId, {
      type: 'CLICK',
      data: {
        target: { x, y },
        button: 'left'
      }
    });
  }

  async _typeText(tabId, coordinates, text) {
    return chrome.tabs.sendMessage(tabId, {
      type: 'TYPE',
      data: {
        target: { x: coordinates.x, y: coordinates.y },
        text: text,
        clearFirst: true
      }
    });
  }
}

/**
 * Done Tool - Signal executor completion
 */
export class DoneTool extends BrowserTool {
  constructor() {
    super(
      "done",
      "Call this when all assigned actions are completed. This signals the executor to return to the planner.",
      z.object({
        summary: z.string().describe("Brief summary of what was accomplished")
      })
    );
  }

  async call({ summary }) {
    console.log(`[Done] Executor completed: ${summary}`);
    return JSON.stringify({
      ok: true,
      output: `Done: ${summary}`
    });
  }
}
