// Miscellaneous browser tools

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { getCurrentActiveTab } from "./utils.js";


// Tool to take a screenshot of the current page with cost-aware options
export class TakeScreenshotTool extends BrowserTool {
  constructor() {
    super(
      "take_screenshot",
      "Take a JPEG screenshot (optionally resized & grayscale) for multimodal analysis. Use when text extraction fails or visual context is needed.",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        maxWidth: z.number().min(320).max(2048).default(1024).nullable().optional()
          .describe("Maximum width in pixels (will maintain aspect ratio)"),
        grayscale: z.boolean().default(false).nullable().optional()
          .describe("Convert to grayscale to reduce token usage when color isn't critical"),
        quality: z.number().min(10).max(90).default(70).nullable().optional()
          .describe("JPEG quality (10-90, higher = better quality but more tokens"),
        detail: z.enum(["low", "high"]).default("low").nullable().optional()
          .describe("Detail level - 'low' for basic layout, 'high' for detailed analysis")
      })
    );
    
    // Rate limiting
    this.lastCaptureTime = 0;
    this.MIN_CAPTURE_INTERVAL_MS = 500; // 2fps max to avoid Chrome throttling
  }

  /**
   * Process an image with canvas for downscaling and grayscale conversion
   * @param {string} dataUrl - Original image data URL
   * @param {Object} options - Processing options
   * @returns {Promise<string>} Processed image data URL
   */
  async processImage(dataUrl, { maxWidth = 1024, grayscale = false, quality = 70, detail = 'low' }) {
    // For low detail mode, use fixed 768px on shortest side (single 85-token tile)
    const targetSize = detail === 'low' ? 768 : maxWidth;
    
    try {
      // Use OffscreenCanvas API which is available in background scripts (Chrome 69+)
      if (typeof OffscreenCanvas !== 'undefined') {
        return await this.processImageWithOffscreenCanvas(dataUrl, { maxWidth, grayscale, quality, detail });
      }
    } catch (error) {
      console.warn('📸 [TakeScreenshotTool] OffscreenCanvas processing failed:', error.message);
    }
    
    console.log('📸 [TakeScreenshotTool] Using raw screenshot (image processing not available)');
    return dataUrl;
  }

  async processImageWithOffscreenCanvas(dataUrl, { maxWidth = 1024, grayscale = false, quality = 70, detail = 'low' }) {
    const targetSize = detail === 'low' ? 768 : maxWidth;
    
    return new Promise((resolve, reject) => {
      // Create an image bitmap from the data URL
      fetch(dataUrl)
        .then(response => response.blob())
        .then(blob => createImageBitmap(blob))
        .then(imageBitmap => {
          // Calculate new dimensions maintaining aspect ratio
          let width = imageBitmap.width;
          let height = imageBitmap.height;
          
          if (detail === 'low') {
            // For low detail, scale so shortest side is 768px
            const scale = Math.min(1, targetSize / Math.min(width, height));
            width = Math.round(width * scale);
            height = Math.round(height * scale);
          } else {
            // For high detail, scale based on maxWidth
            if (width > targetSize) {
              const scale = targetSize / width;
              height = Math.round(height * scale);
              width = targetSize;
            }
          }
          
          // Create offscreen canvas for processing
          const canvas = new OffscreenCanvas(width, height);
          const ctx = canvas.getContext('2d');
          
          // Apply grayscale filter if requested
          if (grayscale) {
            ctx.filter = 'grayscale(100%)';
          }
          
          // Draw image to canvas (this handles the scaling)
          ctx.drawImage(imageBitmap, 0, 0, width, height);
          
          // Convert to blob with specified quality
          const mimeType = quality >= 90 ? 'image/png' : 'image/jpeg';
          const convertQuality = quality >= 90 ? undefined : quality / 100;
          
          canvas.convertToBlob({ type: mimeType, quality: convertQuality })
            .then(blob => {
              const reader = new FileReader();
              reader.onload = () => resolve(reader.result);
              reader.onerror = () => reject(new Error('Failed to convert blob to data URL'));
              reader.readAsDataURL(blob);
            })
            .catch(reject);
        })
        .catch(reject);
    });
  }

  async call({ tabId, maxWidth, grayscale, quality, detail }) {
    // Get default settings from config
    let config = {};
    try {
      // Try to get config from window.vibeAIConfig if available (in extension context)
      if (typeof window !== 'undefined' && window.vibeAIConfig) {
        const configObj = window.vibeAIConfig.getConfig();
        config = {
          maxWidth: configObj.screenshotMaxWidth,
          grayscale: configObj.screenshotGrayscale,
          quality: configObj.screenshotQuality,
          detail: configObj.screenshotDetail
        };
        console.log('Using screenshot config from AIConfigManager:', config);
      }
    } catch (e) {
      console.warn('Could not load screenshot config, using defaults', e);
    }
    
    // Apply any overrides from the tool call
    const settings = {
      maxWidth: maxWidth ?? config.maxWidth ?? 1024,
      grayscale: grayscale ?? config.grayscale ?? false,
      quality: quality ?? config.quality ?? 70,
      detail: detail ?? config.detail ?? 'low'
    };
    
    // Rate limit screenshot capture
    const now = Date.now();
    const timeSinceLastCapture = now - this.lastCaptureTime;
    if (timeSinceLastCapture < this.MIN_CAPTURE_INTERVAL_MS) {
      await new Promise(resolve => setTimeout(resolve, this.MIN_CAPTURE_INTERVAL_MS - timeSinceLastCapture));
    }
    
    try {
      const targetTab = tabId ? { id: tabId } : await getCurrentActiveTab();
      
      // Check if the tab URL is accessible for screenshots
      if (targetTab.url && (targetTab.url.startsWith('chrome://') || targetTab.url.startsWith('chrome-extension://') || targetTab.url === 'about:blank' || !targetTab.url.trim())) {
        throw new Error(`Cannot capture screenshot of system page: ${targetTab.url || 'empty URL'}. Please navigate to a regular webpage first.`);
      }
      
      // Capture the visible tab as JPEG (smaller than PNG)
      const dataUrl = await chrome.tabs.captureVisibleTab(targetTab.windowId, {
        format: 'jpeg',
        quality: Math.round(settings.quality) // Chrome expects integer from 0-100
      });
      
      if (!dataUrl) {
        throw new Error('Failed to capture tab');
      }
      
      // Process the image (resize + grayscale if needed)
      const processedDataUrl = await this.processImage(dataUrl, settings);
      
      // Update last capture time
      this.lastCaptureTime = Date.now();
      
      // Estimate token usage for logging (very rough estimate based on base64 size)
      // OpenAI Vision: 85 base tokens + 170 per 512x512 tile (after resizing)
      const base64Length = processedDataUrl.length - 'data:image/jpeg;base64,'.length;
      const bytes = Math.ceil(base64Length * 0.75); // Base64 to bytes
      const tiles = Math.ceil((bytes / (1024 * 1024)) * 4); // Very rough tiles estimate (1MB ~= 4 tiles)
      const estimatedTokens = 85 + (170 * tiles);
      
      console.log(`Captured screenshot: ${Math.round(bytes / 1024)}KB, ~${estimatedTokens} tokens, ${settings.detail} detail` +
                 `, ${settings.grayscale ? 'grayscale' : 'color'}, ${settings.maxWidth}px max`);
      
      // Simply return the data URL
      return processedDataUrl;
      
    } catch (error) {
      console.error('TakeScreenshotTool error:', error);
      // Return error message instead of throwing to allow graceful fallback
      return `Screenshot capture failed: ${error.message}. Page content extraction is handled internally by the agent.`;
    }
  }
}