/**
 * MemoryManager - Graph-based mind-map memory for the Vibe browser extension
 * Stores memories as nodes with relationships (edges) for contextual understanding
 */

class MemoryManager {
  constructor() {
    this.memories = new Map(); // Legacy text memories
    this.nodes = new Map(); // Graph nodes (observations, tool outputs, reasoning)
    this.edges = new Map(); // Graph edges (relationships between nodes)
    this.loaded = false;
  }

  /**
   * Store a new memory
   * @param {string} content - The content to store
   * @param {Array<string>} tags - Optional tags for categorization
   * @returns {Promise<Object>} The stored memory object
   */
  async storeMemory(content, tags = []) {
    if (!this.loaded) await this.loadMemories();

    const memoryId = `mem_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    const memory = {
      id: memoryId,
      content: content,
      tags: Array.isArray(tags) ? tags : [tags].filter(Boolean),
      createdAt: new Date().toISOString()
    };

    this.memories.set(memoryId, memory);
    await this.saveMemories();

    return memory;
  }

  /**
   * Add a new node to the mind-map graph
   * @param {string} type - Node type ('observation', 'tool_output', 'reasoning', 'goal')
   * @param {string} content - Node content
   * @param {Object} metadata - Additional metadata
   * @returns {Promise<Object>} The created node
   */
  async addNode(type, content, metadata = {}) {
    if (!this.loaded) await this.loadMemories();

    const nodeId = `node_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    const node = {
      id: nodeId,
      type: type,
      content: content,
      metadata: metadata,
      createdAt: new Date().toISOString(),
      connections: [] // Track connected node IDs
    };

    this.nodes.set(nodeId, node);
    await this.saveMemories();

    return node;
  }

  /**
   * Add an edge (relationship) between two nodes
   * @param {string} fromNodeId - Source node ID
   * @param {string} toNodeId - Target node ID
   * @param {string} relationshipType - Type of relationship ('caused_by', 'leads_to', 'related_to', 'depends_on')
   * @param {Object} metadata - Additional metadata
   * @returns {Promise<Object>} The created edge
   */
  async addEdge(fromNodeId, toNodeId, relationshipType, metadata = {}) {
    if (!this.loaded) await this.loadMemories();

    const edgeId = `edge_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    const edge = {
      id: edgeId,
      from: fromNodeId,
      to: toNodeId,
      type: relationshipType,
      metadata: metadata,
      createdAt: new Date().toISOString()
    };

    this.edges.set(edgeId, edge);
    
    // Update node connections
    const fromNode = this.nodes.get(fromNodeId);
    const toNode = this.nodes.get(toNodeId);
    
    if (fromNode && !fromNode.connections.includes(toNodeId)) {
      fromNode.connections.push(toNodeId);
    }
    if (toNode && !toNode.connections.includes(fromNodeId)) {
      toNode.connections.push(fromNodeId);
    }

    await this.saveMemories();
    return edge;
  }

  /**
   * Get a compressed view of the mind-map for LLM context
   * @param {number} maxNodes - Maximum nodes to include
   * @returns {Promise<Object>} Compressed mind-map view
   */
  async getCompressedMindMap(maxNodes = 20) {
    if (!this.loaded) await this.loadMemories();

    const nodes = Array.from(this.nodes.values())
      .sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime())
      .slice(0, maxNodes);

    const relevantEdges = Array.from(this.edges.values())
      .filter(edge => 
        nodes.some(n => n.id === edge.from) && 
        nodes.some(n => n.id === edge.to)
      );

    return {
      nodes: nodes.map(node => ({
        id: node.id,
        type: node.type,
        content: node.content ? node.content.substring(0, 200) + (node.content.length > 200 ? '...' : '') : 'No content',
        connections: node.connections.length,
        createdAt: node.createdAt
      })),
      edges: relevantEdges.map(edge => ({
        from: edge.from,
        to: edge.to,
        type: edge.type
      })),
      summary: {
        totalNodes: this.nodes.size,
        totalEdges: this.edges.size,
        nodeTypes: this.getNodeTypeDistribution()
      }
    };
  }

  /**
   * Search memories by content or tag (legacy support)
   * @param {string} searchTerm - Optional search term for content
   * @param {string} tag - Optional tag filter
   * @returns {Promise<Array<Object>>} Matching memories
   */
  async searchMemories(searchTerm = '', tag = '') {
    if (!this.loaded) await this.loadMemories();

    let results = Array.from(this.memories.values());

    // Filter by search term if provided
    if (searchTerm) {
      const searchLower = searchTerm.toLowerCase();
      results = results.filter((memory) => 
        memory.content.toLowerCase().includes(searchLower)
      );
    }

    // Filter by tag if provided
    if (tag) {
      results = results.filter((memory) => 
        memory.tags.includes(tag)
      );
    }

    // Sort by creation date (newest first)
    results.sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());

    return results;
  }

  /**
   * Get node type distribution for summary
   * @private
   */
  getNodeTypeDistribution() {
    const distribution = {};
    for (const node of this.nodes.values()) {
      distribution[node.type] = (distribution[node.type] || 0) + 1;
    }
    return distribution;
  }

  /**
   * Get all memories
   * @returns {Promise<Array<Object>>} All stored memories
   */
  async getAllMemories() {
    if (!this.loaded) await this.loadMemories();
    
    const memories = Array.from(this.memories.values());
    memories.sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());
    
    return memories;
  }

  /**
   * Get a specific memory by ID
   * @param {string} memoryId - The memory ID
   * @returns {Object|null} The memory or null if not found
   */
  async getMemory(memoryId) {
    if (!this.loaded) await this.loadMemories();
    return this.memories.get(memoryId) || null;
  }

  /**
   * Delete a memory
   * @param {string} memoryId - The memory ID to delete
   * @returns {boolean} True if deleted, false if not found
   */
  async deleteMemory(memoryId) {
    if (!this.loaded) await this.loadMemories();
    
    const deleted = this.memories.delete(memoryId);
    if (deleted) {
      await this.saveMemories();
    }
    
    return deleted;
  }

  /**
   * Get a summary of all tags with usage counts
   * @returns {Object} Tag summary with counts
   */
  async getTagsSummary() {
    if (!this.loaded) await this.loadMemories();

    const tagCounts = {};
    
    for (const memory of this.memories.values()) {
      for (const tag of memory.tags) {
        tagCounts[tag] = (tagCounts[tag] || 0) + 1;
      }
    }

    // Sort by count (descending) and return as array
    const sortedTags = Object.entries(tagCounts)
      .sort((a, b) => b[1] - a[1])
      .map(([tag, count]) => ({ tag, count }));

    return {
      totalTags: Object.keys(tagCounts).length,
      totalMemories: this.memories.size,
      tags: sortedTags
    };
  }

  /**
   * Clear all memories
   * @returns {Promise<void>}
   */
  async clearAllMemories() {
    this.memories.clear();
    await this.saveMemories();
  }

  /**
   * Format memories for display
   * @param {Array<Object>} memories - Memories to format
   * @returns {string} Formatted string representation
   */
  formatMemories(memories) {
    if (!memories.length) {
      return 'No memories found.';
    }

    return memories.map(memory => {
      const tags = memory.tags.length ? ` [${memory.tags.join(', ')}]` : '';
      const date = new Date(memory.createdAt).toLocaleString();
      return `ID: ${memory.id}${tags}\n${date}\n${memory.content}\n`;
    }).join('\n---\n\n');
  }

  /**
   * Save memories to Chrome storage
   */
  async saveMemories() {
    const memoriesData = {
      memories: Array.from(this.memories.entries()),
      lastUpdated: new Date().toISOString()
    };
    
    if (typeof chrome !== 'undefined' && chrome.storage) {
      await chrome.storage.local.set({ vibeAIMemories: memoriesData });
    }
  }

  /**
   * Load memories from Chrome storage
   */
  async loadMemories() {
    if (typeof chrome !== 'undefined' && chrome.storage) {
      const result = await chrome.storage.local.get('vibeAIMemories');
      if (result.vibeAIMemories && result.vibeAIMemories.memories) {
        this.memories = new Map(result.vibeAIMemories.memories);
      }
    }
    this.loaded = true;
  }

  /**
   * Get recent memories for context
   * @param {number} limit - Maximum number of memories to return
   * @returns {Array<Object>} Recent memories
   */
  async getRecentMemories(limit = 5) {
    if (!this.loaded) await this.loadMemories();
    
    const memories = Array.from(this.memories.values());
    memories.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
    
    return memories.slice(0, limit);
  }
}

// Export for use in extension
export default MemoryManager;