// Browser interaction tools with enhanced locator support

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { getValidTabId, waitForPageReady, normalizeLocator, formatErrorMessage } from "./utils.js";

// Common locator schema used across tools
const LocatorSchema = z.union([
  z.string().describe("CSS selector, text content, or role-based locator (e.g., 'button:Submit', 'text:Sign in', 'role:button[name=Submit]')"),
  z.object({
    role: z.string().nullable().optional().describe("ARIA role (e.g., 'button', 'textbox', 'combobox')"),
    name: z.string().nullable().optional().describe("Accessible name or aria-label"),
    text: z.string().nullable().optional().describe("Visible text content"),
    placeholder: z.string().nullable().optional().describe("Placeholder text for inputs"),
    title: z.string().nullable().optional().describe("Title attribute"),
    label: z.string().nullable().optional().describe("Label text associated with the element"),
    selector: z.string().nullable().optional().describe("CSS selector as fallback")
  }).describe("Locator object with various strategies")
]);

// Click tool with enhanced locator support
export class ClickTool extends BrowserTool {
  constructor() {
    super(
      "click",
      "Click an element using selector, text, role, or coordinates. Supports flexible locator strategies.",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        target: z.union([
          LocatorSchema,
          z.object({ 
            x: z.number().describe("X coordinate"),
            y: z.number().describe("Y coordinate") 
          }).describe("Pixel coordinates")
        ]).describe("Click target - either selector/locator or coordinates"),
        index: z.number().nullable().optional().describe("Index of element when multiple match (0-based)"),
        button: z.enum(["left", "right", "middle"]).default("left").nullable().optional().describe("Mouse button to click")
      })
    );
  }

  async call({ tabId, target, index = 0, button = "left" }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      const targetDesc = typeof target === 'string' ? target : 
                        (target.x !== undefined ? `(${target.x}, ${target.y})` : JSON.stringify(target));
      
      // Brief wait for page stability
      await waitForPageReady(validTabId);
      
      // Normalize the locator
      const normalizedTarget = target.x !== undefined ? target : normalizeLocator(target);
      
      // Send command with normalized locator
      const command = {
        type: 'CLICK',
        data: { 
          target: normalizedTarget, 
          index,
          button 
        }
      };
      
      const response = await chrome.tabs.sendMessage(validTabId, command);
      
      if (response && response.error) {
        throw new Error(response.error);
      }
      
      // Check for element count information in response
      if (response && response.elementCount > 1 && index === 0) {
        console.warn(`⚠️ Clicked first of ${response.elementCount} matching elements. Consider using index parameter.`);
      }
      
      // Brief pause after clicking
      await new Promise(resolve => setTimeout(resolve, 300));
      
      return response.message || `Clicked ${targetDesc}`;
    } catch (error) {
      console.error('ClickTool error:', error);
      throw new Error(formatErrorMessage(error, 'Click', target));
    }
  }
}

// Type tool with enhanced capabilities
export class TypeTool extends BrowserTool {
  constructor() {
    super(
      "type", 
      "Type text into the currently focused element or at a specific target.",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        text: z.string().describe("Text to type"),
        target: z.union([
          LocatorSchema,
          z.object({ 
            x: z.number().describe("X coordinate"),
            y: z.number().describe("Y coordinate") 
          }).describe("Pixel coordinates")
        ]).optional().describe("Optional target - if not provided, types into currently focused element"),
        clearFirst: z.boolean().default(false).nullable().optional().describe("Clear existing content before typing")
      })
    );
  }

  async call({ tabId, text, target, clearFirst = false }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      const targetDesc = target ? (typeof target === 'string' ? target : 
                        (target.x !== undefined ? `(${target.x}, ${target.y})` : JSON.stringify(target))) : 'focused element';
      
      // Brief wait for page stability
      await waitForPageReady(validTabId);
      
      // Normalize the locator if provided
      const normalizedTarget = target ? (target.x !== undefined ? target : normalizeLocator(target)) : null;
      
      // Send command
      const command = {
        type: 'TYPE',
        data: { text, target: normalizedTarget, clearFirst }
      };
      
      const response = await chrome.tabs.sendMessage(validTabId, command);
      
      if (response && response.error) {
        throw new Error(response.error);
      }
      
      // Brief pause after typing
      await new Promise(resolve => setTimeout(resolve, 500));
      
      return response.message || `Typed "${text}"${target ? ` into ${targetDesc}` : ''}`;
    } catch (error) {
      console.error('TypeTool error:', error);
      throw new Error(formatErrorMessage(error, 'Type', target || 'focused element'));
    }
  }
}

// Fill tool with enhanced locator support
export class FillTool extends BrowserTool {
  constructor() {
    super(
      "fill",
      "Fill a form field using flexible locator strategies. Automatically handles different input types.",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        target: LocatorSchema.describe("Target element to fill"),
        value: z.string().describe("Value to fill"),
        index: z.number().nullable().optional().describe("Index of element when multiple match (0-based)")
      })
    );
  }
  
  async call({ tabId, target, value, index = 0 }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      const targetDesc = typeof target === 'string' ? target : JSON.stringify(target);
      
      // Brief wait for page stability
      await waitForPageReady(validTabId);
      
      // Normalize the locator
      const normalizedTarget = normalizeLocator(target);
      
      const command = {
        type: 'FILL',
        data: { 
          target: normalizedTarget, 
          value,
          index 
        }
      };
      
      const response = await chrome.tabs.sendMessage(validTabId, command);
      
      if (response && response.error) {
        throw new Error(response.error);
      }
      
      // Check for element type information
      if (response && response.elementType) {
        console.log(`Filled ${response.elementType} field`);
      }
      
      // Brief pause after filling
      await new Promise(resolve => setTimeout(resolve, 500));
      
      return response.message || `Filled ${targetDesc} with "${value}"`;
    } catch (error) {
      console.error('FillTool error:', error);
      throw new Error(formatErrorMessage(error, 'Fill', target));
    }
  }
}

// Find elements tool with modern locator strategies
export class FindElementTool extends BrowserTool {
  constructor() {
    super(
      "find_element",
      "Find elements using role-based, label-based, or text-based locators. Better than CSS selectors for dynamic content.",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        locator: z.object({
          role: z.string().nullable().optional().describe("ARIA role (e.g., 'button', 'textbox', 'combobox')"),
          name: z.string().nullable().optional().describe("Accessible name or aria-label"),
          text: z.string().nullable().optional().describe("Visible text content"),
          placeholder: z.string().nullable().optional().describe("Placeholder text for inputs"),
          title: z.string().nullable().optional().describe("Title attribute"),
          label: z.string().nullable().optional().describe("Label text"),
          selector: z.string().nullable().optional().describe("CSS selector as fallback")
        }).describe("Locator options - at least one required"),
        options: z.object({
          exact: z.boolean().default(false).nullable().optional().describe("Require exact text match"),
          timeout: z.number().min(0).max(30000).default(5000).nullable().optional().describe("Wait timeout in ms")
        }).nullable().optional()
      })
    );
  }

  async call({ tabId, locator, options = {} }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      const locatorDesc = JSON.stringify(locator);
      
      // Brief wait for page stability
      await waitForPageReady(validTabId);
      
      const command = {
        type: 'FIND_ELEMENT',
        data: { locator, options }
      };
      
      const response = await chrome.tabs.sendMessage(validTabId, command);
      
      if (response && response.error) {
        throw new Error(response.error);
      }
      
      if (!response.elements || response.elements.length === 0) {
        return `No elements found matching locator: ${locatorDesc}`;
      }
      
      // Format response with element details
      const elementDetails = response.elements.map((el, idx) => 
        `  ${idx}: ${el.tagName}${el.id ? `#${el.id}` : ''}${el.className ? `.${el.className.split(' ')[0]}` : ''} - "${el.text || el.value || ''}"`.trim()
      ).join('\n');
      
      return `Found ${response.count} element(s) matching locator:\n${elementDetails}`;
    } catch (error) {
      console.error('FindElementTool error:', error);
      throw new Error(formatErrorMessage(error, 'Find element', locator));
    }
  }
}

// Wait for element tool
export class WaitForElementTool extends BrowserTool {
  constructor() {
    super(
      "wait_for_element",
      "Wait for an element to appear on the page using flexible locator strategies",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        target: LocatorSchema.describe("Element to wait for"),
        timeout: z.number().min(1000).max(30000).default(5000).nullable().optional()
          .describe("Maximum time to wait in milliseconds (1-30 seconds)")
      })
    );
  }
  
  async call({ tabId, target, timeout = 5000 }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      console.log(`⏳ Waiting for element to appear on tab ${validTabId} (timeout: ${timeout}ms)`);
      
      // Normalize the locator
      const normalizedTarget = normalizeLocator(target);
      const targetDesc = typeof target === 'string' ? target : JSON.stringify(target);
      
      // Send wait message to content script
      const response = await chrome.tabs.sendMessage(validTabId, {
        type: 'WAIT_FOR_ELEMENT',
        data: { target: normalizedTarget, timeout }
      });
      
      if (response && response.error) {
        throw new Error(response.error);
      }
      
      console.log(`Element appeared on page`);
      return response.message || `Element ${targetDesc} appeared on page within ${timeout}ms`;
    } catch (error) {
      console.error('WaitForElementTool error:', error);
      throw new Error(formatErrorMessage(error, 'Wait for element', target));
    }
  }
}

// Scroll tool
export class ScrollTool extends BrowserTool {
  constructor() {
    super(
      "scroll_page",
      "Scroll the page up or down by a number of pages",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        direction: z.enum(["up", "down"]).describe("Direction to scroll"),
        numPages: z.number().min(0).describe("Number of pages to scroll")
      })
    );
  }
  
  async call({ tabId, direction, numPages }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      const response = await chrome.tabs.sendMessage(validTabId, { 
        type: "scrollPage", 
        data: { direction, numPages } 
      });
      
      if (response && response.error) {
        throw new Error(response.error);
      }
      
      return `Scrolled page ${direction} by ${numPages} pages`;
    } catch (error) {
      throw new Error(`Scroll page failed: ${error.message}`);
    }
  }
}