/**
 * InjectJSTool - Execute JavaScript code in the context of the current page
 * Uses chrome.scripting.executeScript for safe code execution
 */

import { BrowserTool, CommonToolArgs } from "../../../ai_tools_interface.js";
import { z } from "zod";

export class InjectJSTool extends BrowserTool {
  constructor() {
    super(
      "inject_js",
      "Execute JavaScript code in the context of the current page. Use for computations, DOM introspection, or data extraction that goes beyond standard tools.",
      z.object({
        tabId: CommonToolArgs.tabId.optional().describe("Tab ID (optional - will use active tab if not specified)"),
        code: z.string().describe("JavaScript code to execute in the page context"),
        timeout: z.number().min(1000).max(30000).default(10000).optional()
          .describe("Execution timeout in milliseconds (1-30 seconds)")
      })
    );
  }

  async call({ tabId, code, timeout = 10000 }) {
    try {
      // Get valid tab ID
      const validTabId = tabId || await this.getCurrentActiveTab();
      
      // Security validation - prevent dangerous operations
      const dangerousPatterns = [
        /document\.cookie/i,
        /localStorage/i,
        /sessionStorage/i,
        /window\.location\s*=/i,
        /eval\s*\(/i,
        /Function\s*\(/i,
        /setTimeout\s*\(/i,
        /setInterval\s*\(/i,
        /XMLHttpRequest/i,
        /fetch\s*\(/i,
        /import\s*\(/i,
        /require\s*\(/i
      ];

      for (const pattern of dangerousPatterns) {
        if (pattern.test(code)) {
          throw new Error(`Security violation: Code contains potentially dangerous pattern: ${pattern.source}`);
        }
      }

      // Wrap code in a safe execution context with timeout
      const wrappedCode = `
        (function() {
          try {
            const startTime = Date.now();
            const timeoutMs = ${timeout};
            
            // Create a safe execution context
            const result = (function() {
              ${code}
            })();
            
            const executionTime = Date.now() - startTime;
            
            return {
              success: true,
              result: result,
              executionTime: executionTime,
              type: typeof result
            };
          } catch (error) {
            return {
              success: false,
              error: error.message,
              stack: error.stack
            };
          }
        })();
      `;

      console.log(`🔧 [INJECT_JS] Executing code in tab ${validTabId}:`, code ? code.substring(0, 100) + '...' : 'No code provided');

      // Execute the code using Chrome's scripting API
      const results = await chrome.scripting.executeScript({
        target: { tabId: validTabId },
        func: new Function('return ' + wrappedCode)
      });

      if (!results || results.length === 0) {
        throw new Error('No results returned from script execution');
      }

      const result = results[0].result;
      
      if (!result.success) {
        throw new Error(`JavaScript execution failed: ${result.error}`);
      }

      console.log(`[INJECT_JS] Code executed successfully in ${result.executionTime}ms`);
      
      return {
        success: true,
        result: result.result,
        executionTime: result.executionTime,
        type: result.type,
        message: `JavaScript executed successfully in ${result.executionTime}ms`
      };

    } catch (error) {
      console.error(`❌ [INJECT_JS] Execution failed:`, error);
      throw new Error(`JavaScript injection failed: ${error.message}`);
    }
  }

  async getCurrentActiveTab() {
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tabs || tabs.length === 0) {
      throw new Error('No active tab found');
    }
    return tabs[0].id;
  }
}

export default InjectJSTool;
