// Indexed Interaction Tools - Browser-use inspired approach
// These tools work with element indices instead of selectors

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { getValidTabId, ensureContentScriptCanCommunicate } from "./utils.js";

// Tool to interact with elements by index
class InteractByIndexTool extends BrowserTool {
  constructor() {
    super(
      "interact_by_index",
      "Interact with an element using its index number from get_indexed_elements. Supports click, fill, type, select, check/uncheck actions.",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        index: z.number().describe("The index number of the element to interact with"),
        action: z.enum(['click', 'fill', 'type', 'select', 'check', 'uncheck']).describe("The action to perform"),
        value: z.string().nullable().optional().describe("Value for fill/type/select actions"),
        openInNewTab: z.boolean().nullable().optional().describe("For click action on links: open in new tab instead of navigating current tab")
      })
    );
  }

  async call({ tabId, index, action, value, openInNewTab }) {
    const validTabId = await getValidTabId(tabId);

    // Check if we're on a restricted page where content scripts can't run
    const tab = await chrome.tabs.get(validTabId);
    if (tab?.url && (tab.url.startsWith('chrome://') || tab.url.startsWith('about:') ||
                     tab.url.startsWith('chrome-extension://') || tab.url.startsWith('edge://'))) {
      throw new Error('Cannot interact with browser internal pages');
    }

    // Validate action-value combinations
    if (['fill', 'type', 'select'].includes(action) && !value) {
      throw new Error(`Value is required for ${action} action`);
    }

    const command = {
      type: 'INTERACT_BY_INDEX',
      data: { index, action, value, openInNewTab: openInNewTab || false }
    };

    // Send message to content script
    const response = await chrome.tabs.sendMessage(validTabId, command);

    if (response && response.error) {
      throw new Error(response.error);
    }

    if (!response || !response.result) {
      throw new Error(`Failed to ${action} element at index ${index}`);
    }

    // Brief pause after interaction
    await new Promise(resolve => setTimeout(resolve, 500));

    return response.result;
  }
}

// Convenience tool for clicking by index
export class ClickByIndexTool extends BrowserTool {
  constructor() {
    super(
      "click_by_index",
      "Click an element using its index number. Elements are shown as [index:score] - use the index, not the score. When multiple similar elements exist (e.g., multiple 'Sign in' buttons), choose the one with the HIGHEST SCORE for best results. Use openInNewTab=true to open links in a new tab.",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        index: z.number().describe("The index number of the element to click (from [index:score] format)"),
        openInNewTab: z.boolean().nullable().optional().describe("If true and element is a link, open in new tab instead of navigating current tab")
      })
    );
  }

  async call({ tabId, index, openInNewTab }) {
    const interactTool = new InteractByIndexTool();
    return interactTool.call({ tabId, index, action: 'click', openInNewTab });
  }
}

// Convenience tool for filling by index
export class FillByIndexTool extends BrowserTool {
  constructor() {
    super(
      "fill_by_index",
      "Fill a form field using its index number. Elements are shown as [index:score] - use the index, not the score. Choose elements with higher scores when multiple options exist.",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        index: z.number().describe("The index number of the field to fill (from [index:score] format)"),
        value: z.string().describe("The value to fill into the field")
      })
    );
  }

  async call({ tabId, index, value }) {
    const interactTool = new InteractByIndexTool();
    return interactTool.call({ tabId, index, action: 'fill', value });
  }
}

// Convenience tool for selecting dropdown options by index
export class SelectByIndexTool extends BrowserTool {
  constructor() {
    super(
      "select_by_index",
      "Select an option from a dropdown/select element using its index number. Elements are shown as [index:score] - use the index, not the score. For selecting the option within the dropdown, provide the option text or value.",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        index: z.number().describe("The index number of the select/dropdown element (from [index:score] format)"),
        value: z.string().describe("The option text or value to select from the dropdown")
      })
    );
  }

  async call({ tabId, index, value }) {
    const interactTool = new InteractByIndexTool();
    return interactTool.call({ tabId, index, action: 'select', value });
  }
}

// Tool for controlling media (video/audio) elements by index
export class MediaControlByIndexTool extends BrowserTool {
  constructor() {
    super(
      "media_control_by_index",
      "Control a video or audio element using its index number. Supports play, pause, mute, unmute, and seek actions. Elements are shown as [index:score] - use the index for <video> or <audio> elements.",
      z.object({
        tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        index: z.number().describe("The index number of the video/audio element (from [index:score] format)"),
        action: z.enum(['play', 'pause', 'mute', 'unmute', 'seek', 'get_status']).describe("The action to perform on the media element"),
        seekTime: z.number().nullable().optional().describe("Time in seconds to seek to (required for 'seek' action)")
      })
    );
  }

  async call({ tabId, index, action, seekTime }) {
    const validTabId = await getValidTabId(tabId);

    // Check if we're on a restricted page
    const tab = await chrome.tabs.get(validTabId);
    if (tab?.url && (tab.url.startsWith('chrome://') || tab.url.startsWith('about:') ||
                     tab.url.startsWith('chrome-extension://') || tab.url.startsWith('edge://'))) {
      throw new Error('Cannot control media on browser internal pages');
    }

    // Validate seek action requires seekTime
    if (action === 'seek' && (seekTime === null || seekTime === undefined)) {
      throw new Error('seekTime is required for seek action');
    }

    const command = {
      type: 'MEDIA_CONTROL_BY_INDEX',
      data: { index, action, seekTime }
    };

    const response = await chrome.tabs.sendMessage(validTabId, command);

    if (response && response.error) {
      throw new Error(response.error);
    }

    if (!response || !response.result) {
      throw new Error(`Failed to ${action} media element at index ${index}`);
    }

    return response.result;
  }
}
