/**
 * Google Workspace Tools - Direct Gmail & Calendar API integration
 * 
 * Uses OAuth tokens stored in chrome.storage.local to call Google APIs directly.
 * No Node.js dependencies - works in browser extension environment.
 * 
 * Authentication flow:
 * 1. User clicks "Connect Google Workspace" in Settings
 * 2. OAuth popup via chrome.identity.launchWebAuthFlow
 * 3. Tokens stored in chrome.storage.local under GOOGLE_WORKSPACE_AUTH_KEY
 * 4. Tools read tokens via getGoogleAccessToken() function
 * 5. Automatic token refresh when expired
 */

import { z } from 'zod';
import { BrowserTool } from '../../../ai_tools_interface.js';

// Storage key for Google Workspace OAuth tokens
export const GOOGLE_WORKSPACE_AUTH_KEY = 'google_workspace_direct';

// Google OAuth configuration
const GOOGLE_CLIENT_ID = '349491536786-p5ljprl35rqhfpc0446cp9nsdti1gojh.apps.googleusercontent.com';
const GOOGLE_CLIENT_SECRET = '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';
const GOOGLE_TOKEN_URL = 'https://oauth2.googleapis.com/token';

// API base URLs
const GMAIL_API_BASE = 'https://gmail.googleapis.com/gmail/v1/users/me';
const CALENDAR_API_BASE = 'https://www.googleapis.com/calendar/v3';

/**
 * Get Google OAuth access token from storage
 * Handles automatic token refresh when expired
 */
async function getGoogleAccessToken() {
  const result = await chrome.storage.local.get('vibe_mcp_auth');
  const auth = result.vibe_mcp_auth?.[GOOGLE_WORKSPACE_AUTH_KEY];
  
  if (!auth?.tokens?.accessToken) {
    throw new Error('Google Workspace not connected. Please connect in Settings > Google Workspace.');
  }
  
  // Check if token is expired (with 5 minute buffer)
  const now = Math.floor(Date.now() / 1000);
  const expiresAt = auth.tokens.expiresAt || 0;
  const isExpired = expiresAt > 0 && (expiresAt - 300) < now;
  
  if (isExpired && auth.tokens.refreshToken) {
    console.log('[Google Workspace] Access token expired, refreshing...');
    try {
      const response = await fetch(GOOGLE_TOKEN_URL, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
          grant_type: 'refresh_token',
          refresh_token: auth.tokens.refreshToken,
          client_id: GOOGLE_CLIENT_ID,
          client_secret: GOOGLE_CLIENT_SECRET
        }).toString()
      });
      
      if (!response.ok) {
        throw new Error(`Token refresh failed: ${await response.text()}`);
      }
      
      const newTokens = await response.json();
      
      // Update storage with new tokens
      const allAuth = result.vibe_mcp_auth || {};
      allAuth[GOOGLE_WORKSPACE_AUTH_KEY] = {
        tokens: {
          accessToken: newTokens.access_token,
          refreshToken: auth.tokens.refreshToken,
          expiresAt: Math.floor(Date.now() / 1000) + newTokens.expires_in
        }
      };
      await chrome.storage.local.set({ vibe_mcp_auth: allAuth });
      
      console.log('[Google Workspace] Token refreshed successfully');
      return newTokens.access_token;
    } catch (refreshError) {
      console.error('[Google Workspace] Token refresh failed:', refreshError);
      throw new Error('Google access token expired and refresh failed. Please reconnect in Settings.');
    }
  }
  
  if (isExpired && !auth.tokens.refreshToken) {
    throw new Error('Google access token expired. Please reconnect in Settings > Google Workspace.');
  }
  
  return auth.tokens.accessToken;
}

/**
 * Make authenticated request to Google API
 */
async function googleApiRequest(url, options = {}) {
  const token = await getGoogleAccessToken();
  const response = await fetch(url, {
    ...options,
    headers: {
      'Authorization': `Bearer ${token}`,
      'Content-Type': 'application/json',
      ...options.headers
    }
  });
  
  if (!response.ok) {
    const error = await response.text();
    throw new Error(`Google API error (${response.status}): ${error}`);
  }
  
  return response.json();
}

/**
 * Check if Google Workspace is connected
 */
export async function isGoogleWorkspaceConnected() {
  try {
    const result = await chrome.storage.local.get('vibe_mcp_auth');
    const auth = result.vibe_mcp_auth?.[GOOGLE_WORKSPACE_AUTH_KEY];
    return !!auth?.tokens?.accessToken;
  } catch {
    return false;
  }
}

/**
 * Get Google Workspace connection status
 */
export async function getGoogleWorkspaceStatus() {
  try {
    const result = await chrome.storage.local.get('vibe_mcp_auth');
    const auth = result.vibe_mcp_auth?.[GOOGLE_WORKSPACE_AUTH_KEY];
    
    if (!auth?.tokens?.accessToken) {
      return { connected: false, expired: false };
    }
    
    const now = Math.floor(Date.now() / 1000);
    const expiresAt = auth.tokens.expiresAt || 0;
    const expired = expiresAt > 0 && expiresAt < now;
    const hasRefreshToken = !!auth.tokens.refreshToken;
    
    return { 
      connected: true, 
      expired,
      canRefresh: hasRefreshToken,
      expiresAt: expiresAt > 0 ? new Date(expiresAt * 1000).toISOString() : null
    };
  } catch {
    return { connected: false, expired: false };
  }
}

// ============================================================================
// Gmail Tools
// ============================================================================

/**
 * Gmail Search Tool - Search for emails
 */
class GmailSearchToolClass extends BrowserTool {
  constructor() {
    super(
      'gmail_search',
      'PREFERRED: Search for emails via Gmail API (no browser navigation needed). Use this instead of navigating to mail.google.com. Returns matching emails with subject, sender, date, and snippet. Supports Gmail search operators: "from:", "to:", "subject:", "is:unread", "has:attachment", "after:", "before:".',
      z.object({
        query: z.string().describe('Gmail search query. Examples: "from:user@example.com", "is:unread", "subject:meeting after:2024/01/01"'),
        maxResults: z.number().optional().nullable().default(10).describe('Maximum number of results (1-50)')
      })
    );
  }
  
  async call({ query, maxResults = 10 }) {
    try {
      // Search for messages
      const searchUrl = `${GMAIL_API_BASE}/messages?q=${encodeURIComponent(query)}&maxResults=${Math.min(maxResults, 50)}`;
      const searchResult = await googleApiRequest(searchUrl);
      
      if (!searchResult.messages || searchResult.messages.length === 0) {
        return 'No emails found matching your search query.';
      }
      
      // Get details for each message
      const messages = await Promise.all(
        searchResult.messages.slice(0, maxResults).map(async (msg) => {
          const msgUrl = `${GMAIL_API_BASE}/messages/${msg.id}?format=metadata&metadataHeaders=Subject&metadataHeaders=From&metadataHeaders=Date`;
          const msgData = await googleApiRequest(msgUrl);
          
          const headers = msgData.payload?.headers || [];
          const getHeader = (name) => headers.find(h => h.name.toLowerCase() === name.toLowerCase())?.value || '';
          
          return {
            id: msg.id,
            threadId: msg.threadId,
            subject: getHeader('Subject') || '(no subject)',
            from: getHeader('From'),
            date: getHeader('Date'),
            snippet: msgData.snippet
          };
        })
      );
      
      return JSON.stringify(messages, null, 2);
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      return `Error searching emails: ${errorMessage}`;
    }
  }
}

/**
 * Gmail Get Message Tool - Get full email content
 */
class GmailGetMessageToolClass extends BrowserTool {
  constructor() {
    super(
      'gmail_get_message',
      'PREFERRED: Get full email content via Gmail API (no browser navigation needed). Use gmail_search first to find message IDs.',
      z.object({
        messageId: z.string().describe('The Gmail message ID to retrieve')
      })
    );
  }
  
  async call({ messageId }) {
    try {
      const msgUrl = `${GMAIL_API_BASE}/messages/${messageId}?format=full`;
      const msgData = await googleApiRequest(msgUrl);
      
      const headers = msgData.payload?.headers || [];
      const getHeader = (name) => headers.find(h => h.name.toLowerCase() === name.toLowerCase())?.value || '';
      
      // Extract body content
      let body = '';
      const extractBody = (part) => {
        if (part.body?.data) {
          body += atob(part.body.data.replace(/-/g, '+').replace(/_/g, '/'));
        }
        if (part.parts) {
          part.parts.forEach(extractBody);
        }
      };
      extractBody(msgData.payload);
      
      return JSON.stringify({
        id: msgData.id,
        threadId: msgData.threadId,
        subject: getHeader('Subject'),
        from: getHeader('From'),
        to: getHeader('To'),
        date: getHeader('Date'),
        body: body.substring(0, 10000) // Limit body size
      }, null, 2);
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      return `Error getting email: ${errorMessage}`;
    }
  }
}

/**
 * Gmail Get Thread Tool - Get all messages in a thread
 */
class GmailGetThreadToolClass extends BrowserTool {
  constructor() {
    super(
      'gmail_get_thread',
      'PREFERRED: Get all messages in an email thread via Gmail API (no browser navigation needed).',
      z.object({
        threadId: z.string().describe('The Gmail thread ID to retrieve')
      })
    );
  }
  
  async call({ threadId }) {
    try {
      const threadUrl = `${GMAIL_API_BASE}/threads/${threadId}?format=metadata&metadataHeaders=Subject&metadataHeaders=From&metadataHeaders=Date`;
      const threadData = await googleApiRequest(threadUrl);
      
      const messages = threadData.messages?.map(msg => {
        const headers = msg.payload?.headers || [];
        const getHeader = (name) => headers.find(h => h.name.toLowerCase() === name.toLowerCase())?.value || '';
        
        return {
          id: msg.id,
          subject: getHeader('Subject'),
          from: getHeader('From'),
          date: getHeader('Date'),
          snippet: msg.snippet
        };
      }) || [];
      
      return JSON.stringify({ threadId, messageCount: messages.length, messages }, null, 2);
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      return `Error getting thread: ${errorMessage}`;
    }
  }
}

/**
 * Gmail Create Draft Tool - Create email draft
 */
class GmailCreateDraftToolClass extends BrowserTool {
  constructor() {
    super(
      'gmail_create_draft',
      'PREFERRED: Create a new email draft via Gmail API (no browser navigation needed). The draft can be reviewed and sent later by the user.',
      z.object({
        to: z.array(z.string()).describe('List of recipient email addresses'),
        subject: z.string().describe('Email subject line'),
        body: z.string().describe('Email body content'),
        cc: z.array(z.string()).optional().nullable().describe('CC recipients'),
        bcc: z.array(z.string()).optional().nullable().describe('BCC recipients')
      })
    );
  }
  
  async call({ to, subject, body, cc, bcc }) {
    try {
      // Build RFC 2822 message
      const headers = [
        `To: ${to.join(', ')}`,
        `Subject: ${subject}`,
        'Content-Type: text/plain; charset=utf-8'
      ];
      if (cc?.length) headers.push(`Cc: ${cc.join(', ')}`);
      if (bcc?.length) headers.push(`Bcc: ${bcc.join(', ')}`);
      
      const message = `${headers.join('\r\n')}\r\n\r\n${body}`;
      const encodedMessage = btoa(unescape(encodeURIComponent(message)))
        .replace(/\+/g, '-')
        .replace(/\//g, '_')
        .replace(/=+$/, '');
      
      const result = await googleApiRequest(`${GMAIL_API_BASE}/drafts`, {
        method: 'POST',
        body: JSON.stringify({ message: { raw: encodedMessage } })
      });
      
      return `Draft created successfully. Draft ID: ${result.id}`;
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      return `Error creating draft: ${errorMessage}`;
    }
  }
}

/**
 * Gmail Send Message Tool - Send email immediately
 */
class GmailSendMessageToolClass extends BrowserTool {
  constructor() {
    super(
      'gmail_send_message',
      'PREFERRED: Send an email immediately via Gmail API (no browser navigation needed). Use with caution - emails cannot be unsent.',
      z.object({
        to: z.array(z.string()).describe('List of recipient email addresses'),
        subject: z.string().describe('Email subject line'),
        body: z.string().describe('Email body content'),
        cc: z.array(z.string()).optional().nullable().describe('CC recipients'),
        bcc: z.array(z.string()).optional().nullable().describe('BCC recipients')
      })
    );
  }
  
  async call({ to, subject, body, cc, bcc }) {
    try {
      const headers = [
        `To: ${to.join(', ')}`,
        `Subject: ${subject}`,
        'Content-Type: text/plain; charset=utf-8'
      ];
      if (cc?.length) headers.push(`Cc: ${cc.join(', ')}`);
      if (bcc?.length) headers.push(`Bcc: ${bcc.join(', ')}`);
      
      const message = `${headers.join('\r\n')}\r\n\r\n${body}`;
      const encodedMessage = btoa(unescape(encodeURIComponent(message)))
        .replace(/\+/g, '-')
        .replace(/\//g, '_')
        .replace(/=+$/, '');
      
      const result = await googleApiRequest(`${GMAIL_API_BASE}/messages/send`, {
        method: 'POST',
        body: JSON.stringify({ raw: encodedMessage })
      });
      
      return `Email sent successfully. Message ID: ${result.id}`;
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      return `Error sending email: ${errorMessage}`;
    }
  }
}

// ============================================================================
// Google Calendar Tools
// ============================================================================

/**
 * Calendar View Tool - View upcoming events
 */
class CalendarViewToolClass extends BrowserTool {
  constructor() {
    super(
      'calendar_view',
      'PREFERRED: View upcoming Google Calendar events via direct API (no browser navigation needed). Returns event titles, times, locations, and attendees. Use this instead of navigating to calendar.google.com.',
      z.object({
        query: z.string().optional().nullable().describe('Optional search query to filter events'),
        maxResults: z.number().optional().nullable().default(10).describe('Maximum number of events (1-50)'),
        timeMin: z.string().optional().nullable().describe('Start time (ISO 8601). Defaults to now.'),
        timeMax: z.string().optional().nullable().describe('End time (ISO 8601). Defaults to 7 days from now.')
      })
    );
  }
  
  async call({ query, maxResults = 10, timeMin, timeMax }) {
    try {
      const now = new Date();
      const weekFromNow = new Date(now.getTime() + 7 * 24 * 60 * 60 * 1000);
      
      const params = new URLSearchParams({
        maxResults: Math.min(maxResults, 50).toString(),
        orderBy: 'startTime',
        singleEvents: 'true',
        timeMin: timeMin || now.toISOString(),
        timeMax: timeMax || weekFromNow.toISOString()
      });
      if (query) params.append('q', query);
      
      const url = `${CALENDAR_API_BASE}/calendars/primary/events?${params}`;
      const result = await googleApiRequest(url);
      
      if (!result.items || result.items.length === 0) {
        return 'No upcoming events found.';
      }
      
      const events = result.items.map(event => ({
        id: event.id,
        summary: event.summary || '(no title)',
        start: event.start?.dateTime || event.start?.date,
        end: event.end?.dateTime || event.end?.date,
        location: event.location,
        description: event.description?.substring(0, 200),
        attendees: event.attendees?.map(a => a.email)
      }));
      
      return JSON.stringify(events, null, 2);
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      return `Error getting calendar events: ${errorMessage}`;
    }
  }
}

/**
 * Calendar Create Tool - Create new event
 */
class CalendarCreateToolClass extends BrowserTool {
  constructor() {
    super(
      'calendar_create',
      'PREFERRED: Create a new Google Calendar event via direct API (no browser navigation needed). Use this instead of navigating to calendar.google.com.',
      z.object({
        summary: z.string().describe('Event title/name'),
        start: z.string().describe('Start time (ISO 8601 format, e.g., "2024-01-15T10:00:00-08:00")'),
        end: z.string().describe('End time (ISO 8601 format)'),
        description: z.string().optional().nullable().describe('Event description'),
        location: z.string().optional().nullable().describe('Event location'),
        attendees: z.array(z.string()).optional().nullable().describe('Attendee email addresses')
      })
    );
  }
  
  async call({ summary, start, end, description, location, attendees }) {
    try {
      const event = {
        summary,
        start: { dateTime: start },
        end: { dateTime: end }
      };
      
      if (description) event.description = description;
      if (location) event.location = location;
      if (attendees?.length) {
        event.attendees = attendees.map(email => ({ email }));
      }
      
      const result = await googleApiRequest(`${CALENDAR_API_BASE}/calendars/primary/events`, {
        method: 'POST',
        body: JSON.stringify(event)
      });
      
      return `Event created successfully.\nTitle: ${result.summary}\nStart: ${result.start?.dateTime || result.start?.date}\nLink: ${result.htmlLink}`;
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      return `Error creating event: ${errorMessage}`;
    }
  }
}

/**
 * Calendar Delete Tool - Delete an event
 */
class CalendarDeleteToolClass extends BrowserTool {
  constructor() {
    super(
      'calendar_delete',
      'PREFERRED: Delete a Google Calendar event via direct API (no browser navigation needed). Use calendar_view first to find the event ID.',
      z.object({
        eventId: z.string().describe('The Google Calendar event ID to delete')
      })
    );
  }
  
  async call({ eventId }) {
    try {
      const url = `${CALENDAR_API_BASE}/calendars/primary/events/${eventId}`;
      const token = await getGoogleAccessToken();
      
      const response = await fetch(url, {
        method: 'DELETE',
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });
      
      if (!response.ok) {
        const error = await response.text();
        throw new Error(`Google API error (${response.status}): ${error}`);
      }
      
      return `Event deleted successfully. Event ID: ${eventId}`;
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      return `Error deleting event: ${errorMessage}`;
    }
  }
}

// ============================================================================
// Export tool instances
// ============================================================================

export const GmailSearchTool = new GmailSearchToolClass();
export const GmailGetMessageTool = new GmailGetMessageToolClass();
export const GmailGetThreadTool = new GmailGetThreadToolClass();
export const GmailCreateDraftTool = new GmailCreateDraftToolClass();
export const GmailSendMessageTool = new GmailSendMessageToolClass();
export const CalendarViewTool = new CalendarViewToolClass();
export const CalendarCreateTool = new CalendarCreateToolClass();
export const CalendarDeleteTool = new CalendarDeleteToolClass();

/**
 * All Google Workspace tools as an array
 */
export const googleWorkspaceTools = [
  GmailSearchTool,
  GmailGetMessageTool,
  GmailGetThreadTool,
  GmailCreateDraftTool,
  GmailSendMessageTool,
  CalendarViewTool,
  CalendarCreateTool,
  CalendarDeleteTool,
];

/**
 * Get Google Workspace tools
 */
export function getGoogleWorkspaceTools() {
  return googleWorkspaceTools;
}
