// DOM and content extraction tools

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { getCurrentActiveTab } from "./utils.js";

// Tool to get DOM structure
export class GetDomStructureTool extends BrowserTool {
  constructor() {
    super(
      "get_dom_structure",
      "Get detailed DOM structure. Note: This is expensive with high token usage - the page content is already provided in the agent's context.",
      z.object({
        tabId: z.number().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        maxDepth: z.number().min(1).max(10).default(5).optional()
          .describe("Maximum depth of the DOM tree to retrieve (1-10)")
      })
    );
  }

  async call(args) {
    const { tabId, maxDepth = 5 } = args;
    
    try {
      const targetTab = tabId ? { id: tabId } : await getCurrentActiveTab();
      
      // Execute script in the context of the page
      const [result] = await chrome.scripting.executeScript({
        target: { tabId: targetTab.id },
        func: (depth) => {
          try {
            // Build DOM tree function
            function buildDomTree(element, maxDepth, currentDepth = 0) {
              if (currentDepth >= maxDepth || !element) {
                return null;
              }
              
              const node = {
                tagName: element.tagName?.toLowerCase() || 'text',
                id: element.id || null,
                className: element.className || null,
                textContent: element.textContent?.trim().substring(0, 100) || null,
                attributes: {},
                children: []
              };
              
              // Capture important attributes
              if (element.attributes) {
                for (const attr of element.attributes) {
                  if (['href', 'src', 'type', 'name', 'value', 'placeholder', 'alt', 'title', 'role', 'aria-label'].includes(attr.name)) {
                    node.attributes[attr.name] = attr.value;
                  }
                }
              }
              
              // Process children
              for (const child of element.children || []) {
                const childNode = buildDomTree(child, maxDepth, currentDepth + 1);
                if (childNode) {
                  node.children.push(childNode);
                }
              }
              
              return node;
            }
            
            // Find interactive elements function
            function findInteractiveElements() {
              const interactiveSelectors = [
                'a[href]', 'button', 'input', 'select', 'textarea',
                '[onclick]', '[role="button"]', '[tabindex]'
              ];
              
              const elements = [];
              for (const selector of interactiveSelectors) {
                const found = document.querySelectorAll(selector);
                for (const el of found) {
                  elements.push({
                    tagName: el.tagName.toLowerCase(),
                    id: el.id || null,
                    className: el.className || null,
                    textContent: el.textContent?.trim().substring(0, 50) || null,
                    href: el.href || null,
                    type: el.type || null,
                    name: el.name || null,
                    value: el.value || null,
                    role: el.getAttribute('role') || null,
                    ariaLabel: el.getAttribute('aria-label') || null
                  });
                }
              }
              return elements;
            }
            
            // Extract forms function
            function extractForms() {
              const forms = [];
              for (const form of document.querySelectorAll('form')) {
                const formData = {
                  id: form.id || null,
                  action: form.action || null,
                  method: form.method || 'get',
                  fields: []
                };
                
                for (const field of form.querySelectorAll('input, select, textarea')) {
                  formData.fields.push({
                    tagName: field.tagName.toLowerCase(),
                    type: field.type || null,
                    name: field.name || null,
                    id: field.id || null,
                    placeholder: field.placeholder || null,
                    required: field.required || false,
                    value: field.value || null,
                    ariaLabel: field.getAttribute('aria-label') || null
                  });
                }
                
                forms.push(formData);
              }
              return forms;
            }
            
            // Execute the functions
            const dom = buildDomTree(document.documentElement, depth);
            const interactiveElements = findInteractiveElements();
            const forms = extractForms();
            
            return {
              success: true,
              dom,
              interactiveElements,
              forms
            };
          } catch (error) {
            return {
              success: false,
              error: error instanceof Error ? error.message : String(error)
            };
          }
        },
        args: [maxDepth]
      });

      if (!result || !result.result) {
        throw new Error('No result returned from script execution');
      }

      const parsedResult = result.result;
      
      if (!parsedResult.success) {
        throw new Error(parsedResult.error || 'Failed to retrieve DOM structure');
      }

      // Return a string as required by the BrowserTool interface
      return JSON.stringify({
        dom: parsedResult.dom,
        interactiveElements: parsedResult.interactiveElements,
        forms: parsedResult.forms
      });
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      console.error('Error in GetDomStructureTool:', errorMessage);
      throw new Error(`Failed to get DOM structure: ${errorMessage}`);
    }
  }
}