// @ts-nocheck

// Browser automation tools - re-exports from modular organization
// This file maintains backward compatibility while using the new modular structure

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

// Import all browser tools from the modular structure
import {
  // Navigation
  NavigateToURLTool,
  
  // Tab management
  ListTabsTool,
  CreateNewTabTool,
  SwitchToTabTool,
  
  // Interaction
  ClickTool,
  TypeTool,
  FillTool,
  FindElementTool,
  WaitTool,
  ScrollTool,
  
  ClickByIndexTool,
  FillByIndexTool,
  
   // Content extraction
   // GetDomStructureTool removed - now internal agent capability
  
  // Advanced interaction
  HighlightElementsTool,
  KeyboardShortcutTool,
  HoverElementTool,
  SearchAndScrollToTextTool,
  DragDropTool,
  VerifyActionTool,
  
  // Miscellaneous
  TakeScreenshotTool,
  
  // Browser tools array
  browserTools
} from './BrowserToolsIndex.js';

// Import AI planning and reasoning tools
import MemoryManager from "./MemoryManager.js";
import { InjectJSTool } from "./InjectJSTool.js";
import { WebSearchTool } from "./WebSearchTool.js";
import { GetSessionHistoryTool } from "./SessionHistoryTool.js";
import StructuredLogger from "./StructuredLogger.js";
import ToolCallNormalizer from "./ToolCallNormalizer.js";

// Re-export all tools for backward compatibility
export {
  // Browser tools
  NavigateToURLTool,
  ListTabsTool,
  CreateNewTabTool,
  SwitchToTabTool,
  ClickTool,
  TypeTool,
  FillTool,
  FindElementTool,
  WaitTool,
  ScrollTool,
  ClickByIndexTool,
  FillByIndexTool,
  HighlightElementsTool,
  KeyboardShortcutTool,
  HoverElementTool,
  SearchAndScrollToTextTool,
  DragDropTool,
  VerifyActionTool,
  TakeScreenshotTool
};

// Legacy tool classes for backward compatibility
// These map to the new modular tools but maintain the old class names

export class NavigateToURL extends NavigateToURLTool {}
export class ListTabs extends ListTabsTool {}
export class CreateNewTab extends CreateNewTabTool {}
export class SwitchToTab extends SwitchToTabTool {}
export class Click extends ClickTool {}
export class Type extends TypeTool {}
export class Fill extends FillTool {}
export class FindElement extends FindElementTool {}
// export class WaitForElement extends WaitForElementTool {} // Disabled - causing issues
export class Wait extends WaitTool {}
export class Scroll extends ScrollTool {}
// GetPageContent and GetDomStructure removed - now internal agent capabilities
// Disabled for now: export class HighlightElements extends HighlightElementsTool {}
export class KeyboardShortcut extends KeyboardShortcutTool {}
export class HoverElement extends HoverElementTool {}
export class SearchAndScrollToText extends SearchAndScrollToTextTool {}
export class DragDrop extends DragDropTool {}
export class VerifyAction extends VerifyActionTool {}
export class TakeScreenshot extends TakeScreenshotTool {}

// Indexed interaction tools
export class ClickByIndex extends ClickByIndexTool {}
export class FillByIndex extends FillByIndexTool {}

// Legacy duplicate tools that are now consolidated
export class ClickElementTool extends ClickTool {
  constructor() {
    super();
    // Override name to maintain backward compatibility
    this.name = "click_element";
    this.description = "Click an interactive element on the current page by its selector";
    this.schema = z.object({
      tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
      selector: z.string().describe("CSS selector of the element to click"),
    });
  }
  
  async call({ tabId, selector }) {
    // Map to new tool's parameters
    return super.call({ tabId, target: selector });
  }
}

export class FillFormFieldTool extends FillTool {
  constructor() {
    super();
    // Override name to maintain backward compatibility
    this.name = "fill_form_field";
    this.description = "Fill a form field on the current page. Some drop down menus could appear after filling and you may need to click on available options.";
    this.schema = z.object({
      tabId: z.number().nullable().optional().describe("Tab ID (optional - will use active tab if not specified)"),
      selector: z.string().describe("CSS selector of the form field"),
      value: z.string().describe("Value to fill in the field"),
    });
  }
  
  async call({ tabId, selector, value }) {
    // Map to new tool's parameters
    return super.call({ tabId, target: selector, value });
  }
}

// Legacy exports
export class ClickElement extends ClickElementTool {}
export class FillFormField extends FillFormFieldTool {}

// AI Planning and Reasoning Tools

// Initialize singleton instances
let memoryManager = null;


// Tool to store memories for context
export class StoreMemoryTool extends BrowserTool {
  constructor() {
    super(
      "store_memory",
      "Store important information or context for later retrieval",
      z.object({
        content: z.string().describe("The content to remember"),
        tags: z.array(z.string()).nullable().optional().describe("Optional tags for categorization")
      })
    );
  }
  
  async call({ content, tags }) {
    try {
      if (!memoryManager) {
        memoryManager = new MemoryManager();
      }
      
      const memory = await memoryManager.storeMemory(content, tags);
      return `Stored memory (ID: ${memory.id}) with ${memory.tags.length} tags`;
    } catch (error) {
      throw new Error(`Failed to store memory: ${error.message}`);
    }
  }
}

// Tool to search memories
export class SearchMemoriesTool extends BrowserTool {
  constructor() {
    super(
      "search_memories",
      "Search stored memories by content or tag",
      z.object({
        searchTerm: z.string().nullable().optional().describe("Text to search for in memory content"),
        tag: z.string().nullable().optional().describe("Tag to filter by")
      })
    );
  }
  
  async call({ searchTerm, tag }) {
    try {
      if (!memoryManager) {
        memoryManager = new MemoryManager();
      }
      
      const memories = await memoryManager.searchMemories(searchTerm, tag);
      if (memories.length === 0) {
        return "No memories found matching your search.";
      }
      
      return memoryManager.formatMemories(memories.slice(0, 5));
    } catch (error) {
      throw new Error(`Failed to search memories: ${error.message}`);
    }
  }
}

// Tool to get memory summary
export class GetMemorySummaryTool extends BrowserTool {
  constructor() {
    super(
      "get_memory_summary",
      "Get a summary of all stored memories and tags",
      z.object({})
    );
  }
  
  async call() {
    try {
      if (!memoryManager) {
        memoryManager = new MemoryManager();
      }
      
      const summary = await memoryManager.getTagsSummary();
      const recentMemories = await memoryManager.getRecentMemories(3);
      
      return `Memory Summary:
Total Memories: ${summary.totalMemories}
Total Tags: ${summary.totalTags}
Top Tags: ${summary.tags.slice(0, 5).map(t => `${t.tag} (${t.count})`).join(', ')}

Recent Memories:
${memoryManager.formatMemories(recentMemories)}`;
    } catch (error) {
      throw new Error(`Failed to get memory summary: ${error.message}`);
    }
  }
}

/**
 * Wrapper that adds reasoning parameter to any BrowserTool
 * This allows the LLM to explain its thinking for each tool call
 */
export class ReasoningBrowserTool extends BrowserTool {
  constructor(originalTool) {
    const reasoningSchema = originalTool.schema.extend({
      reasoning: z.string().nullable().optional().describe("Explain your reasoning for using this tool and what you expect to achieve")
    });

    super(
      originalTool.name,
      originalTool.description,
      reasoningSchema
    );

    this.originalTool = originalTool;
    // Expose the original tool as _browserTool for ReactGraph to access inherit()
    // This fixes the issue where inherit() wasn't being called because ReactGraph
    // was looking for ._browserTool property which didn't exist on the wrapper
    // IMPORTANT: Cast to any to avoid type errors if needed, though this is JS
    this._browserTool = originalTool;
    
    // Also attach inherit directly to this wrapper, just in case ReactGraph calls it directly
    if (originalTool.inherit) {
        this.inherit = originalTool.inherit.bind(originalTool);
    }
  }

  async call(args) {
    const { reasoning, ...originalArgs } = args;
    return await this.originalTool.call(originalArgs);
  }

  /**
   * Forward setParentLLM to the original tool if it has this method
   * Used by SubAgentTool to receive parent's LLM config
   */
  setParentLLM(llm, modelName) {
    if (this.originalTool?.setParentLLM && typeof this.originalTool.setParentLLM === 'function') {
      this.originalTool.setParentLLM(llm, modelName);
    }
  }

  /**
   * Forward inherit to the original tool if it has this method
   * Used by SubAgentTool to receive parent ReactAgent config
   */
  inherit(config) {
    if (this.originalTool?.inherit && typeof this.originalTool.inherit === 'function') {
      this.originalTool.inherit(config);
    }
  }
}

/**
 * Helper function to wrap all tools with reasoning
 */
export function addReasoningToTools(tools) {
  return tools.map(tool => new ReasoningBrowserTool(tool));
}

// Combine browser tools with AI tools
const aiTools = [
  // Disabled for now: new ReportTool(),
  new StoreMemoryTool(),
  new SearchMemoriesTool(),
  new GetMemorySummaryTool(),
  new GetSessionHistoryTool(),
  // new InjectJSTool(),
  // Disabled for now: new WebSearchTool()
];

// Export combined tools array
export { browserTools };
export const allTools = [...browserTools, ...aiTools];

// Export the tools wrapped with reasoning capability
export const browserToolsWithReasoning = addReasoningToTools(allTools);