// Advanced interaction tools for complex browser automation

import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";
import { getValidTabId, normalizeLocator, formatErrorMessage } from "./utils.js";

// Tool to highlight interactive elements with visual feedback
export class HighlightElementsTool extends BrowserTool {
  constructor() {
    super(
      "highlight_elements",
      "Highlight all interactive elements on the page with numbered overlays. Essential for showing agent what it can interact with visually.",
      z.object({
        tabId: z.number().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        duration: z.number().min(1000).max(30000).default(10000).optional()
          .describe("How long to show highlights in milliseconds (1-30 seconds)"),
        elementTypes: z.array(z.string()).default(["button", "input", "select", "a", "[role='option']"]).optional()
          .describe("Types of elements to highlight"),
        showNumbers: z.boolean().default(true).optional()
          .describe("Show numbered overlays on each element")
      })
    );
  }

  async call({ tabId, duration = 10000, elementTypes = ["button", "input", "select", "a", "[role='option']"], showNumbers = true }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      console.log(`🎯 Highlighting interactive elements on tab ${validTabId} for ${duration}ms`);
      
      const response = await chrome.tabs.sendMessage(validTabId, {
        type: 'HIGHLIGHT_ELEMENTS',
        data: { duration, elementTypes, showNumbers }
      });
      
      if (response && response.error) {
        throw new Error(response.error);
      }
      
      console.log(`Highlighted ${response.count || 0} interactive elements`);
      return `Highlighted ${response.count || 0} interactive elements with numbered overlays for ${duration}ms`;
    } catch (error) {
      console.error('HighlightElementsTool error:', error);
      const errorMessage = error instanceof Error ? error.message : String(error);
      throw new Error(`Element highlighting failed: ${errorMessage}`);
    }
  }
}

// Tool to send keyboard shortcuts and key presses
export class KeypressTool extends BrowserTool {
  constructor() {
    super(
      "keypress",
      "Send keyboard key presses (Enter, Escape, Arrow keys, Tab, etc.) or key combinations (Ctrl+C, Ctrl+V). Use after filling a form to submit with Enter, or to navigate dropdowns with Arrow keys.",
      z.object({
        tabId: z.number().optional().nullable().describe("Tab ID (optional - will use active tab if not specified)"),
        keys: z.string().describe("Key or key combination (e.g., 'Enter', 'Escape', 'ArrowDown', 'Ctrl+C', 'Tab')"),
        index: z.number().optional().nullable().describe("Element index to send keys to (optional - uses focused element if not specified)")
      })
    );
  }

  async call({ tabId, keys, index }) {
    try {
      const validTabId = await getValidTabId(tabId);

      console.log(`⌨️ Sending keypress "${keys}" on tab ${validTabId}${index !== undefined ? ` to element [${index}]` : ''}`);

      const response = await chrome.tabs.sendMessage(validTabId, {
        type: 'KEYPRESS',
        data: { keys, index }
      });

      if (response && response.error) {
        throw new Error(response.error);
      }

      console.log(`Successfully sent keypress: ${keys}`);
      return `Sent keypress "${keys}"${index !== undefined ? ` to element [${index}]` : ' to focused element'}`;
    } catch (error) {
      console.error('KeypressTool error:', error);
      const errorMessage = error instanceof Error ? error.message : String(error);
      throw new Error(`Keypress failed: ${errorMessage}`);
    }
  }
}

// Legacy alias for backward compatibility
export const KeyboardShortcutTool = KeypressTool;

// Tool to hover over elements to trigger tooltips and dropdown menus
export class HoverElementTool extends BrowserTool {
  constructor() {
    super(
      "hover_element",
      "Hover over an element using its index to trigger tooltips, dropdown menus, or hover states. Elements are shown as [index:score] - use the index number.",
      z.object({
        tabId: z.number().optional().nullable().describe("Tab ID (optional - will use active tab if not specified)"),
        index: z.number().describe("Element index to hover over (from [index:score] format)"),
        duration: z.number().min(100).max(5000).default(1000).optional().nullable()
          .describe("How long to maintain hover in milliseconds (100-5000ms)")
      })
    );
  }

  async call({ tabId, index, duration = 1000 }) {
    try {
      const validTabId = await getValidTabId(tabId);

      console.log(`🖱️ Hovering over element [${index}] on tab ${validTabId} for ${duration}ms`);

      const response = await chrome.tabs.sendMessage(validTabId, {
        type: 'HOVER_BY_INDEX',
        data: { index, duration }
      });

      if (response && response.error) {
        throw new Error(response.error);
      }

      console.log(`Successfully hovered over element [${index}]`);
      return `Hovered over element [${index}] for ${duration}ms, triggering any hover effects`;
    } catch (error) {
      console.error('HoverElementTool error:', error);
      const errorMessage = error instanceof Error ? error.message : String(error);
      throw new Error(`Hover failed: ${errorMessage}`);
    }
  }
}

// Tool to search for text on the page and scroll to it
export class SearchAndScrollToTextTool extends BrowserTool {
  constructor() {
    super(
      "search_scroll_to_text",
      "Find text on the page and scroll to it. Supports fuzzy matching to find similar text even with slight differences.",
      z.object({
        tabId: z.number().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        searchText: z.string().describe("Text to search for on the page"),
        fuzzyMatch: z.boolean().default(true).optional()
          .describe("Allow fuzzy matching for similar text"),
        highlightDuration: z.number().min(0).max(10000).default(3000).optional()
          .describe("How long to highlight found text (0 to disable highlighting)")
      })
    );
  }

  async call({ tabId, searchText, fuzzyMatch = true, highlightDuration = 3000 }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      console.log(`🔍 Searching for "${searchText}" on tab ${validTabId} (fuzzy: ${fuzzyMatch})`);
      
      const response = await chrome.tabs.sendMessage(validTabId, {
        type: 'SEARCH_SCROLL_TO_TEXT',
        data: { searchText, fuzzyMatch, highlightDuration }
      });
      
      if (response && response.error) {
        throw new Error(response.error);
      }
      
      console.log(`Found and scrolled to text: "${searchText}"`);
      return `Found "${searchText}" on page and scrolled to it${response.matchedText && response.matchedText !== searchText ? ` (matched: "${response.matchedText}")` : ''}`;
    } catch (error) {
      console.error('SearchAndScrollToTextTool error:', error);
      const errorMessage = error instanceof Error ? error.message : String(error);
      throw new Error(`Text search failed: ${errorMessage}`);
    }
  }
}

// Tool for drag and drop interactions
export class DragDropTool extends BrowserTool {
  constructor() {
    super(
      "drag_drop",
      "Perform drag and drop operations between elements or coordinates. Useful for file uploads, sortable lists, canvas drawing, and sliders.",
      z.object({
        tabId: z.number().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        source: z.union([
          z.string(),
          z.object({ x: z.number(), y: z.number() })
        ]).describe("Source element selector or coordinates {x, y}"),
        target: z.union([
          z.string(),
          z.object({ x: z.number(), y: z.number() })
        ]).describe("Target element selector or coordinates {x, y}"),
        duration: z.number().min(100).max(3000).default(500).optional()
          .describe("Duration of drag operation in milliseconds")
      })
    );
  }

  async call({ tabId, source, target, duration = 500 }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      const sourceDesc = typeof source === 'string' ? source : `(${source.x}, ${source.y})`;
      const targetDesc = typeof target === 'string' ? target : `(${target.x}, ${target.y})`;
      
      console.log(`🎯 Dragging from ${sourceDesc} to ${targetDesc} on tab ${validTabId}`);
      
      const response = await chrome.tabs.sendMessage(validTabId, {
        type: 'DRAG_DROP',
        data: { source, target, duration }
      });
      
      if (response && response.error) {
        throw new Error(response.error);
      }
      
      console.log(`Successfully completed drag and drop operation`);
      return `Dragged from ${sourceDesc} to ${targetDesc} in ${duration}ms`;
    } catch (error) {
      console.error('DragDropTool error:', error);
      const errorMessage = error instanceof Error ? error.message : String(error);
      throw new Error(`Drag and drop failed: ${errorMessage}`);
    }
  }
}

// Tool to verify that an action had the expected effect on the page
export class VerifyActionTool extends BrowserTool {
  constructor() {
    super(
      "verify_action",
      "Verify that a previous action (click, form fill, etc.) had the expected effect by checking for specific changes on the page.",
      z.object({
        tabId: z.number().optional().describe("Tab ID (optional - will use active tab if not specified)"),
        actionType: z.enum(["click", "fill", "submit", "navigation", "dropdown", "general"])
          .describe("Type of action to verify"),
        expectedChanges: z.object({
          newElements: z.array(z.string()).optional()
            .describe("CSS selectors of elements that should appear"),
          removedElements: z.array(z.string()).optional()
            .describe("CSS selectors of elements that should disappear"),
          changedText: z.array(z.object({
            selector: z.string(),
            expectedText: z.string().optional(),
            shouldContain: z.string().optional()
          })).optional().describe("Text changes to verify"),
          urlChange: z.string().optional()
            .describe("Expected URL change or pattern"),
          formValues: z.array(z.object({
            selector: z.string(),
            expectedValue: z.string()
          })).optional().describe("Form field values to verify")
        }).describe("Expected changes after the action"),
        waitTime: z.number().min(100).max(10000).default(2000).optional()
          .describe("Time to wait for changes to appear (ms)")
      })
    );
  }

  async call({ tabId, actionType, expectedChanges, waitTime = 2000 }) {
    try {
      const validTabId = await getValidTabId(tabId);
      
      console.log(`🔍 Verifying ${actionType} action effects on tab ${validTabId}`);
      
      // Wait for changes to take effect
      await new Promise(resolve => setTimeout(resolve, waitTime));
      
      const response = await chrome.tabs.sendMessage(validTabId, {
        type: 'VERIFY_ACTION',
        data: { actionType, expectedChanges, waitTime }
      });
      
      if (response && response.error) {
        throw new Error(response.error);
      }
      
      const { verified, details } = response;
      
      if (verified) {
        console.log(`Action verification successful: ${details.summary}`);
        return `Action verified: ${details.summary}. ${details.changes} changes detected.`;
      } else {
        console.log(`❌ Action verification failed: ${details.summary}`);
        return `❌ Action failed to take effect: ${details.summary}. Consider retry with different approach.`;
      }
      
    } catch (error) {
      console.error('VerifyActionTool error:', error);
      const errorMessage = error instanceof Error ? error.message : String(error);
      throw new Error(`Action verification failed: ${errorMessage}`);
    }
  }
}