// Content script handlers for smart element finding

export const smartFindHandlers = {
  async SMART_FIND(data) {
    const { strategy } = data;
    
    switch (strategy) {
      case 'role':
        return this.findByRole(data);
      case 'label':
        return this.findByLabel(data);
      case 'text':
        return this.findByText(data);
      case 'placeholder':
        return this.findByPlaceholder(data);
      case 'proximity':
        return this.findByProximity(data);
      case 'selector':
        return this.findBySelector(data);
      case 'intelligent':
        return this.findWithIntelligence(data);
      default:
        throw new Error(`Unknown strategy: ${strategy}`);
    }
  },
  
  findByRole(data) {
    const { role, text, name, exact } = data;
    let elements = Array.from(document.querySelectorAll(`[role="${role}"]`));
    
    // Filter by text if provided
    if (text) {
      elements = elements.filter(el => {
        const elText = el.textContent.trim();
        return exact ? elText === text : elText.includes(text);
      });
    }
    
    // Filter by name/aria-label if provided
    if (name) {
      elements = elements.filter(el => {
        const ariaLabel = el.getAttribute('aria-label') || '';
        const nameAttr = el.getAttribute('name') || '';
        return exact ? 
          (ariaLabel === name || nameAttr === name) :
          (ariaLabel.includes(name) || nameAttr.includes(name));
      });
    }
    
    return this.formatElements(elements);
  },
  
  findByLabel(data) {
    const { label, exact } = data;
    const elements = [];
    
    // Find by aria-label
    const ariaElements = Array.from(document.querySelectorAll('[aria-label]')).filter(el => {
      const ariaLabel = el.getAttribute('aria-label');
      return exact ? ariaLabel === label : ariaLabel.includes(label);
    });
    elements.push(...ariaElements);
    
    // Find by associated label
    const labels = Array.from(document.querySelectorAll('label')).filter(l => {
      const text = l.textContent.trim();
      return exact ? text === label : text.includes(label);
    });
    
    labels.forEach(l => {
      const forId = l.getAttribute('for');
      if (forId) {
        const input = document.getElementById(forId);
        if (input && !elements.includes(input)) {
          elements.push(input);
        }
      }
      // Also check nested inputs
      const nestedInput = l.querySelector('input, textarea, select');
      if (nestedInput && !elements.includes(nestedInput)) {
        elements.push(nestedInput);
      }
    });
    
    return this.formatElements(elements);
  },
  
  findByText(data) {
    const { text, exact } = data;
    
    // Use XPath for better text matching
    const xpath = exact ? 
      `//*[normalize-space(text())="${text}"]` :
      `//*[contains(normalize-space(text()), "${text}")]`;
    
    const result = document.evaluate(
      xpath,
      document,
      null,
      XPathResult.ORDERED_NODE_SNAPSHOT_TYPE,
      null
    );
    
    const elements = [];
    for (let i = 0; i < result.snapshotLength; i++) {
      elements.push(result.snapshotItem(i));
    }
    
    return this.formatElements(elements);
  },
  
  findByPlaceholder(data) {
    const { placeholder, exact } = data;
    
    const elements = Array.from(document.querySelectorAll('[placeholder]')).filter(el => {
      const ph = el.getAttribute('placeholder');
      return exact ? ph === placeholder : ph.includes(placeholder);
    });
    
    return this.formatElements(elements);
  },
  
  findByProximity(data) {
    const { near, after, before, under, targetRole, targetText } = data;
    const candidates = [];
    
    // Find reference element
    let refElement;
    if (near) {
      refElement = this.findTextElement(near);
    } else if (after) {
      refElement = this.findTextElement(after);
    } else if (before) {
      refElement = this.findTextElement(before);
    } else if (under) {
      refElement = this.findHeading(under);
    }
    
    if (!refElement) {
      return { elements: [], count: 0 };
    }
    
    // Find nearby interactive elements
    const interactiveSelectors = [
      'input', 'textarea', 'select', 'button', 
      '[role="button"]', '[role="textbox"]', '[role="combobox"]',
      'a[href]', '[contenteditable="true"]'
    ];
    
    const rect = refElement.getBoundingClientRect();
    const allInteractive = document.querySelectorAll(interactiveSelectors.join(','));
    
    Array.from(allInteractive).forEach(el => {
      const elRect = el.getBoundingClientRect();
      const distance = this.calculateDistance(rect, elRect);
      
      // Check position constraints
      let include = true;
      if (after && elRect.left < rect.right) include = false;
      if (before && elRect.left > rect.left) include = false;
      if (under && elRect.top < rect.bottom) include = false;
      
      if (include && distance < 300) { // Within 300px
        candidates.push({ element: el, distance });
      }
    });
    
    // Sort by distance and filter by role/text if specified
    candidates.sort((a, b) => a.distance - b.distance);
    let elements = candidates.map(c => c.element);
    
    if (targetRole) {
      elements = elements.filter(el => el.getAttribute('role') === targetRole);
    }
    
    if (targetText) {
      elements = elements.filter(el => el.textContent.includes(targetText));
    }
    
    return this.formatElements(elements);
  },
  
  findBySelector(data) {
    const { selector } = data;
    
    try {
      const elements = document.querySelectorAll(selector);
      return this.formatElements(Array.from(elements));
    } catch (error) {
      return { error: `Invalid selector: ${selector}` };
    }
  },
  
  findWithIntelligence(data) {
    const { hints } = data;
    const candidates = [];
    
    // Smart detection based on hints
    if (hints.text && (hints.text.toLowerCase().includes('from') || 
                       hints.text.toLowerCase().includes('departure'))) {
      // Look for departure/origin fields
      const selectors = [
        'input[name*="origin"]',
        'input[name*="from"]',
        'input[name*="departure"]',
        '[aria-label*="from"]',
        '[aria-label*="departure"]',
        '[placeholder*="from"]',
        '[placeholder*="departure"]'
      ];
      
      selectors.forEach(sel => {
        try {
          const els = document.querySelectorAll(sel);
          candidates.push(...Array.from(els));
        } catch (e) {}
      });
    }
    
    // Remove duplicates
    const unique = [...new Set(candidates)];
    return this.formatElements(unique);
  },
  
  // Helper methods
  findTextElement(text) {
    const xpath = `//*[contains(normalize-space(text()), "${text}")]`;
    const result = document.evaluate(xpath, document, null, XPathResult.FIRST_ORDERED_NODE_TYPE, null);
    return result.singleNodeValue;
  },
  
  findHeading(text) {
    const headings = document.querySelectorAll('h1, h2, h3, h4, h5, h6');
    return Array.from(headings).find(h => h.textContent.includes(text));
  },
  
  calculateDistance(rect1, rect2) {
    const centerX1 = rect1.left + rect1.width / 2;
    const centerY1 = rect1.top + rect1.height / 2;
    const centerX2 = rect2.left + rect2.width / 2;
    const centerY2 = rect2.top + rect2.height / 2;
    
    return Math.sqrt(
      Math.pow(centerX2 - centerX1, 2) + 
      Math.pow(centerY2 - centerY1, 2)
    );
  },
  
  formatElements(elements) {
    const formatted = elements.slice(0, 10).map(el => ({
      tagName: el.tagName,
      id: el.id,
      className: el.className,
      role: el.getAttribute('role'),
      text: el.textContent?.trim().substring(0, 100),
      value: el.value,
      placeholder: el.getAttribute('placeholder'),
      ariaLabel: el.getAttribute('aria-label'),
      name: el.getAttribute('name'),
      type: el.type,
      visible: this.isVisible(el),
      rect: el.getBoundingClientRect()
    }));
    
    return {
      elements: formatted,
      count: elements.length
    };
  },
  
  isVisible(el) {
    const rect = el.getBoundingClientRect();
    const style = window.getComputedStyle(el);
    
    return rect.width > 0 && 
           rect.height > 0 && 
           style.display !== 'none' && 
           style.visibility !== 'hidden' && 
           style.opacity !== '0';
  },
  
  async GET_DIAGNOSTIC(data) {
    const { locator } = data;
    
    // Gather diagnostic info
    const allInputs = document.querySelectorAll('input, textarea, select');
    const allButtons = document.querySelectorAll('button, [role="button"], input[type="submit"], input[type="button"]');
    const allLinks = document.querySelectorAll('a[href]');
    
    // Find elements with similar text
    const similarTextElements = [];
    if (locator.text) {
      const allText = document.querySelectorAll('*');
      Array.from(allText).forEach(el => {
        const text = el.textContent?.trim();
        if (text && text.length < 100 && text.toLowerCase().includes(locator.text.toLowerCase())) {
          similarTextElements.push(text);
        }
      });
    }
    
    // Find possible matches with scoring
    const possibleMatches = [];
    const interactiveElements = document.querySelectorAll('input, textarea, select, button, [role="button"], a[href]');
    
    Array.from(interactiveElements).forEach(el => {
      let score = 0;
      const description = [];
      
      // Score based on matching attributes
      if (locator.role && el.getAttribute('role') === locator.role) score += 3;
      if (locator.text && el.textContent?.includes(locator.text)) score += 2;
      if (locator.placeholder && el.getAttribute('placeholder')?.includes(locator.placeholder)) score += 3;
      if (locator.label) {
        const ariaLabel = el.getAttribute('aria-label');
        if (ariaLabel && ariaLabel.includes(locator.label)) score += 3;
      }
      
      if (score > 0) {
        const text = el.textContent?.trim().substring(0, 50) || el.value || '';
        description.push(`${el.tagName.toLowerCase()}`);
        if (el.id) description.push(`#${el.id}`);
        if (el.getAttribute('role')) description.push(`[role="${el.getAttribute('role')}"]`);
        if (text) description.push(`"${text}"`);
        
        possibleMatches.push({
          score,
          description: description.join(' ')
        });
      }
    });
    
    // Sort by score
    possibleMatches.sort((a, b) => b.score - a.score);
    
    return {
      url: window.location.href,
      interactiveCount: interactiveElements.length,
      formFieldCount: allInputs.length,
      buttonCount: allButtons.length,
      linkCount: allLinks.length,
      similarTextElements: similarTextElements.slice(0, 5),
      possibleMatches: possibleMatches.slice(0, 5),
      pageTitle: document.title,
      hasIframes: document.querySelectorAll('iframe').length > 0,
      hasShadowDOM: Array.from(document.querySelectorAll('*')).some(el => el.shadowRoot)
    };
  }
};