import { BrowserTool } from "../../../ai_tools_interface.js";
import { z } from "zod";

// Simple wait tool shared across environments
export class WaitTool extends BrowserTool {
  constructor(options = {}) {
    super(
      "wait",
      "Wait for specified seconds before continuing",
      z.object({
        seconds: z
          .number()
          .min(0)
          .max(60)
          .default(1)
          .nullable()
          .optional()
          .describe("Number of seconds to wait"),
        reasoning: z.string().nullable().optional().describe("Reasoning for this action")
      })
    );
    this.options = options;
  }

  async call({ seconds = 1 }) {
    try {
      if (this.options.verbose) {
        console.log(`⏳ Waiting for ${seconds} seconds`);
      }
      await new Promise(resolve => setTimeout(resolve, seconds * 1000));
      if (this.options.verbose) {
        console.log(`Waited for ${seconds} seconds`);
      }
      return `Waited for ${seconds} seconds`;
    } catch (error) {
      const msg = `Wait failed: ${error.message}`;
      console.error(`❌ ${msg}`);
      throw new Error(msg);
    }
  }
}
