/**
 * Vibe API CLI Authentication Module
 * Handles login/signup flow and token storage for CLI usage
 * Token stored at: $HOME/.cache/vibe/token
 */

import fs from 'fs';
import path from 'path';
import os from 'os';
import http from 'http';
import open from 'open';
import crypto from 'crypto';

// VIBE_API_URL can be set for dev environment (e.g., https://api-dev.vibebrowser.app)
const VIBE_AUTH_BASE = process.env.VIBE_API_URL || 'https://api.vibebrowser.app';
const VIBE_CACHE_DIR = path.join(os.homedir(), '.cache', 'vibe');
const VIBE_TOKEN_FILE = path.join(VIBE_CACHE_DIR, 'token');
const VIBE_USER_FILE = path.join(VIBE_CACHE_DIR, 'user.json');

/**
 * Ensure cache directory exists
 */
function ensureCacheDir() {
  if (!fs.existsSync(VIBE_CACHE_DIR)) {
    fs.mkdirSync(VIBE_CACHE_DIR, { recursive: true, mode: 0o700 });
  }
}

/**
 * Get stored Vibe API token
 * @returns {string|null} Token or null if not logged in
 */
export function getToken() {
  try {
    if (fs.existsSync(VIBE_TOKEN_FILE)) {
      return fs.readFileSync(VIBE_TOKEN_FILE, 'utf-8').trim();
    }
  } catch (error) {
    console.error('Failed to read token:', error.message);
  }
  return null;
}

/**
 * Store Vibe API token
 * @param {string} token - API token
 */
export function storeToken(token) {
  ensureCacheDir();
  fs.writeFileSync(VIBE_TOKEN_FILE, token, { mode: 0o600 });
}

/**
 * Clear stored token (logout)
 */
export function clearToken() {
  try {
    if (fs.existsSync(VIBE_TOKEN_FILE)) {
      fs.unlinkSync(VIBE_TOKEN_FILE);
    }
    if (fs.existsSync(VIBE_USER_FILE)) {
      fs.unlinkSync(VIBE_USER_FILE);
    }
  } catch (error) {
    console.error('Failed to clear token:', error.message);
  }
}

/**
 * Get stored user info
 * @returns {Object|null} User object or null
 */
export function getStoredUser() {
  try {
    if (fs.existsSync(VIBE_USER_FILE)) {
      return JSON.parse(fs.readFileSync(VIBE_USER_FILE, 'utf-8'));
    }
  } catch (error) {
    // Ignore
  }
  return null;
}

/**
 * Store user info
 * @param {Object} user - User object
 */
function storeUser(user) {
  ensureCacheDir();
  fs.writeFileSync(VIBE_USER_FILE, JSON.stringify(user, null, 2), { mode: 0o600 });
}

/**
 * Check if logged in
 * @returns {boolean}
 */
export function isLoggedIn() {
  return !!getToken();
}

/**
 * Generate PKCE code verifier
 */
function generateCodeVerifier() {
  return crypto.randomBytes(32).toString('base64url');
}

/**
 * Generate PKCE code challenge
 */
function generateCodeChallenge(verifier) {
  return crypto.createHash('sha256').update(verifier).digest('base64url');
}

/**
 * Login via browser - opens browser and waits for callback
 * @returns {Promise<string>} Access token
 */
export async function login() {
  const state = crypto.randomUUID();
  const verifier = generateCodeVerifier();
  const challenge = generateCodeChallenge(verifier);

  // Start local server to receive callback
  return new Promise((resolve, reject) => {
    const server = http.createServer(async (req, res) => {
      const url = new URL(req.url, `http://localhost`);

      if (url.pathname === '/callback') {
        const code = url.searchParams.get('code');
        const returnedState = url.searchParams.get('state');
        const error = url.searchParams.get('error');

        if (error) {
          res.writeHead(200, { 'Content-Type': 'text/html' });
          res.end(`
            <html>
              <body style="font-family: system-ui; padding: 40px; text-align: center;">
                <h1>Login Failed</h1>
                <p>${error}</p>
                <p>You can close this window.</p>
              </body>
            </html>
          `);
          server.close();
          reject(new Error(error));
          return;
        }

        if (returnedState !== state) {
          res.writeHead(400, { 'Content-Type': 'text/html' });
          res.end(`
            <html>
              <body style="font-family: system-ui; padding: 40px; text-align: center;">
                <h1>Security Error</h1>
                <p>State mismatch - possible CSRF attack.</p>
              </body>
            </html>
          `);
          server.close();
          reject(new Error('State mismatch'));
          return;
        }

        try {
          // Exchange code for token
          const tokenResponse = await fetch(`${VIBE_AUTH_BASE}/auth/token`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
              grant_type: 'authorization_code',
              code,
              code_verifier: verifier,
              client_id: 'vibe-cli'
            })
          });

          if (!tokenResponse.ok) {
            throw new Error(`Token exchange failed: ${await tokenResponse.text()}`);
          }

          const data = await tokenResponse.json();
          const token = data.access_token || data.key;

          // Store token
          storeToken(token);

          // Get user info
          try {
            const user = await getCurrentUser(token);
            if (user) storeUser(user);
          } catch (e) {
            // Ignore user fetch errors
          }

          res.writeHead(200, { 'Content-Type': 'text/html' });
          res.end(`
            <html>
              <body style="font-family: system-ui; padding: 40px; text-align: center;">
                <h1>Login Successful!</h1>
                <p>You can close this window and return to your terminal.</p>
              </body>
            </html>
          `);

          server.close();
          resolve(token);
        } catch (err) {
          res.writeHead(500, { 'Content-Type': 'text/html' });
          res.end(`
            <html>
              <body style="font-family: system-ui; padding: 40px; text-align: center;">
                <h1>Error</h1>
                <p>${err.message}</p>
              </body>
            </html>
          `);
          server.close();
          reject(err);
        }
      } else {
        res.writeHead(404);
        res.end('Not found');
      }
    });

    // Find available port
    server.listen(0, '127.0.0.1', async () => {
      const port = server.address().port;
      const redirectUri = `http://127.0.0.1:${port}/callback`;

      // Build auth URL
      const params = new URLSearchParams({
        client_id: 'vibe-cli',
        redirect_uri: redirectUri,
        response_type: 'code',
        scope: 'api',
        state,
        code_challenge: challenge,
        code_challenge_method: 'S256'
      });

      const authUrl = `${VIBE_AUTH_BASE}/auth/login?${params.toString()}`;

      console.log('Opening browser for login...');
      console.log(`If browser doesn't open, visit: ${authUrl}`);

      try {
        await open(authUrl);
      } catch (err) {
        console.log(`Please open this URL in your browser:\n${authUrl}`);
      }
    });

    // Timeout after 5 minutes
    setTimeout(() => {
      server.close();
      reject(new Error('Login timed out'));
    }, 5 * 60 * 1000);
  });
}

/**
 * Get current user info from API
 * @param {string} [token] - Optional token (uses stored if not provided)
 * @returns {Promise<Object|null>}
 */
export async function getCurrentUser(token) {
  const apiToken = token || getToken();
  if (!apiToken) return null;

  try {
    const response = await fetch(`${VIBE_AUTH_BASE}/auth/me`, {
      headers: {
        'Authorization': `Bearer ${apiToken}`
      }
    });

    if (!response.ok) {
      if (response.status === 401) {
        // Token expired
        clearToken();
        return null;
      }
      return null;
    }

    return await response.json();
  } catch (error) {
    return null;
  }
}

/**
 * Logout - clear local token and optionally revoke on server
 */
export async function logout() {
  const token = getToken();

  if (token) {
    try {
      await fetch(`${VIBE_AUTH_BASE}/auth/logout`, {
        method: 'POST',
        headers: { 'Authorization': `Bearer ${token}` }
      });
    } catch (error) {
      // Best effort
    }
  }

  clearToken();
}

/**
 * Print login status
 */
export async function status() {
  const token = getToken();

  if (!token) {
    console.log('Not logged in to Vibe API');
    console.log('Run: vibe login');
    return { loggedIn: false };
  }

  const user = await getCurrentUser(token);

  if (!user) {
    console.log('Token expired or invalid');
    console.log('Run: vibe login');
    clearToken();
    return { loggedIn: false };
  }

  console.log(`Logged in as: ${user.email}`);
  if (user.plan) console.log(`Plan: ${user.plan}`);
  if (user.usage) {
    console.log(`Usage: ${user.usage.tokens_used.toLocaleString()} / ${user.usage.tokens_limit.toLocaleString()} tokens`);
  }

  return { loggedIn: true, user };
}

export default {
  getToken,
  storeToken,
  clearToken,
  isLoggedIn,
  login,
  logout,
  status,
  getCurrentUser,
  getStoredUser
};
