// @ts-nocheck

/**
 * DOM Tree Builder for Vibe AI Browser Co-Pilot
 * Builds a simplified DOM representation for AI agent analysis
 */

function buildDomTree(element = document.body, maxDepth = 5, currentDepth = 0) {
  if (currentDepth >= maxDepth || !element) {
    return null;
  }

  const node = {
    tagName: element.tagName?.toLowerCase(),
    id: element.id || null,
    className: element.className || null,
    textContent: getCleanTextContent(element),
    attributes: getRelevantAttributes(element),
    children: [],
    selector: generateSelector(element),
    bounds: getBounds(element),
    visible: isVisible(element),
    interactive: isInteractive(element)
  };

  // Only include relevant child elements
  const children = Array.from(element.children).filter(child =>
    isRelevantElement(child)
  );

  for (const child of children.slice(0, 20)) { // Limit children
    const childNode = buildDomTree(child, maxDepth, currentDepth + 1);
    if (childNode) {
      node.children.push(childNode);
    }
  }

  return node;
}

/**
 * Get clean text content (first 100 chars)
 */
function getCleanTextContent(element) {
  if (!element.textContent) return null;

  const text = element.textContent.trim();
  if (text.length === 0) return null;

  // Get only direct text content, not from children
  let directText = "";
  for (const node of element.childNodes) {
    if (node.nodeType === Node.TEXT_NODE) {
      directText += node.textContent;
    }
  }

  directText = directText.trim();
  return directText.length > 0 ? directText.substring(0, 100) : null;
}

/**
 * Get relevant attributes for AI analysis
 */
function getRelevantAttributes(element) {
  const relevantAttrs = [
    "href", "src", "alt", "title", "placeholder", "value", "type",
    "name", "role", "aria-label", "data-testid", "data-automation-id"
  ];

  const attrs = {};
  for (const attr of relevantAttrs) {
    const value = element.getAttribute(attr);
    if (value) {
      attrs[attr] = value;
    }
  }

  return Object.keys(attrs).length > 0 ? attrs : null;
}

/**
 * Generate a unique CSS selector for the element
 */
function generateSelector(element) {
  if (!element || element === document.body) {
    return "body";
  }

  let selector = element.tagName.toLowerCase();

  if (element.id) {
    return `#${element.id}`;
  }

  if (element.className) {
    const classes = element.className.split(" ").filter(c => c.length > 0);
    if (classes.length > 0) {
      selector += "." + classes.slice(0, 3).join(".");
    }
  }

  // Add nth-child if needed for uniqueness
  const parent = element.parentElement;
  if (parent) {
    const siblings = Array.from(parent.children).filter(
      child => child.tagName === element.tagName
    );

    if (siblings.length > 1) {
      const index = siblings.indexOf(element) + 1;
      selector += `:nth-child(${index})`;
    }
  }

  return selector;
}

/**
 * Get element bounds
 */
function getBounds(element) {
  try {
    const rect = element.getBoundingClientRect();
    return {
      x: Math.round(rect.x),
      y: Math.round(rect.y),
      width: Math.round(rect.width),
      height: Math.round(rect.height)
    };
  } catch (error) {
    return null;
  }
}

/**
 * Check if element is visible
 */
function isVisible(element) {
  if (!element) return false;

  const style = window.getComputedStyle(element);
  const rect = element.getBoundingClientRect();

  return style.display !== "none" &&
    style.visibility !== "hidden" &&
    style.opacity !== "0" &&
    rect.width > 0 &&
    rect.height > 0;
}

/**
 * Check if element is interactive
 */
function isInteractive(element) {
  const interactiveTags = ["a", "button", "input", "select", "textarea"];
  const interactiveRoles = ["button", "link", "tab", "option"];

  return interactiveTags.includes(element.tagName.toLowerCase()) ||
    interactiveRoles.includes(element.getAttribute("role")) ||
    element.hasAttribute("onclick") ||
    element.style.cursor === "pointer";
}

/**
 * Check if element is relevant for AI analysis
 */
function isRelevantElement(element) {
  const irrelevantTags = ["script", "style", "meta", "link", "title"];
  const tagName = element.tagName.toLowerCase();

  if (irrelevantTags.includes(tagName)) {
    return false;
  }

  // Skip hidden elements
  if (!isVisible(element)) {
    return false;
  }

  // Skip very small elements
  const rect = element.getBoundingClientRect();
  if (rect.width < 10 || rect.height < 10) {
    return false;
  }

  return true;
}

/**
 * Find interactive elements on the page
 */
function findInteractiveElements() {
  const interactiveElements = [];
  const elements = document.querySelectorAll(
    "a, button, input, select, textarea, [role=\"button\"], [onclick], [tabindex]"
  );

  for (const element of elements) {
    if (isVisible(element) && isRelevant(element)) {
      interactiveElements.push({
        selector: generateSelector(element),
        tagName: element.tagName.toLowerCase(),
        text: getCleanTextContent(element),
        attributes: getRelevantAttributes(element),
        bounds: getBounds(element),
        type: getInteractionType(element)
      });
    }
  }

  return interactiveElements.slice(0, 50); // Limit results
}

/**
 * Get interaction type
 */
function getInteractionType(element) {
  const tagName = element.tagName.toLowerCase();

  if (tagName === "a") return "link";
  if (tagName === "button") return "button";
  if (tagName === "input") return element.type || "input";
  if (tagName === "select") return "select";
  if (tagName === "textarea") return "textarea";
  if (element.getAttribute("role") === "button") return "button";
  if (element.hasAttribute("onclick")) return "clickable";

  return "interactive";
}

/**
 * Check if element is relevant for interaction
 */
function isRelevant(element) {
  // Skip very small elements
  const rect = element.getBoundingClientRect();
  if (rect.width < 5 || rect.height < 5) {
    return false;
  }

  // Skip elements with no meaningful content
  const text = getCleanTextContent(element);
  const attrs = getRelevantAttributes(element);

  return text || attrs;
}

/**
 * Extract forms data
 */
function extractForms() {
  const forms = [];

  document.querySelectorAll("form").forEach(form => {
    if (!isVisible(form)) return;

    const formData = {
      selector: generateSelector(form),
      action: form.action || null,
      method: form.method || "get",
      fields: []
    };

    form.querySelectorAll("input, select, textarea").forEach(field => {
      if (isVisible(field)) {
        formData.fields.push({
          selector: generateSelector(field),
          name: field.name || null,
          type: field.type || field.tagName.toLowerCase(),
          placeholder: field.placeholder || null,
          value: field.value || null,
          required: field.required || false
        });
      }
    });

    if (formData.fields.length > 0) {
      forms.push(formData);
    }
  });

  return forms;
}
