// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// Bridge between browser settings and component extension settings
export class SettingsBridge {
  private static instance: SettingsBridge | null = null;
  
  static getInstance(): SettingsBridge {
    if (!SettingsBridge.instance) {
      SettingsBridge.instance = new SettingsBridge();
    }
    return SettingsBridge.instance;
  }

  // Sync settings from browser preferences to extension storage
  async syncFromBrowserSettings(): Promise<void> {
    try {
      // Get settings from browser via Chrome settings API
      const response = await this.callBrowserSettingsAPI('getVibeAIState');
      
      if (response && response.provider) {
        // Store in extension storage
        await chrome.storage.local.set({
          'vibe_ai_provider': response.provider,
          'vibe_ai_model': response.model,
          'vibe_ai_temperature': response.temperature,
          'vibe_ai_max_tokens': response.maxTokens,
          'vibe_ai_agent_timeout': response.agentTimeout,
          'vibe_ai_env_config': response.environmentConfiguration
        });
        
        console.log('Synced settings from browser to extension:', response);
      }
    } catch (error) {
      console.error('Failed to sync settings from browser:', error);
    }
  }

  // Sync settings from extension storage to browser preferences
  async syncToBrowserSettings(settings: any): Promise<void> {
    try {
      if (settings.provider) {
        await this.callBrowserSettingsAPI('setVibeAIProvider', [settings.provider]);
      }
      
      console.log('Synced settings from extension to browser:', settings);
    } catch (error) {
      console.error('Failed to sync settings to browser:', error);
    }
  }

  // Get combined settings (browser + extension + environment)
  async getCombinedSettings(): Promise<any> {
    try {
      const [browserSettings, extensionSettings] = await Promise.all([
        this.getBrowserSettings(),
        this.getExtensionSettings()
      ]);

      // Merge settings with priority: environment > browser > extension > defaults
      const combined = {
        provider: 'auto',
        model: 'auto',
        temperature: 0.7,
        maxTokens: 1000,
        agentTimeout: 30000,
        ...extensionSettings,
        ...browserSettings
      };

      // Apply environment variable overrides
      if (browserSettings.environmentConfiguration) {
        if (browserSettings.environmentConfiguration.openaiConfigured) {
          combined.provider = 'openai';
          combined.model = combined.model === 'auto' ? 'gpt-5-mini' : combined.model;
          // Pass the actual API key if available
          if (browserSettings.environmentConfiguration.openaiApiKey) {
            combined.apiKey = browserSettings.environmentConfiguration.openaiApiKey;
          }
        }
        if (browserSettings.environmentConfiguration.geminiConfigured) {
          combined.provider = 'gemini';
          combined.model = combined.model === 'auto' ? 'gemini-1.5-flash' : combined.model;
          // Pass the actual API key if available
          if (browserSettings.environmentConfiguration.geminiApiKey) {
            combined.apiKey = browserSettings.environmentConfiguration.geminiApiKey;
          }
        }
      }

      return combined;
    } catch (error) {
      console.error('Failed to get combined settings:', error);
      return this.getDefaultSettings();
    }
  }

  // Check if API key is available (from storage or environment)
  async hasApiKey(provider: string): Promise<boolean> {
    try {
      const settings = await this.getCombinedSettings();
      
      // Check environment variables first
      if (settings.environmentConfiguration) {
        if (provider === 'openai' && settings.environmentConfiguration.openaiConfigured) {
          return true;
        }
        if (provider === 'gemini' && settings.environmentConfiguration.geminiConfigured) {
          return true;
        }
      }
      
      // Check stored API keys
      const result = await chrome.storage.local.get([`vibe_ai_api_key_${provider}`]);
      return !!result[`vibe_ai_api_key_${provider}`];
    } catch (error) {
      console.error(`Failed to check API key for ${provider}:`, error);
      return false;
    }
  }

  // Save API key to extension storage
  async saveApiKey(provider: string, apiKey: string): Promise<boolean> {
    try {
      await chrome.storage.local.set({
        [`vibe_ai_api_key_${provider}`]: apiKey
      });
      
      // Also sync to browser settings
      await this.callBrowserSettingsAPI('setVibeAIApiKey', [provider, apiKey]);
      
      return true;
    } catch (error) {
      console.error(`Failed to save API key for ${provider}:`, error);
      return false;
    }
  }

  // Test API key validity
  async testApiKey(provider: string): Promise<{isValid: boolean, errorMessage?: string}> {
    try {
      const result = await this.callBrowserSettingsAPI('testVibeAIApiKey', [provider]);
      return {
        isValid: result.isValid,
        errorMessage: result.errorMessage
      };
    } catch (error) {
      console.error(`Failed to test API key for ${provider}:`, error);
      return {
        isValid: false,
        errorMessage: error instanceof Error ? error.message : 'Unknown error'
      };
    }
  }

  private async getBrowserSettings(): Promise<any> {
    try {
      return await this.callBrowserSettingsAPI('getVibeAIState');
    } catch (error) {
      console.error('Failed to get browser settings:', error);
      return {};
    }
  }

  private async getExtensionSettings(): Promise<any> {
    try {
      const result = await chrome.storage.local.get([
        'vibe_ai_provider',
        'vibe_ai_model',
        'vibe_ai_temperature',
        'vibe_ai_max_tokens',
        'vibe_ai_agent_timeout'
      ]);
      
      return {
        provider: result.vibe_ai_provider,
        model: result.vibe_ai_model,
        temperature: result.vibe_ai_temperature,
        maxTokens: result.vibe_ai_max_tokens,
        agentTimeout: result.vibe_ai_agent_timeout
      };
    } catch (error) {
      console.error('Failed to get extension settings:', error);
      return {};
    }
  }

  private getDefaultSettings(): any {
    return {
      provider: 'auto',
      model: 'auto',
      temperature: 0.7,
      maxTokens: 1000,
      agentTimeout: 30000,
      environmentConfiguration: {
        openaiConfigured: false,
        geminiConfigured: false
      }
    };
  }

  private async callBrowserSettingsAPI(method: string, args: any[] = []): Promise<any> {
    return new Promise((resolve, reject) => {
      // Check if we're in an extension context
      if (typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.sendMessage) {
        // Send message to content script or background script to call browser settings
        chrome.runtime.sendMessage({
          type: 'BROWSER_SETTINGS_CALL',
          method,
          args
        }, (response: any) => {
          if (chrome.runtime.lastError) {
            reject(new Error(chrome.runtime.lastError.message));
          } else if (response && response.error) {
            reject(new Error(response.error));
          } else {
            resolve(response);
          }
        });
      } else {
        // Direct call if we're in the browser context
        try {
          chrome.send(method, [
            (result: any) => resolve(result),
            ...args
          ]);
        } catch (error) {
          reject(error);
        }
      }
    });
  }
}

// Export singleton instance
export const settingsBridge = SettingsBridge.getInstance();