/**
 * TypeScript declarations for Chrome Built-in AI (Gemini Nano)
 *
 * Chrome 137+ exposes self.ai.languageModel for on-device AI.
 *
 * @see https://developer.chrome.com/docs/ai/get-started
 */

declare global {
  interface AILanguageModelCapabilities {
    /** Whether the model is available: 'readily' | 'after-download' | 'no' */
    available: 'readily' | 'after-download' | 'no';
    /** Default temperature value */
    defaultTemperature?: number;
    /** Default top-K value */
    defaultTopK?: number;
    /** Maximum top-K value */
    maxTopK?: number;
  }

  interface AILanguageModelCreateOptions {
    /** Temperature for generation (0.0 - 1.0) */
    temperature?: number;
    /** Top-K sampling parameter */
    topK?: number;
    /** System prompt to set context */
    systemPrompt?: string;
    /** Initial prompts for context (deprecated, use systemPrompt) */
    initialPrompts?: Array<{ role: 'system' | 'user' | 'assistant'; content: string }>;
    /** Signal for aborting the creation */
    signal?: AbortSignal;
    /** Monitor for download progress when status is 'after-download' */
    monitor?: (monitor: AILanguageModelDownloadProgressMonitor) => void;
  }

  interface AILanguageModelDownloadProgressMonitor extends EventTarget {
    addEventListener(
      type: 'downloadprogress',
      listener: (event: AILanguageModelDownloadProgressEvent) => void
    ): void;
  }

  interface AILanguageModelDownloadProgressEvent extends Event {
    /** Bytes downloaded so far */
    loaded: number;
    /** Total bytes to download */
    total: number;
  }

  interface AILanguageModelSession {
    /** Send a prompt and get a response */
    prompt(input: string, options?: { signal?: AbortSignal }): Promise<string>;

    /** Send a prompt and get a streaming response */
    promptStreaming(input: string, options?: { signal?: AbortSignal }): ReadableStream<string>;

    /** Clone the session with the same context */
    clone(options?: { signal?: AbortSignal }): Promise<AILanguageModelSession>;

    /** Destroy the session and free resources */
    destroy(): void;

    /** Count tokens in the input (approximate) */
    countPromptTokens?(input: string, options?: { signal?: AbortSignal }): Promise<number>;

    /** Maximum number of tokens the model can handle */
    maxTokens: number;

    /** Number of tokens already used in context */
    tokensSoFar: number;

    /** Number of tokens remaining */
    tokensLeft: number;

    /** Current top-K value */
    topK: number;

    /** Current temperature value */
    temperature: number;
  }

  interface AILanguageModel {
    /** Check model capabilities and availability */
    capabilities(): Promise<AILanguageModelCapabilities>;

    /** Create a new language model session */
    create(options?: AILanguageModelCreateOptions): Promise<AILanguageModelSession>;
  }

  interface AI {
    /** Language model API (Gemini Nano) */
    languageModel?: AILanguageModel;

    /** Summarizer API */
    summarizer?: unknown;

    /** Writer API */
    writer?: unknown;

    /** Rewriter API */
    rewriter?: unknown;
  }

  interface WindowOrWorkerGlobalScope {
    /** Chrome Built-in AI APIs */
    ai?: AI;
  }

  // Extend self for service workers
  var ai: AI | undefined;
}

export {};
