// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// Configuration management for Vibe AI agent
export interface AIAgentConfig {
  // LLM Provider settings
  provider: 'auto' | 'openai' | 'claude' | 'gemini' | 'local';
  model: string;
  apiKey?: string;
  baseURL?: string; // For local LLMs
  
  // Model parameters
  temperature: number;
  maxTokens: number;
  streaming: boolean;
  
  // Agent behavior
  agentTimeout: number; // milliseconds
  maxIterations: number;
  verboseLogging: boolean;
  
  // Browser automation settings
  defaultWaitTimeout: number; // for wait_for_element
  screenshotFormat: 'png' | 'jpeg';
  screenshotQuality: number; // 0-100 for jpeg
  screenshotMaxWidth: number; // max width in pixels (default: 1024)
  screenshotGrayscale: boolean; // convert to grayscale (default: false)
  screenshotDetail: 'low' | 'high'; // detail level (default: 'high')
}

// Default configurations for each provider
const DEFAULT_CONFIGS: Record<string, Partial<AIAgentConfig>> = {
  auto: {
    provider: 'auto',
    model: 'auto',
    temperature: 0.7,
    maxTokens: 1000,
    streaming: false,
    agentTimeout: 30000,
    maxIterations: 5,
    verboseLogging: true,
    defaultWaitTimeout: 5000,
    screenshotFormat: 'jpeg',
    screenshotQuality: 70,
    screenshotMaxWidth: 1024,
    screenshotGrayscale: false,
    screenshotDetail: 'high',
  },
  
  openai: {
    provider: 'openai',
    model: 'openrouter:openai/gpt-oss-120b',
    temperature: 0.7,
    maxTokens: 1000,
    streaming: false,
    agentTimeout: 30000,
    maxIterations: 5,
    verboseLogging: true,
    defaultWaitTimeout: 5000,
    screenshotFormat: 'jpeg',
    screenshotQuality: 70,
    screenshotMaxWidth: 1024,
    screenshotGrayscale: false,
    screenshotDetail: 'high',
  },
  
  claude: {
    provider: 'claude',
    model: 'claude-3-5-haiku-20241022',
    temperature: 0.7,
    maxTokens: 1000,
    streaming: false,
    agentTimeout: 30000,
    maxIterations: 5,
    verboseLogging: true,
    defaultWaitTimeout: 5000,
    screenshotFormat: 'jpeg',
    screenshotQuality: 70,
    screenshotMaxWidth: 1024,
    screenshotGrayscale: false,
    screenshotDetail: 'high',
  },
  
  gemini: {
    provider: 'gemini',
    model: 'gemini-1.5-flash',
    temperature: 0.7,
    maxTokens: 1000,
    streaming: false,
    agentTimeout: 30000,
    maxIterations: 5,
    verboseLogging: true,
    defaultWaitTimeout: 5000,
    screenshotFormat: 'jpeg',
    screenshotQuality: 70,
    screenshotMaxWidth: 1024,
    screenshotGrayscale: false,
    screenshotDetail: 'high',
  },
  
  local: {
    provider: 'local',
    model: 'llama3.1',
    baseURL: 'http://localhost:11434/v1',
    temperature: 0.7,
    maxTokens: 1000,
    streaming: false,
    agentTimeout: 45000, // Local LLMs might be slower
    maxIterations: 5,
    verboseLogging: true,
    defaultWaitTimeout: 5000,
    screenshotFormat: 'jpeg',
    screenshotQuality: 70,
    screenshotMaxWidth: 1024,
    screenshotGrayscale: false,
    screenshotDetail: 'high',
  },
};



export class AIConfigManager {
  private currentConfig: AIAgentConfig;
  private configStorageKey = 'vibe-ai-agent-config';
  
  constructor() {
    this.currentConfig = DEFAULT_CONFIGS['auto'] as AIAgentConfig;
    this.loadConfig().then(config => {
      this.currentConfig = config;
    }).catch(error => {
      console.warn('Failed to load AI config, using defaults:', error);
    });
  }

  async loadConfig(): Promise<AIAgentConfig> {
    try {
      const data = await chrome.storage.local.get(this.configStorageKey);
      if (data && data[this.configStorageKey]) {
        const stored = data[this.configStorageKey];
        const parsed = JSON.parse(stored);
        // Merge with defaults to ensure all fields are present
        return { ...DEFAULT_CONFIGS[parsed.provider] || DEFAULT_CONFIGS['auto'], ...parsed };
      }
    } catch (error: any) {
      console.warn('Failed to load AI config, using defaults:', error);
    }

    return DEFAULT_CONFIGS['auto'] as AIAgentConfig;
  }

  saveConfig(config: Partial<AIAgentConfig>): void {
    this.currentConfig = { ...this.currentConfig, ...config };

    chrome.storage.local.set({ [this.configStorageKey]: JSON.stringify(this.currentConfig) })
      .then(() => {
        console.log('AI config saved successfully');
      })
      .catch((error: any) => {
        console.error('Failed to save AI config:', error);
      });
  }
  
  getConfig(): AIAgentConfig {
    return { ...this.currentConfig };
  }
  
  setProvider(provider: string): void {
    if (!(provider in DEFAULT_CONFIGS)) {
      throw new Error(`Unsupported provider: ${provider}`);
    }
    
    const newConfig = {
      ...DEFAULT_CONFIGS[provider],
      provider: provider as any,
    };
    
    this.saveConfig(newConfig);
  }
  
  setModel(model: string): void {
    // Model validation is now handled by the Vibe settings page.
    this.saveConfig({ model });
  }
  
  setApiKey(apiKey: string): void {
    this.saveConfig({ apiKey });
  }
  
  setBaseUrl(baseURL: string): void {
    this.saveConfig({ baseURL });
  }
  
  updateModelParams(params: Partial<Pick<AIAgentConfig, 'temperature' | 'maxTokens' | 'streaming'>>): void {
    this.saveConfig(params);
  }
  
  updateBehaviorSettings(settings: Partial<Pick<AIAgentConfig, 'agentTimeout' | 'maxIterations' | 'verboseLogging'>>): void {
    this.saveConfig(settings);
  }
  
  updateBrowserSettings(settings: Partial<Pick<AIAgentConfig, 'defaultWaitTimeout' | 'screenshotFormat' | 'screenshotQuality' | 'screenshotMaxWidth' | 'screenshotGrayscale' | 'screenshotDetail'>>): void {
    this.saveConfig(settings);
  }
  
  getAvailableProviders(): string[] {
    return Object.keys(DEFAULT_CONFIGS);
  }
  
  // Model options are now handled by the Vibe settings page UI.
  getSupportedModels(provider?: string): string[] {
    return [];
  }
  
  validateConfig(config: Partial<AIAgentConfig>): string[] {
    const errors: string[] = [];
    
    if (config.provider && !['auto', 'openai', 'claude', 'gemini', 'local'].includes(config.provider)) {
      errors.push('Invalid provider');
    }
    
    if (config.temperature !== undefined && (config.temperature < 0 || config.temperature > 2)) {
      errors.push('Temperature must be between 0 and 2');
    }
    
    if (config.maxTokens !== undefined && config.maxTokens < 1) {
      errors.push('Max tokens must be at least 1');
    }
    
    if (config.agentTimeout !== undefined && config.agentTimeout < 1000) {
      errors.push('Agent timeout must be at least 1000ms');
    }
    
    if (config.maxIterations !== undefined && config.maxIterations < 1) {
      errors.push('Max iterations must be at least 1');
    }
    
    if (config.defaultWaitTimeout !== undefined && config.defaultWaitTimeout < 100) {
      errors.push('Default wait timeout must be at least 100ms');
    }
    
    if (config.screenshotQuality !== undefined && (config.screenshotQuality < 10 || config.screenshotQuality > 100)) {
      errors.push('Screenshot quality must be between 10 and 100');
    }
    
    if (config.screenshotMaxWidth !== undefined && (config.screenshotMaxWidth < 320 || config.screenshotMaxWidth > 4096)) {
      errors.push('Screenshot max width must be between 320 and 4096 pixels');
    }
    
    if (config.screenshotDetail !== undefined && !['low', 'high'].includes(config.screenshotDetail)) {
      errors.push('Screenshot detail must be either "low" or "high"');
    }
    
    if (config.provider === 'local' && config.baseURL && !config.baseURL.startsWith('http')) {
      errors.push('Local LLM base URL must start with http:// or https://');
    }
    
    return errors;
  }
  
  resetToDefaults(provider?: string): void {
    const p = provider || this.currentConfig.provider;
    const defaultConfig = DEFAULT_CONFIGS[p];
    if (defaultConfig) {
      this.saveConfig(defaultConfig);
    }
  }
  
  exportConfig(): string {
    return JSON.stringify(this.currentConfig);
  }
  
  importConfig(configJson: string): boolean {
    try {
      const config = JSON.parse(configJson);
      const errors = this.validateConfig(config);
      
      if (errors.length > 0) {
        console.error('Config validation errors:', errors);
        return false;
      }
      
      this.saveConfig(config);
      return true;
    } catch (error: any) {
      console.error('Failed to import config:', error);
      return false;
    }
  }
  
  // Convert to format expected by VibeLangChain
  toLangChainConfig(): any {
    return {
      backend: this.currentConfig.provider,
      model_name: this.currentConfig.model,
      temperature: this.currentConfig.temperature,
      max_tokens: this.currentConfig.maxTokens,
      api_endpoint: this.currentConfig.baseURL,
    };
  }
}

// Global instance
declare global {
  interface Window {
    vibeAIConfig: AIConfigManager;
  }
}

// Initialize and expose globally
window.vibeAIConfig = new AIConfigManager();

// Signal to other components that config manager is ready
document.dispatchEvent(new CustomEvent('vibe-ai-config-ready'));