/**
 * Centralized API Key Storage Utilities
 * Provides consistent interface for managing provider-specific API keys
 * across extension UI and service worker contexts.
 */

/**
 * Supported AI providers
 */
export const PROVIDERS = {
  VIBE: 'vibe',
  OPENAI: 'openai',
  OPENROUTER: 'openrouter',
  ANTHROPIC: 'anthropic',
  GEMINI: 'gemini',
  AZURE: 'azure',
  CHROME: 'chrome',  // Chrome Built-in AI (Gemini Nano) - no API key required
  GITHUB_COPILOT: 'github-copilot',  // GitHub Copilot subscription
  CLAUDE_CODE: 'claude-code'  // Claude Pro/Max subscription (Anthropic OAuth)
};

/**
 * Storage keys for provider-specific API keys
 */
const STORAGE_KEYS = {
  MODEL: 'vibeModel',
  BASE_URL: 'vibeBaseUrl',
  API_KEY_PREFIX: 'vibeApiKey_'
};

/**
 * Get storage key for provider's API key
 * @param {string} provider - Provider name (openai, anthropic, etc.)
 * @returns {string} Storage key
 */
export function getApiKeyStorageKey(provider) {
  return `${STORAGE_KEYS.API_KEY_PREFIX}${provider}`;
}

/**
 * Get all API key storage keys
 * @returns {string[]} Array of storage keys
 */
export function getAllApiKeyStorageKeys() {
  return Object.values(PROVIDERS).map(getApiKeyStorageKey);
}

/**
 * Extract provider from model string
 * @param {string} model - Model string (e.g., "openai:gpt-5-mini")
 * @returns {string|null} Provider name or null
 */
export function getProviderFromModel(model) {
  if (!model || typeof model !== 'string') {
    return null;
  }
  const [provider] = model.split(':');
  return provider || null;
}

/**
 * Get configuration for a specific provider
 * @param {string} provider - Provider name
 * @returns {Promise<{apiKey: string|null}>}
 */
export async function getProviderConfig(provider) {
  // Check if running in extension context (chrome.storage is available)
  if (typeof chrome === 'undefined' || !chrome.storage) {
    return { apiKey: null };
  }

  const storageKey = getApiKeyStorageKey(provider);
  const data = await chrome.storage.local.get([storageKey]);

  return {
    apiKey: data[storageKey] || null
  };
}

/**
 * Get configuration for the currently selected model
 * @returns {Promise<{model: string|null, provider: string|null, apiKey: string|null, baseURL: string|null}>}
 */
export async function getCurrentModelConfig() {
  // Check if running in extension context
  if (typeof chrome === 'undefined' || !chrome.storage) {
    return { model: null, provider: null, apiKey: null, baseURL: null };
  }

  const keys = [
    STORAGE_KEYS.MODEL,
    STORAGE_KEYS.BASE_URL,
    ...getAllApiKeyStorageKeys()
  ];

  const data = await chrome.storage.local.get(keys);
  const model = data[STORAGE_KEYS.MODEL] || null;
  const provider = getProviderFromModel(model);

  let apiKey = null;
  if (provider) {
    apiKey = data[getApiKeyStorageKey(provider)] || null;
  }

  return {
    model,
    provider,
    apiKey,
    baseURL: data[STORAGE_KEYS.BASE_URL] || null
  };
}

/**
 * Get all provider configurations
 * @returns {Promise<{model: string|null, baseURL: string|null, providers: Object}>}
 */
export async function getAllProviderConfigs() {
  const keys = [
    STORAGE_KEYS.MODEL,
    STORAGE_KEYS.BASE_URL,
    ...getAllApiKeyStorageKeys()
  ];

  const data = await chrome.storage.local.get(keys);

  const providers = {};
  Object.values(PROVIDERS).forEach(provider => {
    providers[provider] = data[getApiKeyStorageKey(provider)] || null;
  });

  return {
    model: data[STORAGE_KEYS.MODEL] || null,
    baseURL: data[STORAGE_KEYS.BASE_URL] || null,
    providers
  };
}

/**
 * Set API key for a specific provider
 * @param {string} provider - Provider name
 * @param {string} apiKey - API key value
 * @returns {Promise<void>}
 */
export async function setProviderApiKey(provider, apiKey) {
  const storageKey = getApiKeyStorageKey(provider);
  await chrome.storage.local.set({
    [storageKey]: apiKey
  });
}

/**
 * Set configuration for current model
 * @param {Object} config - Configuration object
 * @param {string} config.model - Model string (e.g., "openai:gpt-5-mini")
 * @param {string} config.apiKey - API key for the provider
 * @param {string} [config.baseURL] - Optional base URL
 * @returns {Promise<void>}
 */
export async function setModelConfig(config) {
  const { model, apiKey, baseURL } = config;
  const provider = getProviderFromModel(model);

  if (!provider) {
    throw new Error('Invalid model format. Expected "provider:model-name"');
  }

  const updates = {
    [STORAGE_KEYS.MODEL]: model
  };

  if (apiKey !== undefined) {
    updates[getApiKeyStorageKey(provider)] = apiKey;
  }

  if (baseURL !== undefined) {
    updates[STORAGE_KEYS.BASE_URL] = baseURL;
  }

  await chrome.storage.local.set(updates);
}

/**
 * Load API key from storage
 * OAuth tokens are stored directly in vibeApiKey_${provider} by TokenManager
 * @param {string} provider - Provider name (openai, anthropic, etc.)
 * @returns {Promise<string|null>} - API key or null if not found
 */
export async function loadApiKey(provider) {
  // Check provider-specific API key in storage (works for both manual and OAuth)
  // OAuth's TokenManager stores tokens directly in vibeApiKey_${provider}
  const config = await getProviderConfig(provider);
  return config.apiKey;
}

/**
 * Check if LLM is properly configured with an API key
 * @returns {Promise<boolean>}
 */
export async function isLLMConfigured() {
  try {
    const config = await getCurrentModelConfig();
    return !!(config.model && config.apiKey);
  } catch (error) {
    console.warn('Failed to check LLM configuration:', error);
    return false;
  }
}

/**
 * Clear all provider API keys
 * @returns {Promise<void>}
 */
export async function clearAllApiKeys() {
  const keys = getAllApiKeyStorageKeys();
  await chrome.storage.local.remove(keys);
}

/**
 * Clear API key for specific provider
 * @param {string} provider - Provider name
 * @returns {Promise<void>}
 */
export async function clearProviderApiKey(provider) {
  const storageKey = getApiKeyStorageKey(provider);
  await chrome.storage.local.remove(storageKey);
}
