import yaml from 'js-yaml';

/**
 * YAML serializer for agent reflection and thinking progress
 * Provides cleaner, more readable output than JSON.stringify()
 */

/**
 * Serialize data to YAML format
 * @param {any} data - Data to serialize
 * @param {Object} options - YAML dump options
 * @returns {string} YAML string
 */
export function toYAML(data, options = {}) {
  const defaultOptions = {
    indent: 2,
    noRefs: true,
    sortKeys: false,
    lineWidth: 120,
    skipInvalid: true,
    ...options
  };
  
  try {
    return yaml.dump(data, defaultOptions);
  } catch (error) {
    console.error('[YAML] Serialization error:', error);
    // Fallback to JSON if YAML fails
    return JSON.stringify(data, null, 2);
  }
}

/**
 * Parse YAML string back to object
 * @param {string} yamlString - YAML string to parse
 * @returns {any} Parsed object
 */
export function fromYAML(yamlString) {
  try {
    return yaml.load(yamlString);
  } catch (error) {
    console.error('[YAML] Parse error:', error);
    // Try JSON parse as fallback
    try {
      return JSON.parse(yamlString);
    } catch {
      return null;
    }
  }
}

/**
 * Format thinking/reflection data for YAML output
 * @param {Object} data - Reflection or thinking data
 * @returns {string} Formatted YAML string
 */
export function formatThinkingYAML(data) {
  // Clean up data for better YAML representation
  const cleanData = {
    ...data,
    timestamp: new Date().toISOString()
  };
  
  // Remove undefined values
  Object.keys(cleanData).forEach(key => {
    if (cleanData[key] === undefined) {
      delete cleanData[key];
    }
  });
  
  return toYAML(cleanData, {
    indent: 2,
    lineWidth: 120,
    sortKeys: false
  });
}

/**
 * Format reflection data for YAML output
 * @param {Object} reflection - Reflection data from LLM
 * @returns {string} Formatted YAML string
 */
export function formatReflectionYAML(reflection) {
  const formatted = {
    taskComplete: reflection.taskComplete,
    completionPercentage: reflection.completionPercentage,
    progress: reflection.progress,
    issues: reflection.issues || [],
    nextSteps: reflection.nextSteps || [],
    critique: reflection.critique,
    requestMoreIterations: reflection.requestMoreIterations || false,
    additionalIterationsNeeded: reflection.additionalIterationsNeeded || 0
  };
  
  return toYAML(formatted, {
    indent: 2,
    lineWidth: 120,
    sortKeys: false
  });
}

/**
 * Format tool call data for YAML output
 * @param {string} toolName - Name of the tool
 * @param {Object} args - Tool arguments
 * @returns {string} Formatted YAML string
 */
export function formatToolCallYAML(toolName, args) {
  const formatted = {
    tool: toolName,
    arguments: args,
    timestamp: new Date().toISOString()
  };
  
  return toYAML(formatted, {
    indent: 2,
    lineWidth: 80,
    sortKeys: false
  });
}