/**
 * Page State Manager
 *
 * Centralizes page state gathering for agents.
 * Handles:
 * - Page content extraction (indexedNodesMarkdown, URL, title, HTML)
 * - Browser tabs enumeration
 * - Screenshot capture (for vision-enabled models)
 *
 * Returns raw data that agents can compose into messages as needed.
 */

import { HumanMessage } from "@langchain/core/messages";

/**
 * Get current page state with optional content types
 *
 * @param {Object} options
 * @param {string|null} options.tabId - Tab ID to get content from (null = active tab)
 * @param {Object} options.getPageContentTool - Tool to get page content
 * @param {Function} options.getTabs - Function to get all browser tabs
 * @param {Function} options.takeScreenshotCallback - Callback function to capture screenshots
 * @param {boolean} options.supportsVision - Whether the model supports vision
 * @param {Object} options.screenshotOptions - Screenshot options (maxWidth, quality, detail)
 * @param {boolean} options.includeIndexedNodesMarkdown - Include markdown with indexed nodes [123]
 * @param {boolean} options.includeIndexedNodesHtml - Include HTML content
 * @param {boolean} options.includeScreenshot - Capture and include screenshot
 * @returns {Promise<{
 *   indexedNodesMarkdown: string|null,
 *   indexedNodesHtml: string|null,
 *   screenshot: string|null,
 *   tabs: Array,
 *   pageUrl: string,
 *   pageTitle: string,
 *   tabId: any,
 *   error: string|null
 * }>}
 */
export async function getPageState(options = {}) {
  const {
    tabId = null,
    getPageContentTool,
    getTabs,
    takeScreenshotCallback,
    supportsVision = false,
    screenshotOptions = {},
    includeIndexedNodesMarkdown = false,
    includeIndexedNodesHtml = false,
    includeScreenshot = false
  } = options;

  // Initialize result object
  const result = {
    indexedNodesMarkdown: null,
    indexedNodesHtml: null,
    screenshot: null,
    tabs: [],
    pageUrl: "",
    pageTitle: "",
    tabId: undefined,
    error: null
  };

  // Get page content if any content type is requested
  if (includeIndexedNodesMarkdown || includeIndexedNodesHtml) {
    try {
      const pageResult = await getPageContentTool.execute(tabId);

      // Always extract metadata first
      result.pageUrl = pageResult.url || '';
      result.pageTitle = pageResult.title || '';
      result.tabId = pageResult.tabId;

      // Extract data based on flags
      if (includeIndexedNodesMarkdown) {
        result.indexedNodesMarkdown = pageResult.content || '';
      }

      if (includeIndexedNodesHtml) {
        result.indexedNodesHtml = pageResult.htmlContent || '';
      }
    } catch (error) {
      console.log('🌐 Page content extraction error:', error.message);
      result.error = `[Page content could not be extracted due to: ${error.message}. This typically happens during page navigation or when the page context is destroyed. The page may still have loaded successfully.]`;
    }
  }

  // Get all browser tabs
  if (getTabs) {
    try {
      result.tabs = await getTabs();
    } catch (error) {
      console.log('🌐 Failed to get tabs:', error.message);
    }
  }

  // Capture screenshot if enabled
  if (includeScreenshot && supportsVision && takeScreenshotCallback) {
    // Wait for DOM highlights to render
    await new Promise(resolve => setTimeout(resolve, 300));

    try {
      const screenshotResult = await takeScreenshotCallback({
        detail: screenshotOptions.detail || 'high',
        maxWidth: screenshotOptions.maxWidth || 1280,
        quality: screenshotOptions.quality || 80,
        reasoning: screenshotOptions.reasoning || 'Agent context',
        tabId: tabId // Pass tabId (null = active tab)
      });

      if (typeof screenshotResult === 'string' && screenshotResult.startsWith('data:image')) {
        result.screenshot = screenshotResult;
        console.log('📸 Screenshot captured successfully');
      } else {
        console.log('📸 Screenshot callback returned invalid result:', typeof screenshotResult);
      }
    } catch (error) {
      console.warn('🌐 Screenshot capture failed:', error);
    }
  }

  return result;
}

/**
 * DEPRECATED: Use getPageState() instead
 * Legacy function for backward compatibility during migration
 * 
 * @deprecated This function will be removed in a future version
 */
export async function getStateMessage(options = {}) {
  const {
    tabId = null,
    takeScreenshot = false,
    getPageContentTool,
    getTabs,
    takeScreenshotCallback,
    supportsVision = false,
    reasoning = '',
    currentReasoning = null,
    needsVerification = false,
    verificationAction = '',
    screenshotOptions = {},
    includeAllPages = true
  } = options;

  // Call new getPageState function
  const pageState = await getPageState({
    tabId,
    getPageContentTool,
    getTabs,
    takeScreenshotCallback,
    supportsVision,
    screenshotOptions,
    includeIndexedNodesMarkdown: true,
    includeIndexedNodesHtml: false,
    includeScreenshot: takeScreenshot
  });

  // Build message content
  let messageText = '';

  // Add current reasoning if present
  if (currentReasoning) {
    messageText += `Goal: ${currentReasoning}\n`;
  }

  // Add current tab info
  messageText += `Currently opened active tab page\n`;
  messageText += `Tab ID: ${pageState.tabId || 'Unknown'}\n`;
  messageText += `Title: ${pageState.pageTitle}\n`;
  messageText += `Url: ${pageState.pageUrl}\n`;

  // Add all browser tabs
  if (pageState.tabs.length > 0) {
    messageText += `\nAll browser tabs (${pageState.tabs.length} total):\n`;
    pageState.tabs.forEach((tab, index) => {
      const activeMarker = tab.active ? ' [ACTIVE]' : '';
      messageText += `  ${index + 1}. Tab ${tab.tabId}${activeMarker}: ${tab.title}\n     URL: ${tab.url}\n`;
    });
    messageText += '\n';
  }

  // Add verification instruction if needed
  if (needsVerification && verificationAction) {
    messageText += `\nVERIFICATION REQUIRED: You just performed '${verificationAction}'.\n`;
    messageText += `Before proceeding with the next action:\n`;
    messageText += `1. Carefully review the current page content below\n`;
    messageText += `2. Verify that your previous action succeeded (field was filled, element was clicked, etc.)\n`;
    messageText += `3. Check if the expected change occurred on the page\n`;
    messageText += `4. Only then decide on the next tool to use\n\n`;
  }

  // Add previous reasoning context
  if (currentReasoning) {
    messageText += `\nPrevious Tool Reasoning:\n\n`;
  }

  // Add page content
  if (!pageState.error) {
    messageText += `\nBody:\n`;
    messageText += `\`\`\`markdown\n`;
    messageText += pageState.indexedNodesMarkdown;
    messageText += `\n\`\`\``;
  } else {
    messageText += `\nError: ${pageState.error}`;
  }

  // Build HumanMessage with text + optional screenshot
  let messageContent;
  let visionUsed = false;

  if (pageState.screenshot) {
    // Multimodal format for vision-enabled models
    messageContent = [
      { type: "text", text: messageText },
      { type: "image_url", image_url: { url: pageState.screenshot } }
    ];
    visionUsed = true;
  } else {
    // Simple string format for text-only models
    messageContent = messageText;
  }

  const message = new HumanMessage({
    content: messageContent,
    additional_kwargs: { isPageContent: true }
  });

  // Return legacy format
  return {
    message,
    pageState: {
      pageContent: pageState.indexedNodesMarkdown,
      htmlContent: pageState.indexedNodesHtml,
      pageUrl: pageState.pageUrl,
      pageTitle: pageState.pageTitle,
      tabId: pageState.tabId,
      error: pageState.error
    },
    visionUsed
  };
}
