// Loader for opencode-compatible tools in Vibe
// Allows dynamic import and registration of opencode tools (e.g. from @opencode-ai/plugin)

const { z } = require('zod');

/**
 * Loads an opencode tool and adapts it to Vibe's tool registry format.
 * @param {string} toolModulePath - Path to the opencode tool module (e.g. '@opencode-ai/plugin/dist/webfetch')
 * @param {string} toolName - Name of the tool export (default: 'default')
 * @returns {Promise<object>} - Adapted tool definition for Vibe
 */
async function loadOpencodeTool(toolModulePath, toolName = 'default') {
  const mod = await import(toolModulePath);
  const tool = mod[toolName] || mod.default;
  if (!tool) throw new Error(`Tool not found in module: ${toolModulePath}`);

  // Opencode tool: { description, args, execute }
  if (!tool.description || !tool.args || typeof tool.execute !== 'function') {
    throw new Error('Invalid opencode tool format');
  }

  // Adapt to Vibe's expected tool interface
  return {
    name: toolName,
    description: tool.description,
    args: tool.args, // Zod schema
    async call(args, context) {
      // Validate args using Zod
      const parsedArgs = z.object(tool.args).parse(args);
      return tool.execute(parsedArgs, context);
    },
  };
}

module.exports = { loadOpencodeTool };
