/**
 * Context Compaction - Adapted from OpenCode's SessionCompaction
 * @see https://github.com/sst/opencode/blob/main/packages/opencode/src/session/compaction.ts
 */

import { HumanMessage, AIMessage, ToolMessage, BaseMessage } from "@langchain/core/messages";
import { BaseChatModel } from "@langchain/core/language_models/chat_models";

export const PRUNE_MINIMUM = 20_000;
export const PRUNE_PROTECT = 40_000;

/**
 * Remove orphaned ToolMessages that don't have a corresponding tool_call in a preceding AIMessage.
 * This prevents "messages with role 'tool' must be a response to a preceding message with 'tool_calls'" errors.
 * 
 * @param messages - Array of messages to validate
 * @returns Cleaned array with orphaned ToolMessages removed
 */
export function removeOrphanedToolMessages(messages: BaseMessage[]): BaseMessage[] {
  // Collect all valid tool_call_ids from AIMessages
  const validToolCallIds = new Set<string>();
  for (const msg of messages) {
    if (msg instanceof AIMessage && msg.tool_calls && Array.isArray(msg.tool_calls)) {
      for (const tc of msg.tool_calls) {
        if (tc.id) {
          validToolCallIds.add(tc.id);
        }
      }
    }
    // Also check additional_kwargs.tool_calls for LangChain compatibility
    if (msg instanceof AIMessage && msg.additional_kwargs?.tool_calls) {
      for (const tc of msg.additional_kwargs.tool_calls) {
        if (tc.id) {
          validToolCallIds.add(tc.id);
        }
      }
    }
  }

  // Filter out orphaned ToolMessages
  let removedCount = 0;
  const filtered = messages.filter(msg => {
    if (msg instanceof ToolMessage && msg.tool_call_id) {
      if (!validToolCallIds.has(msg.tool_call_id)) {
        removedCount++;
        return false; // Remove orphaned ToolMessage
      }
    }
    return true;
  });

  if (removedCount > 0) {
    console.log(`🗜️ Removed ${removedCount} orphaned ToolMessage(s) before LLM call`);
  }

  return filtered;
}

// OpenCode's default compaction prompt
const DEFAULT_PROMPT =
  "Provide a detailed prompt for continuing our conversation above. Focus on information that would be helpful for continuing the conversation, including what we did, what we're doing, which files we're working on, and what we're going to do next considering new session will not have access to our conversation.";

/**
 * Estimate tokens for a string (matches OpenCode's Token.estimate)
 */
function estimate(text: string | undefined): number {
  if (!text) return 0;
  return Math.ceil(text.length / 4);
}

/**
 * Get content string from a message
 */
function getMessageContent(msg: BaseMessage): string {
  if (typeof msg.content === "string") {
    return msg.content;
  }
  if (Array.isArray(msg.content)) {
    return msg.content.map((c) => (typeof c === "string" ? c : JSON.stringify(c))).join("\n");
  }
  return msg.content ? JSON.stringify(msg.content) : "";
}

/**
 * Prune messages by clearing old tool outputs.
 *
 * Direct adaptation of OpenCode's SessionCompaction.prune()
 *
 * Goes backwards through messages until there are PRUNE_PROTECT tokens worth of tool
 * calls. Then clears output of previous tool calls. Idea is to throw away old
 * tool calls that are no longer relevant.
 */
export async function prune(messages: BaseMessage[]): Promise<BaseMessage[]> {
  console.log("🗜️ pruning");

  let total = 0;
  let pruned = 0;
  const toPrune: { index: number; estimate: number }[] = [];
  let turns = 0;

  // Go backwards through messages (like OpenCode's loop)
  loop: for (let msgIndex = messages.length - 1; msgIndex >= 0; msgIndex--) {
    const msg = messages[msgIndex];

    // Count user turns
    if (msg instanceof HumanMessage) turns++;

    // Skip until we have at least 2 user turns (like OpenCode: turns < 2)
    if (turns < 2) continue;

    // If we hit a message that was already compacted (has summary marker), stop
    if (msg.additional_kwargs?.summary) break loop;

    // Count ALL tokens towards depth
    const content = getMessageContent(msg);
    const est = estimate(content);
    total += est;

    // Only prune ToolMessage content (equivalent to OpenCode's part.type === "tool")
    if (msg instanceof ToolMessage) {
      // Check if already compacted
      if (msg.additional_kwargs?.compacted) break loop;

      // Only prune if we're past the protection threshold
      if (total > PRUNE_PROTECT) {
        pruned += est;
        toPrune.push({ index: msgIndex, estimate: est });
      }
    }
  }

  console.log(`🗜️ found`, { pruned, total });

  // Only prune if we found enough tokens to make it worthwhile (PRUNE_MINIMUM)
  if (pruned > PRUNE_MINIMUM) {
    // Create new messages array with pruned content
    const prunedMessages = [...messages];

    for (const item of toPrune) {
      const originalMsg = prunedMessages[item.index] as ToolMessage;

      // Replace with truncated version (like OpenCode sets output to "[Old tool result content cleared]")
      prunedMessages[item.index] = new ToolMessage({
        content: "[Old tool result content cleared]",
        tool_call_id: originalMsg.tool_call_id,
        name: originalMsg.name,
        additional_kwargs: {
          ...originalMsg.additional_kwargs,
          compacted: Date.now(),
        },
      });
    }

    console.log(`🗜️ pruned`, { count: toPrune.length });
    return prunedMessages;
  }

  return messages;
}

/**
 * Check if context is overflowing
 * Adapted from OpenCode's SessionCompaction.isOverflow()
 */
export function isOverflow(input: {
  tokens: { input: number; output: number; cache?: { read: number } };
  contextLimit: number;
  outputMax?: number;
}): boolean {
  const context = input.contextLimit;
  if (context === 0) return false;

  const count = input.tokens.input + (input.tokens.cache?.read || 0) + input.tokens.output;
  const outputMax = input.outputMax || 32_000;
  const usable = context - outputMax;

  return count > usable;
}

/**
 * Process compaction - create LLM summary of conversation
 * 
 * Direct adaptation of OpenCode's SessionCompaction.process()
 * 
 * CRITICAL: First prunes old tool outputs to reduce token count,
 * THEN sends pruned messages to LLM for summarization.
 */
export async function process(input: {
  messages: BaseMessage[];
  llm: BaseChatModel;
  prompt?: string;
}): Promise<BaseMessage[]> {
  const inputTokens = input.messages.reduce((sum, msg) => sum + estimate(getMessageContent(msg)), 0);
  console.log(`🗜️ processing compaction (input: ~${inputTokens} tokens, ${input.messages.length} messages)`);

  // STEP 1: Prune old tool outputs FIRST to reduce token count
  // This is critical - without pruning, we'd send 100k+ tokens to the summarizer
  const prunedMessages = await prune(input.messages);
  const prunedTokens = prunedMessages.reduce((sum, msg) => sum + estimate(getMessageContent(msg)), 0);
  console.log(`🗜️ after pruning: ~${prunedTokens} tokens (saved ~${inputTokens - prunedTokens})`);

  const promptText = input.prompt ?? DEFAULT_PROMPT;

  // STEP 2: Remove orphaned ToolMessages to prevent API errors
  // This ensures no ToolMessage exists without a corresponding tool_call in a preceding AIMessage
  const validatedMessages = removeOrphanedToolMessages(prunedMessages);

  // STEP 3: Build messages for LLM summarization
  // Use validated messages (with orphans removed) + compaction prompt
  const messagesForLlm = [
    ...validatedMessages,
    new HumanMessage({
      content: promptText,
    }),
  ];

  // STEP 4: Call LLM to generate summary
  const response = await input.llm.invoke(messagesForLlm);

  // Mark response as summary (like OpenCode's summary: true)
  response.additional_kwargs = {
    ...response.additional_kwargs,
    summary: true,
  };

  const outputTokens = estimate(getMessageContent(response));
  console.log(`🗜️ compaction complete: ${inputTokens} -> ${outputTokens} tokens (${Math.round((1 - outputTokens/inputTokens) * 100)}% reduction)`);

  // Return just the summary as the new conversation context
  // The caller will add "Continue if you have next steps" user message
  return [response];
}
