/**
 * CaptchaDetector - Detect and handle CAPTCHA/blocking pages
 * Provides detection patterns and recovery strategies
 */

export class CaptchaDetector {
  constructor() {
    // Common CAPTCHA and blocking indicators
    this.captchaPatterns = {
      // Text patterns that indicate CAPTCHA presence
      textPatterns: [
        /captcha/i,
        /unusual.{0,20}traffic/i,
        /verify.{0,20}you.{0,20}(human|robot)/i,
        /i.{0,10}am.{0,10}not.{0,10}a.{0,10}robot/i,
        /security.{0,20}check/i,
        /automated.{0,20}(requests|behavior)/i,
        /suspicious.{0,20}activity/i,
        /please.{0,20}verify/i,
        /confirm.{0,20}you.{0,20}are.{0,20}human/i,
        /access.{0,20}denied/i,
        /blocked/i,
        /rate.{0,20}limit/i,
        /too.{0,20}many.{0,20}requests/i,
        /forbidden/i,
        /unauthorized/i
      ],
      
      // DOM selectors that typically indicate CAPTCHA
      domSelectors: [
        // Google reCAPTCHA
        'iframe[src*="recaptcha"]',
        'div.g-recaptcha',
        '#g-recaptcha',
        '.g-recaptcha',
        
        // hCaptcha
        'iframe[src*="hcaptcha"]',
        'div.h-captcha',
        '.h-captcha',
        
        // Cloudflare
        'div.cf-browser-verification',
        'div#cf-wrapper',
        '.cf-challenge-form',
        
        // Generic CAPTCHA elements
        'img[src*="captcha"]',
        'input[name*="captcha"]',
        'div[class*="captcha"]',
        'form[action*="captcha"]',
        
        // Blocking/verification pages
        '#challenge-form',
        '.challenge-form',
        'form[name="challenge"]'
      ],
      
      // URL patterns that indicate blocking
      urlPatterns: [
        /\/captcha/i,
        /\/challenge/i,
        /\/verify/i,
        /\/security-check/i,
        /\/blocked/i,
        /sorry\/index/i,  // Google's sorry page
        /\/error/i
      ],
      
      // Title patterns
      titlePatterns: [
        /attention required/i,
        /security check/i,
        /access denied/i,
        /blocked/i,
        /verify/i,
        /captcha/i,
        /one more step/i,
        /please wait/i,
        /checking your browser/i
      ]
    };
    
    // Service-specific patterns
    this.servicePatterns = {
      google: {
        indicators: [
          'Our systems have detected unusual traffic',
          'sorry/index',
          'www.google.com/sorry',
          'Please show you\'re not a robot'
        ],
        recovery: 'Try using a different search engine like DuckDuckGo or Bing'
      },
      cloudflare: {
        indicators: [
          'Checking your browser',
          'cf-browser-verification',
          'ray ID',
          'Please allow up to 5 seconds',
          'DDoS protection by Cloudflare'
        ],
        recovery: 'Wait for automatic verification or try accessing the site later'
      },
      amazon: {
        indicators: [
          'Enter the characters you see below',
          'Type the characters you see in this image',
          'Robot Check'
        ],
        recovery: 'Amazon requires manual verification - task cannot be automated'
      }
    };
    
    this.detectionHistory = [];
    this.maxHistorySize = 10;
  }

  /**
   * Detect CAPTCHA in page content
   */
  async detect(pageContent, pageUrl = '', pageTitle = '') {
    const detection = {
      hasCaptcha: false,
      confidence: 0,
      type: null,
      service: null,
      indicators: [],
      recovery: null,
      timestamp: Date.now()
    };
    
    // Check URL patterns
    if (pageUrl) {
      for (const pattern of this.captchaPatterns.urlPatterns) {
        if (pattern.test(pageUrl)) {
          detection.indicators.push(`URL matches: ${pattern}`);
          detection.confidence += 30;
        }
      }
    }
    
    // Check title patterns
    if (pageTitle) {
      for (const pattern of this.captchaPatterns.titlePatterns) {
        if (pattern.test(pageTitle)) {
          detection.indicators.push(`Title matches: ${pattern}`);
          detection.confidence += 25;
        }
      }
    }
    
    // Check page content for text patterns
    if (pageContent) {
      const contentStr = typeof pageContent === 'string' ? pageContent : JSON.stringify(pageContent);
      
      // Check text patterns
      for (const pattern of this.captchaPatterns.textPatterns) {
        if (pattern.test(contentStr)) {
          const match = contentStr.match(pattern);
          detection.indicators.push(`Text found: "${match[0]}"`);
          detection.confidence += 20;
        }
      }
      
      // Check for DOM selectors in content
      for (const selector of this.captchaPatterns.domSelectors) {
        if (contentStr.includes(selector.replace(/[\[\].*+?^${}()|\\]/g, ''))) {
          detection.indicators.push(`DOM element: ${selector}`);
          detection.confidence += 25;
        }
      }
      
      // Check service-specific patterns
      for (const [service, config] of Object.entries(this.servicePatterns)) {
        let serviceMatch = false;
        for (const indicator of config.indicators) {
          if (contentStr.includes(indicator)) {
            serviceMatch = true;
            detection.indicators.push(`${service} indicator: "${indicator}"`);
            detection.confidence += 15;
          }
        }
        if (serviceMatch) {
          detection.service = service;
          detection.recovery = config.recovery;
        }
      }
    }
    
    // Determine if CAPTCHA is present
    if (detection.confidence >= 50) {
      detection.hasCaptcha = true;
      
      // Determine type based on indicators
      if (detection.indicators.some(i => i.includes('recaptcha') || i.includes('g-recaptcha'))) {
        detection.type = 'recaptcha';
      } else if (detection.indicators.some(i => i.includes('hcaptcha'))) {
        detection.type = 'hcaptcha';
      } else if (detection.indicators.some(i => i.includes('cloudflare'))) {
        detection.type = 'cloudflare';
      } else if (detection.service) {
        detection.type = `${detection.service}-specific`;
      } else {
        detection.type = 'generic';
      }
      
      // Set default recovery if not already set
      if (!detection.recovery) {
        detection.recovery = this.getDefaultRecovery(detection.type);
      }
    }
    
    // Store in history
    this.addToHistory(detection);
    
    return detection;
  }

  /**
   * Get default recovery strategy for CAPTCHA type
   */
  getDefaultRecovery(type) {
    const recoveryStrategies = {
      'recaptcha': 'Google reCAPTCHA detected. Manual intervention required or try a different approach.',
      'hcaptcha': 'hCaptcha detected. Manual intervention required or try a different approach.',
      'cloudflare': 'Cloudflare protection detected. Wait a few seconds for automatic verification.',
      'google-specific': 'Google blocking detected. Try: 1) Use a different search engine, 2) Add delays between requests, 3) Use more natural interaction patterns.',
      'amazon-specific': 'Amazon CAPTCHA detected. Manual verification required - automation not possible.',
      'generic': 'Verification required. Try: 1) Slower, more natural interactions, 2) Different access method, 3) Manual intervention.'
    };
    
    return recoveryStrategies[type] || recoveryStrategies.generic;
  }

  /**
   * Check if repeated CAPTCHAs indicate terminal failure
   */
  isTerminalFailure() {
    // Check recent history for repeated CAPTCHAs
    const recentDetections = this.detectionHistory.slice(-3);
    
    if (recentDetections.length < 2) {
      return false;
    }
    
    // If we've hit CAPTCHAs 3 times in a row, it's terminal
    const allHaveCaptcha = recentDetections.every(d => d.hasCaptcha);
    if (allHaveCaptcha && recentDetections.length >= 3) {
      return true;
    }
    
    // If same service CAPTCHA appears repeatedly
    const services = recentDetections.map(d => d.service).filter(Boolean);
    if (services.length >= 2 && services.every(s => s === services[0])) {
      return true;
    }
    
    return false;
  }

  /**
   * Get handling strategy - always ask user to solve CAPTCHA
   */
  getHandlingStrategy(detection) {
    if (!detection.hasCaptcha) {
      return null;
    }
    
    // Single strategy: Ask user to solve CAPTCHA
    const strategy = {
      shouldAskUser: true,
      captchaType: detection.type || 'verification',
      service: detection.service,
      message: this.getUserMessage(detection),
      instructions: this.getSolvingInstructions(detection)
    };
    
    return strategy;
  }
  
  /**
   * Get user-friendly message about the CAPTCHA
   */
  getUserMessage(detection) {
    const messages = {
      'recaptcha': 'Google reCAPTCHA detected. Please solve it manually to continue.',
      'hcaptcha': 'hCaptcha detected. Please solve it manually to continue.',
      'cloudflare': 'Cloudflare security check detected. Please complete the verification to continue.',
      'google-specific': 'Google has detected unusual traffic. Please complete the CAPTCHA to continue.',
      'amazon-specific': 'Amazon requires verification. Please complete the CAPTCHA to continue.',
      'generic': 'Security verification required. Please complete it to continue.'
    };
    
    return messages[detection.type] || messages.generic;
  }
  
  /**
   * Get instructions for solving the CAPTCHA
   */
  getSolvingInstructions(detection) {
    const instructions = {
      'recaptcha': '1. Click on "I\'m not a robot" checkbox\n2. Complete any image challenges if prompted\n3. Wait for the green checkmark',
      'hcaptcha': '1. Click on the hCaptcha checkbox\n2. Complete the image challenge\n3. Wait for verification',
      'cloudflare': '1. Wait for automatic verification (5-10 seconds)\n2. If prompted, click the checkbox\n3. Complete any challenges',
      'google-specific': '1. Type the characters shown in the image\n2. Click Submit/Continue\n3. If failed, try the audio challenge',
      'amazon-specific': '1. Type the characters shown in the image\n2. Click Continue\n3. Make sure characters match exactly',
      'generic': '1. Look for verification prompt or CAPTCHA\n2. Complete the required challenge\n3. Submit and wait for page to load'
    };
    
    return instructions[detection.type] || instructions.generic;
  }

  /**
   * Add detection to history
   */
  addToHistory(detection) {
    this.detectionHistory.push(detection);
    
    // Maintain max history size
    if (this.detectionHistory.length > this.maxHistorySize) {
      this.detectionHistory.shift();
    }
  }

  /**
   * Clear detection history
   */
  clearHistory() {
    this.detectionHistory = [];
  }

  /**
   * Get summary of recent detections
   */
  getDetectionSummary() {
    const recent = this.detectionHistory.slice(-5);
    const captchaCount = recent.filter(d => d.hasCaptcha).length;
    const types = [...new Set(recent.filter(d => d.type).map(d => d.type))];
    const services = [...new Set(recent.filter(d => d.service).map(d => d.service))];
    
    return {
      recentDetections: recent.length,
      captchasFound: captchaCount,
      types,
      services,
      isBlocked: this.isTerminalFailure()
    };
  }
}

export default CaptchaDetector;