/**
 * State serialization utilities for persisting ReactGraph state
 * Handles conversion between LangChain message objects and serializable format
 */

import { BaseMessage, HumanMessage, AIMessage, SystemMessage, ToolMessage } from "@langchain/core/messages";

import { deserializeMessage, deserializeMessages } from "../shared/messageUtils.js";

export interface SerializedMessage {
  type: string;
  content: any;
  additional_kwargs: any;
  tool_calls?: any[];
  usage_metadata?: any;
  tool_call_id?: string;
  name?: string;
  response_metadata?: any;
}

export interface SerializedState {
  messages: SerializedMessage[];
  allMessages: SerializedMessage[];
  iterations: number;
  maxIterations: number;
  reflectionFeedback: { feedback: string; completed: boolean };
  captchaBlocked?: boolean;
  currentReasoning?: string | null;
  tokenUsage: {
    input_tokens: number;
    output_tokens: number;
    total_tokens: number;
  };
  toolResultCache: Record<string, any>;
  plan?: any[];
  lastUpdated: number;
}

export interface TokenUsage {
  input_tokens: number;
  output_tokens: number;
  total_tokens: number;
}

export interface ReactGraphState {
  messages: BaseMessage[];
  allMessages: BaseMessage[];
  iterations: number;
  maxIterations: number;
  reflectionFeedback: { feedback: string; completed: boolean };
  captchaBlocked?: boolean;
  currentReasoning?: string | null;
  tokenUsage: TokenUsage;
  toolResultCache: Record<string, any>;
  plan?: any[];
  previousPageState?: {
    url: string;
    contentHash: string;
  } | null;
}

export class StateSerializer {
  /**
   * Serialize ReactGraph state for storage
   * @param state - The ReactGraph state object
   * @returns Serializable state object
   */
  static serializeState(state: ReactGraphState | null): SerializedState | null {
    if (!state) return null;

    return {
      messages: state.messages ? this.serializeMessages(state.messages) : [],
      allMessages: state.allMessages ? this.serializeMessages(state.allMessages) : [],
      iterations: state.iterations || 0,
      maxIterations: state.maxIterations || 512,
      reflectionFeedback: state.reflectionFeedback || { feedback: "", completed: false },
      captchaBlocked: state.captchaBlocked || false,
      currentReasoning: state.currentReasoning || null,
      tokenUsage: state.tokenUsage || {
        input_tokens: 0,
        output_tokens: 0,
        total_tokens: 0
      },
      toolResultCache: state.toolResultCache || {},
      plan: state.plan || [],
      // Add timestamp for cleanup
      lastUpdated: Date.now()
    };
  }

  /**
   * Serialize array of LangChain messages
   * @param messages - Array of LangChain message objects
   * @returns Array of serializable message objects
   */
  static serializeMessages(messages: BaseMessage[]): SerializedMessage[] {
    if (!messages || !Array.isArray(messages)) return [];

    return messages.map(msg => this.serializeMessage(msg)).filter((msg): msg is SerializedMessage => msg !== null);
  }

  /**
   * Serialize a single LangChain message
   * @param msg - LangChain message object
   * @returns Serializable message object
   */
  static serializeMessage(msg: BaseMessage): SerializedMessage | null {
    if (!msg) return null;

    let type = msg.constructor.name;
    let msgType: string | null = null;

    // Use _getType() if available for more reliable type checking (handles minification)
    if (typeof msg._getType === 'function') {
      msgType = msg._getType();
      switch (msgType) {
        case 'human': type = 'HumanMessage'; break;
        case 'ai': type = 'AIMessage'; break;
        case 'system': type = 'SystemMessage'; break;
        case 'tool': type = 'ToolMessage'; break;
        case 'function': type = 'FunctionMessage'; break;
        case 'generic': type = 'ChatMessage'; break;
      }
    }

    const serialized: SerializedMessage = {
      type,
      content: msg.content || "",
      // Preserve all additional data
      additional_kwargs: msg.additional_kwargs || {},
      response_metadata: (msg as any).response_metadata || undefined
    };

    // Handle AIMessage specific fields (use _getType() result to handle minification)
    if (msgType === 'ai' || msg.constructor.name === 'AIMessage') {
      const aiMsg = msg as AIMessage;
      if (aiMsg.tool_calls && aiMsg.tool_calls.length > 0) {
        serialized.tool_calls = aiMsg.tool_calls;
      }
      if (aiMsg.usage_metadata) {
        serialized.usage_metadata = aiMsg.usage_metadata;
      }
    }

    // Handle ToolMessage specific fields (use _getType() result to handle minification)
    if (msgType === 'tool' || msg.constructor.name === 'ToolMessage') {
      const toolMsg = msg as ToolMessage;
      serialized.tool_call_id = toolMsg.tool_call_id;
      serialized.name = toolMsg.name;
    }

    return serialized;
  }

  /**
   * Deserialize state from storage
   * @param serialized - Serialized state object
   * @returns ReactGraph state object
   */
  static deserializeState(serialized: SerializedState | null): ReactGraphState | null {
    if (!serialized) return null;

    return {
      messages: serialized.messages ? this.deserializeMessages(serialized.messages) : [],
      allMessages: serialized.allMessages ? this.deserializeMessages(serialized.allMessages) : [],
      iterations: serialized.iterations || 0,
      maxIterations: serialized.maxIterations || 512,
      reflectionFeedback: serialized.reflectionFeedback || { feedback: "", completed: false },
      captchaBlocked: serialized.captchaBlocked || false,
      currentReasoning: serialized.currentReasoning || null,
      tokenUsage: serialized.tokenUsage || {
        input_tokens: 0,
        output_tokens: 0,
        total_tokens: 0
      },
      toolResultCache: serialized.toolResultCache || {},
      plan: serialized.plan || [],
      previousPageState: null
    };
  }

  /**
   * Deserialize array of messages
   * @param serialized - Array of serialized message objects
   * @returns Array of LangChain message objects
   */
  static deserializeMessages(serialized: SerializedMessage[]): BaseMessage[] {
    return deserializeMessages(serialized);
  }

  /**
   * Deserialize a single message
   * @param msg - Serialized message object
   * @returns LangChain message object
   */
  static deserializeMessage(msg: SerializedMessage): BaseMessage | null {
    return deserializeMessage(msg);
  }

  /**
   * Validate if a serialized state is still valid
   * @param serialized - Serialized state object
   * @param maxAge - Maximum age in milliseconds (default: 24 hours)
   * @returns Whether the state is still valid
   */
  static isStateValid(serialized: SerializedState | null, maxAge: number = 24 * 60 * 60 * 1000): boolean {
    if (!serialized) return false;
    if (!serialized.lastUpdated) return false;

    const age = Date.now() - serialized.lastUpdated;
    return age < maxAge;
  }

  /**
   * Compress state for storage if needed
   * @param state - State object to compress
   * @returns Compressed state (currently just returns state as-is)
   */
  static compressState(state: SerializedState): SerializedState {
    // TODO: Implement compression if storage becomes an issue
    // Could use LZ-string or similar compression library
    return state;
  }

  /**
   * Decompress state from storage
   * @param compressed - Compressed state object
   * @returns Decompressed state
   */
  static decompressState(compressed: SerializedState): SerializedState {
    // TODO: Implement decompression if compression is added
    return compressed;
  }

  /**
   * Workflow interface: Serialize state to string
   * @param state - The ReactGraph state object
   * @returns JSON string representation of the serialized state
   */
  static serialize(state: ReactGraphState | null): string {
    const serialized = this.serializeState(state);
    return JSON.stringify(serialized);
  }

  /**
   * Workflow interface: Deserialize state from string
   * @param data - JSON string or base64 encoded string
   * @returns Deserialized ReactGraph state object
   */
  static deserialize(data: string | null): ReactGraphState | null {
    if (!data) return null;

    try {
      // Try to parse as JSON string
      const serialized = JSON.parse(data) as SerializedState;
      return this.deserializeState(serialized);
    } catch (error) {
      // If JSON parse fails, try base64 decode
      try {
        const decoded = atob(data);
        const serialized = JSON.parse(decoded) as SerializedState;
        return this.deserializeState(serialized);
      } catch (decodeError) {
        console.error('Failed to deserialize state:', error, decodeError);
        return null;
      }
    }
  }

  /**
   * Serialize state to base64 encoded string
   * @param state - The ReactGraph state object
   * @returns Base64 encoded string representation
   */
  static serializeToBase64(state: ReactGraphState | null): string {
    const jsonString = this.serialize(state);
    return btoa(jsonString);
  }

  /**
   * Deserialize state from base64 encoded string
   * @param base64Data - Base64 encoded string
   * @returns Deserialized ReactGraph state object
   */
  static deserializeFromBase64(base64Data: string): ReactGraphState | null {
    try {
      const jsonString = atob(base64Data);
      return this.deserialize(jsonString);
    } catch (error) {
      console.error('Failed to deserialize from base64:', error);
      return null;
    }
  }
}