
import { StateGraph, Annotation } from "@langchain/langgraph/web";
import { BaseMessage, HumanMessage, AIMessage, SystemMessage, ToolMessage } from "@langchain/core/messages";
import { ChatPromptTemplate, MessagesPlaceholder } from "@langchain/core/prompts";
import { DynamicStructuredTool } from "@langchain/core/tools";
import { z } from "zod";
// @ts-expect-error - JS module without types
import { EVENT_TYPES } from "./event-types.js";
import { process as processCompaction, removeOrphanedToolMessages } from "./improvements/compaction.js";
// @ts-expect-error - JS module without types
import { CaptchaDetector } from "./improvements/captcha-detector.js";
// @ts-expect-error - JS module without types
import { toYAML, formatToolCallYAML } from "./yaml-serializer.js";
import { StateSerializer } from "./StateSerializer.js";
// @ts-expect-error - JS module without types
import { getPageState } from "./pageState.js";
import { SkillService } from "../skills/SkillService.js";

const Nodes = {
  ASSISTANT: "assistant",
  TOOLS: "tools",
  REFLECTION: "reflection",
  END: "__end__"
} as const;

const DEFAULT_MAX_ITERATIONS = 512; // Increased from 100 to support complex multi-site research tasks

// Tools that modify page state and require fresh page content extraction
const PAGE_MODIFYING_TOOLS = new Set([
  'navigate_to_url',
  'navigate',  // back/forward/refresh
  'switch_tab',
  'create_new_tab',
  'click_by_index',
  'fill_by_index',
  'scroll_page',
  'go_back',
  'go_forward',
  'close_tab',
  'submit_form'
]);

// Safe error message extraction - handles undefined, null, and non-Error objects
function getErrorMessage(error: unknown): string {
  if (error === undefined || error === null) {
    return 'Unknown error (undefined)';
  }
  if (error instanceof Error) {
    return error.message;
  }
  if (typeof error === 'string') {
    return error;
  }
  if (typeof error === 'object' && 'message' in error && typeof (error as any).message === 'string') {
    return (error as any).message;
  }
  return String(error);
}

// Simple hash function for content comparison (djb2 algorithm)
function simpleHash(str: string): string {
  let hash = 5381;
  for (let i = 0; i < str.length; i++) {
    hash = ((hash << 5) + hash) + str.charCodeAt(i);
  }
  return hash.toString(36);
}

/**
 * Extract tool execution history from messages for reflection context
 * Only processes ToolMessage objects which contain actual execution results.
 * @param messages - Full message history  
 * @returns Formatted string summary of tool executions
 */
function extractToolHistorySummary(messages: any[]): string {
  const summary: string[] = [];
  
  for (const msg of messages) {
    // Only process ToolMessage objects (contain actual execution results)
    if (msg?._getType?.() === 'tool' && msg?.name) {
      const content = typeof msg.content === 'string' ? msg.content : JSON.stringify(msg.content);
      const contentLower = content.toLowerCase();
      
      // Determine if this was an error/failure
      const failed = contentLower.includes('error:') || 
                     contentLower.includes('failed:') ||
                     contentLower.includes('tool execution failed');
      
      const status = failed ? 'FAILED' : 'SUCCESS';
      const errorDetail = failed ? ` - ${content.slice(0, 80)}` : '';
      summary.push(`- ${msg.name}: ${status}${errorDetail}`);
    }
  }
  
  return summary.length > 0 ? summary.join('\n') : 'No tools were executed.';
}

/**
 * Check if recent tool executions include any page-modifying tools
 * @param messages - Message history to check
 * @returns true if any recent tool execution modified page state
 */
function hasRecentPageModifyingTool(messages: any[]): boolean {
  // Find the last AIMessage with tool_calls - tools after this are from the most recent execution
  let lastAIMessageIndex = -1;
  for (let i = messages.length - 1; i >= 0; i--) {
    const msg = messages[i];
    if (msg?._getType?.() === 'ai' && msg?.tool_calls?.length > 0) {
      lastAIMessageIndex = i;
      break;
    }
  }

  // If no AI message with tool_calls found, no tools were executed
  if (lastAIMessageIndex === -1) {
    return false;
  }

  // Check only ToolMessages AFTER the last AIMessage with tool_calls
  const detectedTools: string[] = [];
  for (let i = lastAIMessageIndex + 1; i < messages.length; i++) {
    const msg = messages[i];

    // ToolMessage has a 'name' property with the tool name
    if (msg?.name) {
      detectedTools.push(msg.name);
      if (PAGE_MODIFYING_TOOLS.has(msg.name)) {
        return true;
      }
    }
    // Also check tool_call_id presence (indicates it's a ToolMessage)
    if (msg?.tool_call_id && msg?.content) {
      // Extract tool name from content if present (format: "Tool: toolname\n...")
      const match = msg.content.match(/^Tool:\s*(\w+)/);
      if (match) {
        detectedTools.push(match[1]);
        if (PAGE_MODIFYING_TOOLS.has(match[1])) {
          return true;
        }
      }
    }
  }

  return false;
}

// Task status types for plan tracking (Claude Code-style)
type TaskStatus = 'pending' | 'in_progress' | 'completed';

interface TodoItem {
  id: string; // Add required 'id' to interface
  content: string;      // Brief description of the task
  status: TaskStatus;
  priority: string; // Add required 'priority'
  activeForm?: string;   // Optional present continuous form
}

// Simple state structure - just messages like MessagesState
const MessagesState = Annotation.Root({
  // Core messaging state - compressed for LLM efficiency
  messages: Annotation({
    reducer: (x: any, y: any) => y,  // Replace with full messages array (handles compression)
    default: () => [],
  }),

  // Full uncompressed message history for reflection and AI_UPDATE events
  allMessages: Annotation({
    reducer: (x: any[], y: any[]) => y,
    default: () => [],
  }),

  // Simple iteration counter
  iterations: Annotation({
    reducer: (x: number, y: number) => y,
    default: () => 0,
  }),

  // Maximum iterations allowed
  maxIterations: Annotation({
    reducer: (x: number, y: number) => y,
    default: () => DEFAULT_MAX_ITERATIONS,
  }),

  // Whether the task is complete
  reflectionFeedback: Annotation({
    reducer: (x: { feedback: string; completed: boolean }, y: { feedback: string; completed: boolean }) => y,
    default: () => ({ feedback: "", completed: false }),
  }),


  // Token usage tracking - cumulative across all LLM calls
  tokenUsage: Annotation({
    reducer: (x: { input_tokens: number; output_tokens: number; total_tokens: number }, y: { input_tokens: number; output_tokens: number; total_tokens: number }) => ({
      input_tokens: (x?.input_tokens || 0) + (y?.input_tokens || 0),
      output_tokens: (x?.output_tokens || 0) + (y?.output_tokens || 0),
      total_tokens: (x?.total_tokens || 0) + (y?.total_tokens || 0),
    }),
    default: () => ({ input_tokens: 0, output_tokens: 0, total_tokens: 0 }),
  }),


  // Task decomposition plan (OpenCode-style todowrite)
  plan: Annotation({
    reducer: (x: TodoItem[], y: TodoItem[]) => y, // Replace with new plan
    default: () => [] as TodoItem[],
  }),

  // Tool result cache to prevent repeated identical tool calls
  toolResultCache: Annotation({
    reducer: (x: Record<string, any>, y: Record<string, any>) => ({ ...x, ...y }),
    default: () => ({} as Record<string, any>),
  }),

  // Previous page state for deduplication (URL + content hash)
  previousPageState: Annotation({
    reducer: (x: { url: string; contentHash: string } | null, y: { url: string; contentHash: string } | null) => y,
    default: () => null as { url: string; contentHash: string } | null,
  }),

  // Skills loaded during this session (for analytics and testing)
  skillsCalled: Annotation({
    reducer: (x: string[], y: string[]) => [...new Set([...x, ...y])], // Merge and dedupe
    default: () => [] as string[],
  }),
});

// ReAct prompt - optimized for minimal tokens
const REACT_PROMPT = ChatPromptTemplate.fromMessages([
  [
    "system",
    `Browser automation agent using ReAct. ACT IMMEDIATELY - never ask for clarification.

## ReAct Cycle
1. THINK: Analyze page content, decide next action
2. ACT: Call tool or provide final answer
3. OBSERVE: Check result, repeat until done

## Tool Selection
- **navigate_to_url**: Sites requiring auth/interaction (Facebook, Amazon, X, marketplaces). Opens browser for click/fill/scroll.
- **web_fetch**: Public read-only sites (Wikipedia, news, docs). Fast, no interaction.

## Rules
1. ACT FIRST. "find X on Y" → navigate to Y immediately
2. Use defaults. No preferences given → proceed anyway
3. Answer visible → respond immediately, no tools
4. Handle popups/modals first
5. Click highest-scored elements
6. NEVER call web_fetch on pages already open in browser tabs
7. Text wrapped in **>>text<<** is USER-SELECTED - prioritize actions on selected content

## Task Planning (TodoWrite)
Use TodoWrite for tasks with 3+ steps or multiple items. Skip for simple single actions.

**When to use**: Multi-site research, compare items, sequential workflows, user lists multiple tasks
**Skip for**: Single navigation, one search, answering visible question

**Rules**:
1. Create plan BEFORE first action (list subtasks as "pending")
2. Only ONE task "in_progress" at a time
3. Mark "completed" IMMEDIATELY when done (never batch)
4. Be specific: "Search Amazon for X" not "Find product"

## Parallel Execution (MANDATORY for multi-item tasks)
**WARNING: Sequential create_new_tab loops WILL trigger safety limits. USE PARALLEL PATTERN.**

When task requires checking/extracting from multiple items (e.g., "check 10 listings", "extract from 5 tabs"):
1. **First**: Open ALL target tabs in ONE batch:
   - Call navigate_to_url(url1, newTab: true) → get tabId: 111
   - Call navigate_to_url(url2, newTab: true) → get tabId: 222
   - (repeat until all tabs open)
2. **Then**: Use SINGLE parallel() call with ALL subagents:
   {{ "tool": "parallel", "args": {{ "tool_uses": [
     {{ "name": "subagent", "args": {{ "systemPrompt": "Extract X from current page", "tabId": 111 }} }},
     {{ "name": "subagent", "args": {{ "systemPrompt": "Extract X from current page", "tabId": 222 }} }}
   ] }} }}
3. **KEY**: Subagent with tabId works on that EXISTING tab - NO new navigation needed

**ANTI-PATTERN (triggers doom loop)**: Opening tabs one-by-one and processing sequentially

When you encounter a login form: STOP, tell user to enter credentials in the browser form, add <suggestion>Continue</suggestion> at end of message. Resume after user clicks Continue.`
  ],
  new MessagesPlaceholder("messages"),
]);

interface AgentOptions {
  onEvent?: (type: string, event: any) => void;
  isAborted?: () => string | null;
  initialTabId?: number | null; // Set initial tab for subagent isolation
  getPageContentTool?: any;
  getTabs?: () => Promise<any[]>;
  takeScreenshotCallback?: (options: any) => Promise<string>;
  takeScreenshot?: boolean;
  maxTokens?: number;
  compressionThreshold?: number;
  modelName?: string;
  sessionId?: string; // Session ID for cross-session context tools
}

interface AgentParams {
  llm: any;
  langchainTools: any[];
  options: AgentOptions;
  memory?: any;
}

interface PageState {
  indexedNodesMarkdown?: string;
  pageUrl?: string;
  pageTitle?: string;
  tabId?: number;
  tabs: any[];
  screenshot?: string;
  error?: string;
  markdown?: string;
}

interface State {
  messages: any[];
  allMessages: any[];
  iterations: number;
  maxIterations: number;
  reflectionFeedback: {
    feedback: string;
    completed: boolean;
  };
  tokenUsage: {
    input_tokens: number;
    output_tokens: number;
    total_tokens: number;
  };
  plan: TodoItem[];
  toolResultCache: Record<string, any>;
  previousPageState: {
    url: string;
    contentHash: string;
  } | null;
  skillsCalled: string[];
}

export class ReactAgent {
  private llm: any;
  private langchainTools: any[];
  private onEvent: (type: string, event: any) => void;
  private isAborted: () => string | null;
  private getPageContentTool: any;
  private getTabs: () => Promise<any[]>;
  private takeScreenshotCallback: (options: any) => Promise<string>;
  private captchaDetector: any;
  private takeScreenshot: boolean;
  private assistantChain: any;
  private state: State = {} as State;
  private workflow: any;
  private mode: string;
  private lastVisionUsed = false;
  private currentTabId: number | null = null; // Virtual tab focus for subagent isolation
  private todoWriteTool: any;
  private skillTool: any;
  private skillDescriptions: string = ''; // Skill descriptions for system prompt injection
  private modelName: string;

  constructor({ llm, langchainTools, options, memory, mode = 'agent' }: AgentParams & { mode?: string }) {
    this.llm = llm;
    this.langchainTools = langchainTools;
    this.mode = mode;
    this.modelName = options?.modelName || '';
    // Set up unified event emitter
    this.onEvent = options?.onEvent || (() => { });
    this.isAborted = options?.isAborted || (() => null);
    this.getPageContentTool = options?.getPageContentTool;
    this.getTabs = options?.getTabs || (() => Promise.resolve([]));
    this.takeScreenshotCallback = options?.takeScreenshotCallback || (() => Promise.resolve(""));
    this.currentTabId = options?.initialTabId || null; // Set initial tab for subagent isolation

    // Create the todowrite tool for OpenCode-style task decomposition
    this.todoWriteTool = this.createTodoWriteTool();
    // Create the skill tool for loading skill content on-demand
    this.skillTool = this.createSkillTool();
    // Add internal tools to langchainTools
    this.langchainTools = [...(langchainTools || []), this.todoWriteTool, this.skillTool];

    // Pass parent config to SubAgentTool so it can create child ReactAgents
    for (const tool of this.langchainTools) {
      const browserTool = (tool as any)._browserTool;
      if (browserTool?.inherit && typeof browserTool.inherit === 'function') {
        browserTool.inherit({
          llm: this.llm,
          langchainTools: this.langchainTools,
          options,
          modelName: this.modelName,
          sessionId: options?.sessionId  // Pass sessionId for cross-session context tools
        });
      }
    }

    this.captchaDetector = new CaptchaDetector(); // CAPTCHA/blocking detection

    this.takeScreenshot = options?.takeScreenshot !== false; // Default to true
    // Set up the chain with tools - only bind tools to LLM in agent mode
    const toolsToBind = this.mode === 'agent' ? (this.langchainTools || []) : [];
    this.assistantChain = REACT_PROMPT.pipe(
      this.llm.bindTools(toolsToBind)
    );

    this.workflow = this.buildGraph();
  }

  /**
   * Create the TodoWrite tool for task decomposition (Claude Code-style)
   * This tool allows the agent to track its progress through complex tasks
   */
   private createTodoWriteTool(): DynamicStructuredTool {
    const TodoItemSchema = z.object({
      content: z.string().describe("Brief description of the task"),
      status: z.enum(["pending", "in_progress", "completed", "cancelled"]).describe("Current status of the task"),
      id: z.string().describe("Unique identifier for the todo item"),
      priority: z.enum(["high", "medium", "low"]).describe("Priority level of the task"),
      activeForm: z.string().optional().nullable().describe("Present continuous form for display (e.g., 'Searching for data')")
    });

    const TodoListSchema = z.object({
      todos: z.array(TodoItemSchema).describe("The updated todo list")
    });

    // Cast to any to avoid "Type instantiation is excessively deep" errors with LangChain's complex types
    // The schema is correct at runtime
    const toolConfig: any = {
      name: "TodoWrite",
      description: "Update your todo list with tasks. Use this to decompose complex tasks into smaller steps and track progress. Mark tasks as 'pending', 'in_progress', or 'completed'.",
      schema: TodoListSchema,
      func: async ({ todos }: { todos: any[] }) => {
        // Determine what changed compared to previous state
        const previousTodos = this.state.plan || [];
        const changes: string[] = [];

        for (const todo of todos) {
          const prev = previousTodos.find(p => p.content === todo.content);
          if (!prev) {
            changes.push(`+ [${todo.status}] ${todo.content}`);
          } else if (prev.status !== todo.status) {
            changes.push(`~ ${todo.content}: ${prev.status} -> ${todo.status}`);
          }
        }

        // Update the plan in state
        this.state.plan = todos;

        // Emit PLAN_UPDATE event for UI
        this.onEvent(EVENT_TYPES.PLAN_UPDATE, {
          plan: todos,
          changes,
          iteration: this.state.iterations,
          timestamp: Date.now()
        });

        // Build Claude Code-style output showing what changed (like opencode)
        const activeTodos = todos.filter(t => t.status !== 'completed');
        const title = `${activeTodos.length} todos`;
        const output = changes.length > 0
          ? changes.join('\n')
          : todos.map(t => `[${t.status}] ${t.content}`).join('\n');

        return JSON.stringify({ title, output, todos });
      }
    };

    return new DynamicStructuredTool(toolConfig);
  }

  /**
   * Create the skill tool for loading skill content on-demand
   * This tool allows the agent to load full skill instructions when needed
   */
  private createSkillTool(): DynamicStructuredTool {
    const SkillSchema = z.object({
      name: z.string().describe("The skill identifier from available_skills (e.g., 'web-search', 'form-fill')")
    });

    const toolConfig: any = {
      name: "skill",
      description: "Load a skill to get detailed instructions for a specific task. Skills provide specialized knowledge and step-by-step guidance. Use this when a task matches an available skill's description.",
      schema: SkillSchema,
      func: async ({ name }: { name: string }) => {
        try {
          const skillService = SkillService.getInstance();
          const skill = await skillService.getParsedByName(name);
          
          if (!skill) {
            return `Error: Skill "${name}" not found. Use one of the available skills from the system prompt.`;
          }
          
          if (!skill.enabled) {
            return `Error: Skill "${name}" is disabled.`;
          }
          
          // Track skill usage in state
          if (!this.state.skillsCalled) {
            this.state.skillsCalled = [];
          }
          if (!this.state.skillsCalled.includes(name)) {
            this.state.skillsCalled.push(name);
          }
          
          // Emit event for UI visibility and tracking
          this.onEvent(EVENT_TYPES.TOOL_CALL, {
            tool: 'skill',
            input: { name },
            output: { name: skill.name, description: skill.description },
            skillsCalled: this.state.skillsCalled,
            timestamp: Date.now()
          });
          
          // Return plain markdown body as per OpenCode specification
          return skill.body;
        } catch (error: unknown) {
          return `Error: Failed to load skill: ${getErrorMessage(error)}`;
        }
      }
    };

    return new DynamicStructuredTool(toolConfig);
  }

  /**
   * Load skill descriptions for system prompt injection
   * Called before agent execution to get available skills
   */
  async loadSkillDescriptions(): Promise<void> {
    try {
      const skillService = SkillService.getInstance();
      const summaries = await skillService.getSkillSummaries();
      
      if (summaries.length > 0) {
        const skillXml = summaries.map(s => 
          `  <skill>\n    <name>${s.name}</name>\n    <description>${s.description}</description>\n  </skill>`
        ).join('\n');
        this.skillDescriptions = `\n\nLoad a skill to get detailed instructions for a specific task. Skills provide specialized knowledge and step-by-step guidance. Use this when a task matches an available skill's description.\n<available_skills>\n${skillXml}\n</available_skills>`;
      } else {
        this.skillDescriptions = '';
      }
    } catch (error) {
      console.warn('Failed to load skill descriptions:', error);
      this.skillDescriptions = '';
    }
  }

  private emitSystemEvent(app_type: string, content: string, state: State, additional_kwargs: any = {}): SystemMessage {
    const message = new SystemMessage({
      content,
      additional_kwargs: {
        app_type,
        iteration: state.iterations + 1,
        modelName: this.modelName,
        visionUsed: this.lastVisionUsed,
        ...additional_kwargs
      }
    });

    // Ephemeral messages are appended to the event's messages[] for display
    // but NOT persisted to state.allMessages (which is used for storage/context)
    const currentMessages = state.allMessages || [];
    const displayMessages = [...currentMessages, message];

    this.onEvent(EVENT_TYPES.AI_UPDATE, {
      ...state,
      messages: StateSerializer.serializeMessages(displayMessages)
    });

    return message;
  }

  /**
   * Assistant node - performs reasoning and decides on actions
   */
  async assistantNode(state: State): Promise<Partial<State>> {
    let messages = state.messages;
    let newMessages: any[] = [];


    // Check iteration limits
    if (state.iterations >= state.maxIterations) {
      const finalMessage = new AIMessage({
        content: `Reached maximum iterations (${state.maxIterations}). Task may not be fully complete.`,
        additional_kwargs: { app_type: 'answer' },
      });
      return {
        messages: [...messages, finalMessage],
        allMessages: [...(state.allMessages || []), finalMessage],
        reflectionFeedback: { feedback: "Max iterations reached", completed: true }
      };
    }

    // Emit thinking event immediately so UI shows indicator while we gather page content
    // visionUsed will be updated after getPageState completes
    this.emitSystemEvent('thinking', 'is thinking', state, { visionUsed: this.lastVisionUsed });

    // Get page state with optional screenshot
    const pageState: PageState = await getPageState({
      tabId: this.currentTabId,
      getPageContentTool: this.getPageContentTool,
      getTabs: this.getTabs,
      takeScreenshotCallback: this.takeScreenshotCallback,
      supportsVision: true, // ai_agent.js already decided if screenshots should be taken
      screenshotOptions: {},
      includeIndexedNodesMarkdown: true,
      includeMarkdown: true,
      includeIndexedNodesHtml: false,
      includeScreenshot: this.takeScreenshot
    });

    const visionUsed = !!pageState.screenshot;
    this.lastVisionUsed = visionUsed;

    // Emit page content captured event for debug tracing
    if (pageState.indexedNodesMarkdown) {
      this.onEvent(EVENT_TYPES.PAGE_CONTENT_CAPTURED, {
        content: pageState.indexedNodesMarkdown,
        url: pageState.pageUrl,
        title: pageState.pageTitle,
        iteration: state.iterations + 1,
        messages: state.messages
      });
    }

    // Check for CAPTCHA blocking
    if (pageState.indexedNodesMarkdown) {
      const detection = await this.captchaDetector.detect(pageState.indexedNodesMarkdown, pageState.pageUrl, pageState.pageTitle);

      if (detection.hasCaptcha) {
        console.log(`🤖 CAPTCHA Detected ${detection.type} (confidence: ${detection.confidence}%), Indicators:`, detection.indicators.slice(0, 3).join(', '));
        const captchaMessage = new AIMessage({
          content: "Task failed due to CAPTCHA/blocking. Manual intervention required or use alternative approach.",
          additional_kwargs: { app_type: 'answer' },
        });

        return {
          messages: [...messages, captchaMessage],
          allMessages: [...(state.allMessages || []), captchaMessage],
          reflectionFeedback: { feedback: "CAPTCHA detected", completed: true }
        };
      }
    }

    let messageText = '';

    // Add reflection feedback as a proper message in the conversation (not in page state)
    // This ensures the LLM sees it as guidance from the reflection step
    if (state.reflectionFeedback && state.reflectionFeedback.feedback && !state.reflectionFeedback.completed) {
      const feedbackMessage = new SystemMessage({
        content: `[Reflection Feedback] Task not yet complete. ${state.reflectionFeedback.feedback}`,
        additional_kwargs: { app_type: 'reflection_feedback' }
      });
      messages = [...messages, feedbackMessage];
      newMessages.push(feedbackMessage);
      console.log(`🤖 Added reflection feedback as message: ${state.reflectionFeedback.feedback.substring(0, 100)}...`);
    }

    // Add todo status if there are any todos
    if (state.plan && state.plan.length > 0) {
      const completed = state.plan.filter(t => t.status === 'completed');

      messageText += `# Todo (${completed.length}/${state.plan.length} completed)\n`;
      let i = 1;
      for (const todo of state.plan) {
        messageText += `${i}. ${todo.content}:${todo.status}\n`;
        i += 1;
      }
    }

    messageText += '# Current browser state:\n';

    // Add all browser tabs
    if (pageState.tabs.length > 0) {
      messageText += `\nAll browser tabs (${pageState.tabs.length} total):\n`;
      pageState.tabs.forEach((tab, index) => {
        const activeMarker = tab.active ? ' [ACTIVE]' : '';
        messageText += `  ${index + 1}. Tab ${tab.tabId}${activeMarker}: ${tab.title}\n     URL: ${tab.url}\n`;
      });
      messageText += '\n';
    }

    // Add current tab info
    messageText += `Current Date/Time: ${(new Date()).toLocaleString()}\n`;
    messageText += `Currently opened active tab page\n`;
    messageText += `Tab ID: ${pageState.tabId || 'Unknown'}\n`;
    messageText += `Title: ${pageState.pageTitle}\n`;
    messageText += `Url: ${pageState.pageUrl}\n`;

    // Add page content
    if (!pageState.error) {
      messageText += `\`\`\`markdown\n`;
      messageText += pageState.indexedNodesMarkdown;
      messageText += `\n\`\`\``;
    } else {
      messageText += `\nError: ${pageState.error}`;
    }

    messageText += `\nOutput
1) Final answer if you gathered enough knowledge.

OR 

2) If not enough knowledge:
  a) The reasoning and the next steps and tool calls
  b) Extract important knowledge from the page content in order to complete the task
`;

    // https://platform.openai.com/docs/guides/images-vision?format=base64-encoded
    // Build message content with optional screenshot
    let messageContent: any[] = [{ type: "text", text: messageText }];
    if (!!pageState.screenshot) {
      messageContent.push(
        { type: "image_url", image_url: { url: pageState.screenshot } }
      );
    }

    // Create the new page content message
    const newPageMessage = new HumanMessage({
      content: messageContent,
      additional_kwargs: { isPageContent: true }
    });

    // Helper to detect token limit errors
    const isTokenLimitError = (error: unknown): boolean => {
      let errorStr = String(error).toLowerCase();
      
      // Also check message property if available (for object-like errors)
      if (typeof error === 'object' && error !== null) {
        const msg = (error as any).message;
        if (typeof msg === 'string') {
          errorStr += ' ' + msg.toLowerCase();
        }
        const code = (error as any).code;
        if (typeof code === 'string') {
          errorStr += ' ' + code.toLowerCase();
        }
      }

      return errorStr.includes('context length') ||
        errorStr.includes('context_length') ||
        errorStr.includes('token limit') ||
        errorStr.includes('maximum context') ||
        errorStr.includes('too many tokens') ||
        errorStr.includes('exceeds the model') ||
        errorStr.includes('exceeds the limit') ||
        errorStr.includes('prompt token count') ||
        errorStr.includes('request too large');
    };

    let response;
    const llmStartTime = Date.now();
    console.log(`🤖 LLM invoke`, messageText);
    try {
      // Validate message coherence before LLM call - remove orphaned ToolMessages
      // This prevents "messages with role 'tool' must be a response to a preceding message with 'tool_calls'" errors
      const validatedMessages = removeOrphanedToolMessages(messages);
      
      // Timeout is handled by LLMWithRetry.wrapBoundLLM (90s max via maxOverallTimeout)
      // The bindTools() call returns a wrapped LLM that enforces timeout on invoke
      response = await this.assistantChain.invoke({
        messages: [...validatedMessages, newPageMessage]
      });
      console.log(`🤖 LLM invoke completed in ${Date.now() - llmStartTime}ms, finish_reason: ${response.response_metadata?.finish_reason}`);
    } catch (error) {
      // Serialize error properly - DOMException and other errors don't log well directly
      const errorDetails = {
        message: error instanceof Error ? error.message : String(error),
        name: error instanceof Error ? error.name : 'Unknown',
        stack: error instanceof Error ? error.stack?.split('\n').slice(0, 3).join('\n') : undefined
      };
      console.error(`🤖 LLM invoke FAILED after ${Date.now() - llmStartTime}ms:`, JSON.stringify(errorDetails));
      
      const isTokenError = isTokenLimitError(error);
      if (!isTokenError) {
        console.log(`ℹ️ Error was NOT detected as token limit. String(error): "${String(error)}", message: "${(error as any)?.message}"`);
      }

      if (isTokenError && messages.length > 0) {
        console.log(`Error: ${String(error).slice(0, 200)}`);
        console.log(`🗜️ Token limit error detected, compressing context and retrying...`);

        // Check if we are already in a retry loop due to insufficient compaction
        if (state.messages.some(m => m.additional_kwargs?.isCompactionContinuation)) {
            console.error('❌ Compaction failed to reduce context enough or recursion detected. Aborting to avoid infinite loop.');
            throw error;
        }

        this.emitSystemEvent('compressing', 'Context too large, compacting...', state, {
          error: String(error).slice(0, 100),
          messageCount: messages.length,
          visionUsed
        });

        // Process compaction with LLM summarization (like OpenCode's SessionCompaction.process())
        // This will: 1) prune old tool outputs, 2) summarize with LLM
        const compactedMessages = await processCompaction({
          messages,
          llm: this.llm
        });
        
        // Add continuation message (like OpenCode does after compaction)
        messages = [
          ...compactedMessages,
          new HumanMessage({
            content: "Continue with the task. Here is the original request for reference:\n\n" + 
              (state.allMessages?.find((m: any) => m._getType?.() === 'human')?.content || 'Complete the user task'),
            additional_kwargs: { isCompactionContinuation: true }
          })
        ];
        
        console.log(`🗜️ Retrying LLM call with compacted context (${messages.length} messages)`);
        this.emitSystemEvent('thinking', 'is thinking (retry)', state, { visionUsed });

        // Validate message coherence before retry - compaction should have cleaned, but double-check
        const validatedRetryMessages = removeOrphanedToolMessages(messages);
        response = await this.assistantChain.invoke({
          messages: [...validatedRetryMessages, newPageMessage]
        });
      } else {
        // Not a token limit error, re-throw
        throw error;
      }
    }

    // Extract token usage from response
    const tokenUsage = {
      input_tokens: response.usage_metadata?.input_tokens || 0,
      output_tokens: response.usage_metadata?.output_tokens || 0,
      total_tokens: response.usage_metadata?.total_tokens || 0
    };
    console.log(JSON.stringify(tokenUsage, null, 2));


    response.additional_kwargs.app_type = 'thought';

    // Track current page state for deduplication
    let newPreviousPageState: { url: string; contentHash: string } | null = null;

    if (pageState.pageUrl && !pageState.error && pageState.indexedNodesMarkdown) {
      const currentContentHash = simpleHash(pageState.indexedNodesMarkdown);
      const prev = state.previousPageState;

      console.log(`📄 Page hash: prev=${prev?.contentHash || 'null'}, current=${currentContentHash}, iteration=${state.iterations}`);

      // Include page content on first iteration OR after page-modifying tools
      if (state.iterations === 0 || hasRecentPageModifyingTool(messages)) {
        newPreviousPageState = { url: pageState.pageUrl, contentHash: currentContentHash };

        // Only add if content actually changed
        if (!prev || prev.url !== pageState.pageUrl || prev.contentHash !== currentContentHash) {
          const pageStateSummary = new SystemMessage({
            content: `${pageState.pageUrl}\n\n${pageState.indexedNodesMarkdown}`,
            additional_kwargs: { app_type: 'page' }
          });
          console.log(`📄 Page content included (hash changed: ${prev?.contentHash} -> ${currentContentHash})`);
          newMessages.push(pageStateSummary);
        } else {
          console.log(`📄 Page content skipped (hash unchanged: ${currentContentHash})`);
        }
      } else {
        console.log(`📄 Page content skipped (no page-modifying tools)`);
        // Preserve previous state if we're skipping
        newPreviousPageState = prev;
      }
    }

    newMessages.push(response);

    return {
      messages: [...messages, ...newMessages],
      allMessages: [...(state.allMessages || []), ...newMessages],
      iterations: state.iterations + 1,
      tokenUsage,
      previousPageState: newPreviousPageState
    };

  }

  /**
   * Tools node - executes tool calls and returns results
   */
  async toolsNode(state: State): Promise<Partial<State>> {
    const messages = [...state.messages];  // Create a copy to avoid mutating state

    try {
      // Get the last message (should be from assistant with tool calls)
      const lastMessage = state.messages?.[state.messages.length - 1];

      // Defensive check: if lastMessage doesn't have tool_calls, find the last AIMessage that does
      let toolCalls = lastMessage?.tool_calls || [];

      if (!Array.isArray(toolCalls)) {
        toolCalls = [];
      }

      if (toolCalls.length === 0) {
        // Search backwards for an AIMessage with tool_calls
        for (let i = state.messages.length - 1; i >= 0; i--) {
          const msg = state.messages[i];
          if (msg?.tool_calls && Array.isArray(msg.tool_calls) && msg.tool_calls.length > 0) {
            toolCalls = msg.tool_calls;
            break;
          }
        }
      }

      if (toolCalls.length === 0) {
        return { messages };
      }

      // Expand parallel meta-tool into individual tool calls
      // This allows LLM to explicitly request parallel execution
      const expandedToolCalls: any[] = [];
      const parallelToolIds: string[] = []; // Track which tool_call_ids are from parallel expansion

      for (const toolCall of toolCalls) {
        const toolName = toolCall.name || toolCall.function?.name;

        if (toolName === 'parallel') {
          // Expand the meta-tool into individual tool calls
          console.log(`🔄 Expanding parallel meta-tool`);
          const args = typeof toolCall.args === 'string' ? JSON.parse(toolCall.args) :
            toolCall.args || JSON.parse(toolCall.function?.arguments || '{}');

          const toolUses = args.tool_uses || [];
          console.log(`🔄 Found ${toolUses.length} tool(s) to execute in parallel`);

          for (let i = 0; i < toolUses.length; i++) {
            const use = toolUses[i];
            const expandedId = `${toolCall.id}_parallel_${i}`;
            // Strip "functions." prefix if present (Azure OpenAI adds this)
            const cleanName = use.name.replace(/^functions\./, '');
            expandedToolCalls.push({
              id: expandedId,
              name: cleanName,
              args: use.args,
              // Mark as expanded from parallel
              _parallelParentId: toolCall.id
            });
            parallelToolIds.push(expandedId);
          }
        } else {
          // Regular tool call - keep as-is
          expandedToolCalls.push(toolCall);
        }
      }

      // Check cache for web_fetch calls to prevent repeated identical fetches
      const cachedResults: any[] = [];
      const uncachedToolCalls: any[] = [];

      for (const toolCall of expandedToolCalls) {
        const toolName = toolCall.name || toolCall.function?.name;
        if (toolName === 'web_fetch') {
          let args: any;
          try {
            args = typeof toolCall.args === 'string' ? JSON.parse(toolCall.args) : toolCall.args || JSON.parse(toolCall.function?.arguments || '{}');
          } catch (e) {
            args = {};
          }
          const cacheKey = `web_fetch:${args.url || ''}:${args.extract || ''}`;
          if (state.toolResultCache[cacheKey]) {
            console.log(`[WebFetch] Using cached result for ${cacheKey}`);
            const cached = state.toolResultCache[cacheKey];
            messages.push(new ToolMessage({
              content: cached.content,
              tool_call_id: toolCall.id,
              name: toolCall.name
            }));
            cachedResults.push({
              toolCall,
              toolName,
              success: true,
              args,
              result: cached.content,
              executionTime: 0
            });
          } else {
            uncachedToolCalls.push(toolCall);
          }
        } else {
          uncachedToolCalls.push(toolCall);
        }
      }

      // Execute uncached tool calls in PARALLEL using Promise.all()
      // Stagger starts by 200ms to avoid overwhelming LLM API TPM quota
      const STAGGER_DELAY_MS = 200;
      console.log(`🤖 Executing ${uncachedToolCalls.length} tool(s) in parallel (${cachedResults.length} cached)`);
      const parallelStartTime = Date.now();

      const toolResults = await Promise.all(
        uncachedToolCalls.map(async (toolCall, index) => {
          // Stagger tool starts to spread API load
          if (index > 0) {
            await new Promise(resolve => setTimeout(resolve, index * STAGGER_DELAY_MS));
          }
          const toolName = toolCall.name || toolCall.function?.name;
          const tool = this.langchainTools.find(t => t.name === toolName);

          if (!tool) {
            console.error(`🤖 Tool ${toolName} not found in available tools`);
            return {
              toolCall,
              toolName,
              success: false,
              error: `Tool ${toolName} not found`,
              executionTime: 0,
              args: null,
              result: null
            };
          }

          const startTime = Date.now();
          let args: any;

          try {
            // Parse arguments if they're a string
            args = typeof toolCall.args === 'string' ? JSON.parse(toolCall.args) :
              toolCall.args || JSON.parse(toolCall.function?.arguments || '{}');

            // Inject currentTabId for tools that need it if not specified
            // This enables tab isolation for subagents and ensures tools use the correct tab
            const toolsNeedingTabId = [
              'navigate_to_url',
              'click_by_index', 'fill_by_index', 'select_by_index',
              'interact_by_index', 'scroll', 'get_indexed_elements'
            ];
            if (toolsNeedingTabId.includes(toolName) && args.tabId === undefined && this.currentTabId !== null) {
              args.tabId = this.currentTabId;
              console.log(`🔧 Injecting tabId=${this.currentTabId} into ${toolName}`);
            }

            const toolResult = await tool.invoke(args);
            const executionTime = Date.now() - startTime;
            const toolOutput = typeof toolResult === 'string' ? toolResult : toYAML(toolResult);
            
            // Update currentTabId from tool results or args
            const newTabId = this.extractTabId(toolName, args, toolResult);
            if (newTabId !== null) {
              this.currentTabId = newTabId;
              console.log(`🔧 Updated currentTabId to ${this.currentTabId} (${toolName})`);
            }

            return {
              toolCall,
              toolName,
              success: true,
              args,
              result: toolOutput,
              rawResult: toolResult,
              executionTime
            };
          } catch (error) {
            const executionTime = Date.now() - startTime;
            const errorMessage = getErrorMessage(error);

            // Provide more helpful error messages for common issues
            let enhancedError = errorMessage;
            if (errorMessage.includes('did not match expected schema') || errorMessage.includes('validation')) {
              enhancedError = `Schema validation failed for tool "${toolName}". ` +
                `Received args: ${JSON.stringify(args, null, 2)}. ` +
                `Check that all required fields are provided with correct types. ` +
                `Original error: ${errorMessage}`;
              console.error(`🤖 Schema validation error for ${toolName}:`, {
                args,
                error: errorMessage,
                hint: 'Ensure systemPrompt is a string and tabId is a number or null'
              });
            } else {
              console.error(`🤖 Error executing tool ${toolName}:`, error);
            }

            return {
              toolCall,
              toolName,
              success: false,
              args,
              error: enhancedError,
              executionTime
            };
          }
        })
      );

      const parallelDuration = Date.now() - parallelStartTime;
      console.log(`🤖 All ${expandedToolCalls.length} tool(s) completed in ${parallelDuration}ms (parallel)`);

      // Group results by parallel parent ID for aggregation
      const parallelResults: Map<string, any[]> = new Map();
      const regularResults: any[] = [];

      for (const result of toolResults) {
        const parentId = result.toolCall._parallelParentId;
        if (parentId) {
          // This result is from a parallel-expanded tool
          if (!parallelResults.has(parentId)) {
            parallelResults.set(parentId, []);
          }
          parallelResults.get(parentId)!.push(result);
        } else {
          // Regular tool result
          regularResults.push(result);
        }
      }

      // Process regular tool results
      for (const result of regularResults) {
        if (result.success) {
          const toolExecutionLog = `Tool: ${result.toolName}
Arguments: ${JSON.stringify(result.args, null, 2)}
Execution Time: ${result.executionTime}ms
Result: ${result.result}`;

          messages.push(new ToolMessage({
            content: toolExecutionLog,
            tool_call_id: result.toolCall.id,
            name: result.toolCall.name
          }));

        } else {
          messages.push(new ToolMessage({
            content: result.error?.includes('not found')
              ? `Error: ${result.error}`
              : `Tool execution failed: ${result.error}. Reflect and retry.`,
            tool_call_id: result.toolCall.id,
            name: result.toolCall.name || result.toolName
          }));

          // Emit tool error event
          this.onEvent(EVENT_TYPES.TOOL_ERROR, {
            toolName: result.toolName,
            error: result.error,
            iteration: state.iterations,
            timestamp: Date.now()
          });
        }
      }

      // Cache successful web_fetch results
      for (const result of toolResults) {
        if (result.success && result.toolName === 'web_fetch') {
          const cacheKey = `web_fetch:${result.args.url}:${result.args.extract || ''}`;
          state.toolResultCache[cacheKey] = {
            content: result.result,
            timestamp: Date.now()
          };
        }
      }

      // Process aggregated parallel results - send ONE message per parallel call
      for (const [parentId, results] of parallelResults) {
        const successCount = results.filter(r => r.success).length;
        const totalTime = results.reduce((sum, r) => sum + r.executionTime, 0);

        // Format aggregated results
        const aggregatedContent = results.map((r, i) => {
          if (r.success) {
            return `[${i + 1}] ${r.toolName}(${JSON.stringify(r.args)})\n    Result: ${r.result}`;
          } else {
            return `[${i + 1}] ${r.toolName}(${JSON.stringify(r.args)})\n    ERROR: ${r.error}`;
          }
        }).join('\n\n');

        const summary = `Parallel execution completed: ${successCount}/${results.length} succeeded in ${parallelDuration}ms\n\n${aggregatedContent}`;

        messages.push(new ToolMessage({
          content: summary,
          tool_call_id: parentId,
          name: 'parallel'
        }));
      }

      // Doom loop detection: check for repetitive tool call patterns (like OpenCode)
      const doomLoopDetected = this.detectDoomLoop(messages);

      if (doomLoopDetected) {
        console.warn('⚠️ DOOM LOOP DETECTED - providing feedback to LLM');
        messages.push(new ToolMessage({
          content: `[SYSTEM] You are stuck in a navigation loop - repeating the same actions. STOP making tool calls. Based on the data you have already gathered, provide your FINAL ANSWER to the user's question NOW. Do not call any more tools.`,
          tool_call_id: toolCalls[toolCalls.length - 1]?.id || 'doom-loop',
          name: 'system'
        }));
      }


      // Calculate new tool messages AFTER any system messages are added
      const newToolMessages = messages.slice(state.messages.length);

      return {
        messages,
        allMessages: [...(state.allMessages || []), ...newToolMessages]
        // Note: Do NOT set reflectionFeedback here - let LLM naturally stop calling tools
      };
    } catch (error) {
      console.error('❌ ERROR in toolsNode:', error);
      console.error('Stack:', error instanceof Error ? error.stack : 'No stack available');
      console.error('State messages length:', state.messages?.length);
      throw error;
    }
  }

  /**
   * Reflection node - validates task completion before stopping
   * Includes retry logic for transient failures
   *
   */
  async reflectionNode(state: State): Promise<Partial<State>> {
    console.log(`🪞 [reflectionNode] START - iterations=${state.iterations}, messages=${state.messages?.length}, allMessages=${state.allMessages?.length}`);
    const messages = state.messages;
    const allMessages = state.allMessages || messages; // Fallback to compressed if no full history

    // Find the original user request from full history
    const userRequest = allMessages.find((msg: any) => msg._getType() === 'human')?.content || 'Unknown task';
    // Get the last AI response from full history
    const lastAIMessage = [...allMessages].reverse().find((msg: any) => msg._getType?.() === 'ai');
    const lastResponse = lastAIMessage?.content || '';
    
    // Extract tool execution history for context (Issue #37)
    const toolHistory = extractToolHistorySummary(allMessages);

    this.emitSystemEvent('reflecting', 'is reflecting', state, { modelName: this.modelName || 'AI', iteration: state.iterations, visionUsed: this.lastVisionUsed });

    const reflectionPrompt = `Review if the agent completed the user's request.

User request: "${userRequest}"

Actions taken:
${toolHistory}

Agent's final response: "${lastResponse}"

IMPORTANT: You are evaluating whether the agent COMPLETED the task, not whether the answer is factually correct.
- Review the actions taken above to understand what the agent did, what succeeded, and what failed.
- If the user asked for data from a webpage and the agent provided a specific value, trust that the agent extracted it correctly from the page.
- Do NOT reject responses based on your own knowledge of what the answer "should be" - the webpage may have different/updated data.
- Focus on: Did the agent attempt the task and provide a response in the requested format?
- If any actions failed, consider whether the agent handled the failure appropriately or if retry is needed.
- If the agent asks the user for clarification or more details, mark the task as COMPLETED so the user can provide the necessary information.

Did the agent complete the user's request?
`;

    // Retry logic for reflection with exponential backoff
    const maxRetries = 3;
    let lastError: Error | null = null;

    for (let attempt = 1; attempt <= maxRetries; attempt++) {
      try {
        if (attempt > 1) {
          const delay = Math.min(1000 * Math.pow(2, attempt - 1), 5000); // Exponential backoff, max 5s
          await new Promise(resolve => setTimeout(resolve, delay));
        }

        // Use structured output - tests and production models expect this
        const reflectionSchema = z.object({
          completed: z.boolean().describe("Whether the task was actually completed"),
          reflection: z.string().describe("Explanation of completion status")
        });

        const reflectionLLM = this.llm.withStructuredOutput(reflectionSchema, {
          name: "task_reflection",
        });


        this.emitSystemEvent('thinking', 'reflecting', state, { visionUsed: this.lastVisionUsed });
        const reflection = await reflectionLLM.invoke(reflectionPrompt);

        // Create a SystemMessage with metadata to identify it as a reflection
        const reflectionMessage = new AIMessage({
          content: `Task ${reflection.completed ? 'completed' : 'not complete'}. ${reflection.reflection}`,
          additional_kwargs: {
            reflection: true,
            task_complete: reflection.completed,
            app_type: 'thought'
          },
          response_metadata: reflection.completed ? { finish_reason: 'stop' } : undefined
        });

        // If task is completed and we have a lastAIMessage, clone it and set app_type to 'answer'
        let finalAIMessage = null;
        if (reflection.completed && lastAIMessage) {
          // Clone the message and update app_type to 'answer' so it displays correctly
          finalAIMessage = new AIMessage({
            content: lastAIMessage.content,
            additional_kwargs: {
              ...lastAIMessage.additional_kwargs,
              app_type: 'answer'
            }
          });
        }

        const messagesToAdd = reflection.completed && finalAIMessage
          ? [reflectionMessage, finalAIMessage]
          : [reflectionMessage];

        return {
          messages: [...messages, reflectionMessage],
          allMessages: [...(state.allMessages || []), ...messagesToAdd],
          reflectionFeedback: { feedback: reflection.reflection, completed: reflection.completed }
        };
      } catch (error) {
        lastError = error instanceof Error ? error : new Error(getErrorMessage(error));
        console.error(`🔍 Reflection attempt ${attempt}/${maxRetries} failed:`, getErrorMessage(error));

        // If this is the last attempt, log full details and re-throw
        if (attempt === maxRetries) {
          console.error('🔍 All reflection attempts exhausted. Error details:', {
            name: lastError?.name,
            message: lastError?.message,
            stack: lastError?.stack,
          });
          throw new Error(`Reflection failed after ${maxRetries} attempts: ${lastError?.message}`);
        }
      }
    }
    throw lastError;
  }

  /**
   * Determine next step after assistant
   * Routes to tools if there are tool calls, otherwise to reflection (for agent mode) or end (for ask mode)
   */
  assistantEdge(state: State): string {
    console.log(`🔀 [assistantEdge] START - iterations=${state.iterations}, mode=${this.mode}`);
    const abortReason = this.isAborted();
    if (abortReason) {
      console.log(`🔀 [assistantEdge] ABORTED: ${abortReason}`);
      throw new Error(abortReason);
    }

    // Get the last message
    const lastMessage = state.messages[state.messages.length - 1];
    const hasToolCalls = !!(lastMessage.tool_calls && lastMessage.tool_calls.length > 0);
    const appType = lastMessage?.additional_kwargs?.app_type;
    console.log(`🔀 [assistantEdge] lastMessage type=${lastMessage?._getType?.()}, hasToolCalls=${hasToolCalls}, app_type=${appType}`);

    if (hasToolCalls) {
      console.log(`🔀 [assistantEdge] -> TOOLS (has tool calls)`);
      return Nodes.TOOLS;
    } else {
      // No tool calls - agent thinks it's done
      if (this.mode === 'ask') {
        // For ask mode, skip reflection and end immediately
        lastMessage.additional_kwargs.app_type = 'answer';
        console.log(`🔀 [assistantEdge] -> __end__ (ask mode)`);
        return "__end__";
      } else {
        // For agent mode: Check if assistantNode already determined this is a final answer
        const isFinalAnswer = lastMessage.additional_kwargs?.app_type === 'answer';
        if (isFinalAnswer) {
          console.log(`🎯 Final answer from assistantNode, skipping reflection`);
          console.log(`🔀 [assistantEdge] -> __end__ (final answer)`);
          return "__end__";
        }

        // For longer responses or unclear completion, go to reflection to verify
        console.log(`🔀 [assistantEdge] -> REFLECTION (need to verify completion)`);
        return Nodes.REFLECTION;
      }
    }
  }

  /**
   * Determine next step after reflection
   */
  reflectionEdge(state: State): string {
    const abortReason = this.isAborted();
    if (abortReason) {
      throw new Error(abortReason);
    }

    // If reflection confirmed completion or hit max iterations, end
    if (state.reflectionFeedback?.completed || state.iterations >= state.maxIterations) {
      return "__end__";
    }

    // Otherwise continue to assistant with feedback
    return "assistant";
  }

  /**
   * Determine next step after tools
   * Routes back to ASSISTANT for the next reasoning step, or to REFLECTION if doom loop detected
   */
  toolsEdge(state: State): string {
    const abortReason = this.isAborted();
    if (abortReason) {
      throw new Error(abortReason);
    }

    // If reflection already confirmed completion, end
    if (state.reflectionFeedback?.completed) {
      return "__end__";
    }

    // If max iterations reached, go to reflection for final answer
    if (state.iterations >= state.maxIterations) {
      return Nodes.REFLECTION;
    }

    return Nodes.ASSISTANT;
  }

  /**
   * Build and compile the graph
   */
  buildGraph(): any {
    const workflow = new StateGraph(MessagesState)
      .addNode(Nodes.ASSISTANT, this.assistantNode.bind(this))
      .addNode(Nodes.TOOLS, this.toolsNode.bind(this))
      // Always add reflection node - it will be skipped for ask mode
      .addNode(Nodes.REFLECTION, this.reflectionNode.bind(this))
      .addEdge("__start__", Nodes.ASSISTANT)  // Start with assistant
      .addConditionalEdges(Nodes.ASSISTANT, this.assistantEdge.bind(this), {
        [Nodes.TOOLS]: Nodes.TOOLS,
        [Nodes.REFLECTION]: Nodes.REFLECTION,
        "__end__": Nodes.END
      } as any)
      .addConditionalEdges(Nodes.TOOLS, this.toolsEdge.bind(this), {
        [Nodes.ASSISTANT]: Nodes.ASSISTANT,
        [Nodes.REFLECTION]: Nodes.REFLECTION,
        "__end__": Nodes.END
      } as any)
      .addConditionalEdges(Nodes.REFLECTION, this.reflectionEdge.bind(this), {
        [Nodes.ASSISTANT]: Nodes.ASSISTANT,
        "__end__": Nodes.END
      } as any);

    const compiled = workflow.compile({});
    return compiled;
  }


  async run(prompt: string, options: any = {}): Promise<any> {
    // Clean up any incomplete tool calls or orphaned tool messages before starting new turn
    this.cleanupIncompleteToolCalls();

    // Load skill descriptions for system prompt injection
    await this.loadSkillDescriptions();

    const newMessage = new HumanMessage(prompt);
    
    // Build initial messages, including skill context if available
    const existingMessages = this.state?.messages || [];
    const existingAllMessages = this.state?.allMessages || [];
    
    // Add skill descriptions as a system message if we have any and this is a new conversation
    const skillsMessage = this.skillDescriptions && existingMessages.length === 0
      ? [new SystemMessage({
          content: this.skillDescriptions,
          additional_kwargs: { app_type: 'skill_context' }
        })]
      : [];
    
    const initialState: Partial<State> = {
      ...(this.state || {}),
      messages: [
        ...existingMessages,
        ...skillsMessage,
        newMessage
      ],
      allMessages: [
        ...existingAllMessages,
        ...skillsMessage,
        newMessage
      ],
      reflectionFeedback: { feedback: "", completed: false }
    };

    try {
      const stream = await this.workflow.stream(initialState, {
        streamMode: "values",
        recursionLimit: 512  // Increased from 100 to match DEFAULT_MAX_ITERATIONS
      });

      // Add timeout protection for stream iteration (2 minutes max per iteration)
      let lastIterationTime = Date.now();
      let streamIterationCount = 0;

      console.log(`🌊 [stream] Starting stream iteration loop`);
      for await (const state of stream) {
        streamIterationCount++;
        const lastMsgType = state.messages?.[state.messages.length - 1]?._getType?.();
        const lastMsgAppType = state.messages?.[state.messages.length - 1]?.additional_kwargs?.app_type;
        console.log(`🌊 [stream] Iteration ${streamIterationCount}: iterations=${state.iterations}, msgs=${state.messages?.length}, lastMsgType=${lastMsgType}, app_type=${lastMsgAppType}`);

        // Check if agent was stopped during execution
        const abortReason = this.isAborted();
        if (abortReason) {
          console.log(`🌊 [stream] ABORTED: ${abortReason}`);
          throw new Error(abortReason);
        }

        lastIterationTime = Date.now();

        this.state = state as State;
        // Send allMessages (uncompressed) for UI display and events
        this.onEvent(EVENT_TYPES.AI_UPDATE, {
          ...state,
          messages: StateSerializer.serializeMessages(state.allMessages)
        });
      }
      console.log(`🌊 [stream] Stream completed after ${streamIterationCount} iterations`);

      console.log('🤖 Agent completed task');
      console.log(`📊 Final stats: ${this.state.allMessages?.length || 0} full messages, ${this.state.messages?.length || 0} compressed messages`);

      const allMessages = this.state.allMessages || this.state.messages;
      const lastAIMessage = [...allMessages].reverse().find((msg: any) => msg._getType?.() === 'ai');

      return {
        completed: true,
        output: lastAIMessage?.content || "",
        iterations: this.state.iterations || 0,
        tokenUsage: this.state.tokenUsage
      };

    } catch (error) {
      // Format user-friendly error message
      const errorStr = String(error);
      let userMessage = 'An error occurred while processing your request.';

      if (errorStr.includes('402') || errorStr.includes('Insufficient credits')) {
        userMessage = 'API credits exhausted. Please add credits to continue.';
      } else if (errorStr.includes('401') || errorStr.includes('Unauthorized')) {
        userMessage = 'API authentication failed. Please check your API key.';
      } else if (errorStr.includes('429') || errorStr.includes('rate limit')) {
        userMessage = 'Rate limit exceeded. Please wait and try again.';
      } else if (errorStr.includes('500') || errorStr.includes('503')) {
        userMessage = 'AI service temporarily unavailable. Please try again later.';
      } else if (error instanceof Error) {
        userMessage = error.message;
      }

      console.error('🤖 Agent error:', errorStr);

      // Emit error event to UI
      const currentState = this.state || initialState as State;
      const errorMessage = new AIMessage({
        content: userMessage,
        additional_kwargs: { app_type: 'error' }
      });

      const displayMessages = [...(currentState.allMessages || []), errorMessage];
      this.onEvent(EVENT_TYPES.AI_UPDATE, {
        ...currentState,
        messages: StateSerializer.serializeMessages(displayMessages)
      });

      throw error;
    }
  }

  /**
   * Serialize the current state for persistence
   * @returns Serialized state
   */
  serializeState(): any {
    return StateSerializer.serializeState(this.state);
  }

  /**
   * Restore state from serialized data
   * @param serialized - Serialized state data
   */
  restoreState(serialized: any): void {
    if (!serialized) {
      console.warn('🤖 No state to restore');
      return;
    }

    const restored = StateSerializer.deserializeState(serialized);
    if (restored) {
      this.state = restored as State;
      console.log(`🤖 State restored: ${restored.messages.length} compressed messages, ${restored.allMessages?.length || 0} full messages, ${restored.iterations} iterations`);
    }
  }

  /**
   * Check if the agent has existing state
   * @returns Whether the agent has state
   */
  hasState(): boolean {
    return this.state && this.state.messages && this.state.messages.length > 0;
  }

  /**
   * Clean up incomplete tool calls and orphaned tool messages from the message history
   * This is called when the agent is stopped or before starting a new turn
   * to prevent "tool_call_id" errors on the next invocation
   */
  cleanupIncompleteToolCalls(): void {
    if (!this.state || !this.state.messages || this.state.messages.length === 0) {
      return;
    }

    let messages = this.state.messages;
    let cleaned = false;

    // Step 1: Collect all valid tool_call_ids from AI messages with tool_calls
    const validToolCallIds = new Set();
    for (const msg of messages) {
      if (msg?.tool_calls && Array.isArray(msg.tool_calls)) {
        for (const tc of msg.tool_calls) {
          if (tc.id) {
            validToolCallIds.add(tc.id);
          }
        }
      }
    }

    // Step 2: Remove orphaned ToolMessages (those without a corresponding tool_call)
    const filteredMessages = messages.filter(msg => {
      if (msg?.tool_call_id && !validToolCallIds.has(msg.tool_call_id)) {
        console.log(`🤖 Removing orphaned ToolMessage with id: ${msg.tool_call_id}`);
        cleaned = true;
        return false; // Remove this message
      }
      return true; // Keep this message
    });

    // Step 3: Find incomplete tool_calls (AI messages with tool_calls that lack responses)
    for (let i = filteredMessages.length - 1; i >= 0; i--) {
      const msg = filteredMessages[i];

      // Check if this is an AI message with tool_calls
      if (msg?.tool_calls && Array.isArray(msg.tool_calls) && msg.tool_calls.length > 0) {
        const toolCallIds = new Set(msg.tool_calls.map((tc: any) => tc.id));

        // Check if all tool_calls have corresponding ToolMessage responses after this message
        const respondedToolCallIds = new Set();
        for (let j = i + 1; j < filteredMessages.length; j++) {
          const nextMsg = filteredMessages[j];
          if (nextMsg?.tool_call_id) {
            respondedToolCallIds.add(nextMsg.tool_call_id);
          }
        }

        // If not all tool_calls have responses, remove the incomplete tool_calls
        const missingResponses = Array.from(toolCallIds).filter(id => !respondedToolCallIds.has(id));
        if (missingResponses.length > 0) {
          console.log(`🤖 Cleaning up ${missingResponses.length} incomplete tool_call(s):`, missingResponses);

          // Remove the tool_calls property from the AI message
          delete msg.tool_calls;

          // Also remove any additional_kwargs.tool_calls if present
          if (msg.additional_kwargs?.tool_calls) {
            delete msg.additional_kwargs.tool_calls;
          }

          cleaned = true;
        }

        // Only check the most recent AI message with tool_calls
        break;
      }
    }

    // Update state with cleaned messages
    if (cleaned) {
      this.state.messages = filteredMessages;
      console.log('🤖 Message history cleaned: removed orphaned ToolMessages and incomplete tool_calls');
    }
  }

  /**
   * Get current tab ID for tool operations
   * Used by tools to operate on the correct tab (especially for subagents)
   */
  getCurrentTabId(): number | null {
    return this.currentTabId;
  }

  /**
   * Set current tab ID (for switch_tab operations)
   */
  setCurrentTabId(tabId: number | null): void {
    this.currentTabId = tabId;
  }

  /**
   * Extract tab ID from tool result or args for currentTabId tracking
   * Consolidates tab ID extraction logic for: navigate_to_url, switch_to_tab,
   * set_working_tab, create_new_tab, new_tab
   */
  private extractTabId(toolName: string, args: any, toolResult: any): number | null {
    // Tools that pass tabId via args
    const tabIdFromArgs = ['switch_to_tab', 'set_working_tab'];
    if (tabIdFromArgs.includes(toolName) && args?.tabId) {
      return args.tabId;
    }

    // Tools that return tabId in result (JSON format)
    const tabIdFromJsonResult = ['navigate_to_url', 'new_tab'];
    if (tabIdFromJsonResult.includes(toolName) && typeof toolResult === 'string') {
      try {
        const parsed = JSON.parse(toolResult);
        if (parsed.tabId && typeof parsed.tabId === 'number') {
          return parsed.tabId;
        }
      } catch (e) {
        // Not JSON, try regex extraction
      }
    }

    // Tools that return tabId in result (string format like "ID: 123456")
    const tabIdFromStringResult = ['create_new_tab', 'new_tab', 'navigate_to_url'];
    if (tabIdFromStringResult.includes(toolName) && typeof toolResult === 'string') {
      const match = toolResult.match(/ID:\s*(\d+)/);
      if (match) {
        return parseInt(match[1], 10);
      }
    }

    return null;
  }

  /**
   * Update the tools available to the agent
   * @param newTools - New array of langchain tools
   */
  setTools(newTools: any[]): void {
    this.langchainTools = newTools || [];
    // Recreate the assistant chain with tools bound based on mode
    const toolsToBind = this.mode === 'agent' ? this.langchainTools : [];
    this.assistantChain = REACT_PROMPT.pipe(
      this.llm.bindTools(toolsToBind)
    );
    // Rebuild the workflow graph with the updated assistant chain
    this.workflow = this.buildGraph();
  }

  /**
   * Update the LLM used by the agent
   * @param newLLM - New LLM instance to use
   */
  setLLM(newLLM: any): void {
    this.llm = newLLM;
    // Recreate the assistant chain with new LLM, binding tools based on mode
    const toolsToBind = this.mode === 'agent' ? this.langchainTools : [];
    this.assistantChain = REACT_PROMPT.pipe(
      this.llm.bindTools(toolsToBind)
    );
    // Rebuild the workflow graph with the updated assistant chain
    this.workflow = this.buildGraph();
  }

  /**
   * Extract the last HumanMessage from messages array
   * @param messages - Array of messages to search through
   * @returns The last HumanMessage found, or null if none exists
   */
  static getLastHumanMessage(messages: any[]): HumanMessage | null {
    if (!Array.isArray(messages) || messages.length === 0) {
      return null;
    }

    // Search backwards through the messages array to find the last HumanMessage
    for (let i = messages.length - 1; i >= 0; i--) {
      const message = messages[i];
      if (message?._getType && message._getType() === 'human') {
        return message as HumanMessage;
      }
    }

    return null;
  }

  /**
   * Get the last human task content from the current state
   * @returns The content of the last human task, or null if none exists
   */
  getLastHumanTask(): string | null {
    // Try to get from allMessages first (full history), then fallback to messages (compressed)
    const messages = this.state?.allMessages || this.state?.messages || [];
    const lastHumanMessage = ReactAgent.getLastHumanMessage(messages);

    if (!lastHumanMessage?.content) {
      return null;
    }

    // Handle different content types: string or array of content blocks
    const content = lastHumanMessage.content;
    if (typeof content === 'string') {
      return content;
    }

    // If content is an array of content blocks, extract text content
    if (Array.isArray(content)) {
      return content
        .filter((block: any) => block.type === 'text' && typeof block.text === 'string')
        .map((block: any) => block.text)
        .join('');
    }

    return null;
  }

  /**
   * Detect doom loops - Enhanced version
   *
   * Checks for:
   * 1. OpenCode style: 3 identical tool calls (same name + args)
   * 2. Navigation loop: Same URL visited 3+ times
   * 3. Pattern loop: Same sequence of tools repeated (even with different args)
   *
   * @param messages - Message history to check
   * @returns true if doom loop detected
   */
  detectDoomLoop(messages: BaseMessage[]): boolean {
    const IDENTICAL_THRESHOLD = 3;
    const URL_VISIT_THRESHOLD = 3;
    const PATTERN_LENGTH = 4;

    // Extract tool calls from AI messages
    const toolCalls: Array<{ name: string, args: any, argsStr: string }> = [];

    for (const msg of messages) {
      const aiMsg = msg as any;
      if (aiMsg?.tool_calls && Array.isArray(aiMsg.tool_calls)) {
        for (const tc of aiMsg.tool_calls) {
          const args = tc.args || {};
          toolCalls.push({
            name: tc.name || tc.function?.name || 'unknown',
            args,
            argsStr: JSON.stringify(args)
          });
        }
      }
    }

    const recentNames = toolCalls.slice(-10).map(tc => tc.name);
    console.log(`[DoomLoop] Checking ${toolCalls.length} recent tool calls: ${recentNames.join(', ')}`);

    if (toolCalls.length < IDENTICAL_THRESHOLD) {
      return false;
    }

    // Check 1: OpenCode style - 3 identical tool calls
    // EXCEPTION: create_new_tab with DIFFERENT URLs is valid (parallel pattern setup)
    const lastThree = toolCalls.slice(-IDENTICAL_THRESHOLD);
    const first = lastThree[0];
    const allIdentical = lastThree.every(
      tc => tc.name === first.name && tc.argsStr === first.argsStr
    );

    if (allIdentical) {
      // If argsStr is identical, it's a doom loop - no exceptions
      // Different URLs → different argsStr → allIdentical is false → this block doesn't run
      console.warn(`⚠️ DOOM LOOP (identical): "${first.name}" called ${IDENTICAL_THRESHOLD}x with identical args`);
      return true;
    }

    // Check 2: Same URL visited too many times (navigation loop)
    // EXCEPTION: newTab navigations are intentional parallel operations
    const navigateUrls: string[] = [];
    for (const tc of toolCalls) {
      if (tc.name === 'navigate_to_url' && tc.args?.url) {
        // Skip newTab navigations - these are legitimate for parallel patterns
        if (tc.args?.newTab === true) {
          continue;
        }
        navigateUrls.push(tc.args.url);
      }
    }

    const urlCounts = new Map<string, number>();
    for (const url of navigateUrls) {
      urlCounts.set(url, (urlCounts.get(url) || 0) + 1);
    }

    for (const [url, count] of urlCounts) {
      if (count >= URL_VISIT_THRESHOLD) {
        console.warn(`⚠️ DOOM LOOP (navigation): URL "${url}" visited ${count} times (excluding newTab)`);
        return true;
      }
    }

    // Check 2.5: Same web_fetch URL called too many times
    const webFetchUrls: string[] = [];
    for (const tc of toolCalls) {
      if (tc.name === 'web_fetch' && tc.args?.url) {
        webFetchUrls.push(tc.args.url);
      }
    }

    const webFetchUrlCounts = new Map<string, number>();
    for (const url of webFetchUrls) {
      webFetchUrlCounts.set(url, (webFetchUrlCounts.get(url) || 0) + 1);
    }

    for (const [url, count] of webFetchUrlCounts) {
      if (count >= 3) {  // Lower threshold for web_fetch since it's cheaper
        console.warn(`⚠️ DOOM LOOP (web_fetch): URL "${url}" fetched ${count} times`);
        return true;
      }
    }

    // Check 3: Pattern loop - same sequence of tool names AND args repeating
    // This must check BOTH names and args to avoid false positives when the agent
    // is doing similar actions (e.g., TodoWrite + navigate) for different tasks.
    // A real doom loop has identical tool calls, not just similar patterns.
    //
    // IMPORTANT: We compare the full argsStr (JSON-serialized args) to ensure
    // that different URLs, different todo lists, etc. are NOT considered identical.
    if (toolCalls.length >= PATTERN_LENGTH * 2) {
      const recentCalls = toolCalls.slice(-PATTERN_LENGTH * 2);
      const firstHalf = recentCalls.slice(0, PATTERN_LENGTH);
      const secondHalf = recentCalls.slice(PATTERN_LENGTH);

      // Debug: Log the actual args being compared
      console.log(`[DoomLoop] Pattern check - comparing:`);
      console.log(`  First half args: ${firstHalf.map(tc => tc.argsStr.substring(0, 80)).join(' | ')}`);
      console.log(`  Second half args: ${secondHalf.map(tc => tc.argsStr.substring(0, 80)).join(' | ')}`);

      // Compare both name AND args - must be truly identical
      const allMatching = firstHalf.every((tc, i) =>
        tc.name === secondHalf[i].name && tc.argsStr === secondHalf[i].argsStr
      );

      if (allMatching) {
        const patternNames = firstHalf.map(tc => tc.name).join(',');
        console.warn(`⚠️ DOOM LOOP (pattern): Repeating sequence [${patternNames}] with identical args`);
        return true;
      } else {
        console.log(`[DoomLoop] Pattern check PASSED - args are different`);
      }
    }

    return false;
  }
}
