/**
 * Playwright Page Content Provider
 *
 * This provider is used for Node.js environments (CLI, tests) where
 * Playwright is available for browser automation.
 *
 * NOTE: This file is separate from PageContentProvider.ts to avoid
 * bundling Node.js dependencies (fs, path, url) in the browser extension build.
 */

import type { Page } from 'playwright';
import { PageContentProvider, PageContentResult } from './PageContentProvider.js';

type PageGetter = ((targetId?: string) => Promise<Page>) | Page;

/**
 * Playwright implementation of PageContentProvider
 */
export class PlaywrightPageContentProvider extends PageContentProvider {
  private pageGetter: PageGetter;

  constructor(pageGetter: PageGetter) {
    super();
    this.pageGetter = pageGetter;
  }

  async getPageContent(targetId?: string): Promise<PageContentResult> {
    const page =
      typeof this.pageGetter === 'function'
        ? await this.pageGetter(targetId)
        : this.pageGetter;

    if (!page) {
      throw new Error('No page available for content extraction');
    }

    try {
      // Wait for page to stabilize
      await this.waitForPageStability(page);

      const url = await page.url();
      const title = await page.title();

      // Wait for financial data on specific sites
      await this.waitForFinancialData(page, url);

      // Extract content
      const { content, htmlContent } = await this.extractContent(page);

      // Format content with title as first line
      const formattedContent = title
         ? `# ${title}\n\n${content || 'Unable to extract page content'}\n\n> Selected Text:\n> ${(await page.evaluate(() => window.getSelection()?.toString())) || 'No text selected'}`
        : content || 'Unable to extract page content';

      return {
        content: formattedContent,
        htmlContent: htmlContent || '',
        url,
        title,
      };
    } catch (error) {
      console.error('[PlaywrightProvider] Content extraction error:', error);
      const errorMessage = error instanceof Error ? error.message : String(error);
      throw new Error(`Page content extraction failed: ${errorMessage}`);
    }
  }

  private async waitForPageStability(page: Page): Promise<void> {
    try {
      const currentUrl = await page.url();
      if (currentUrl.includes('morpho.org') || currentUrl.includes('app.')) {
        console.debug('[PlaywrightProvider] Detected web app, waiting for network idle...');
        await page.waitForLoadState('networkidle', { timeout: 10000 });
      } else {
        await page.waitForLoadState('domcontentloaded', { timeout: 3000 });
      }
    } catch {
      console.debug('[PlaywrightProvider] Page still loading, attempting extraction anyway');
    }
  }

  private async waitForFinancialData(page: Page, url: string): Promise<void> {
    if (
      !url.includes('morpho.org') &&
      !url.includes('app.') &&
      !url.includes('finance') &&
      !url.includes('defi')
    ) {
      return;
    }

    console.debug('[PlaywrightProvider] Waiting for financial data to load...');

    const financialSelectors = [
      'text=/%/',
      'text=/APY/i',
      'text=/\\d+\\.\\d+%/',
      '[class*="apy"]',
      '[class*="rate"]',
      '[class*="yield"]',
      'td:has-text("%")',
      'span:has-text("%")',
    ];

    for (const selector of financialSelectors) {
      try {
        await page.waitForSelector(selector, { timeout: 2000 });
        console.debug(`[PlaywrightProvider] Found financial data with selector: ${selector}`);
        break;
      } catch {
        // Try next selector
      }
    }

    await page.waitForTimeout(1500);

    if (url.includes('morpho.org')) {
      await page.waitForTimeout(2000);
    }
  }

  private async extractContent(
    page: Page
  ): Promise<{ content: string; htmlContent: string }> {
    const { readFileSync } = await import('fs');
    const { fileURLToPath } = await import('url');
    const { dirname, join } = await import('path');

    const __filename = fileURLToPath(import.meta.url);
    const __dirname = dirname(__filename);

    const markdownExtractorCode = readFileSync(
      join(__dirname, '../shared/markdownPageExtractor.js'),
      'utf8'
    );
    const htmlExtractorCode = readFileSync(
      join(__dirname, '../shared/htmlPageExtractor.js'),
      'utf8'
    );

    // Clean export declarations
    const cleanMarkdownExtractorCode = markdownExtractorCode
      .replace(/^export\s+{[^}]+};?\s*$/gm, '')
      .replace(/^export\s+default\s+\w+;?\s*$/gm, '');
    const cleanHtmlExtractorCode = htmlExtractorCode
      .replace(/^export\s+{[^}]+};?\s*$/gm, '')
      .replace(/^export\s+default\s+\w+;?\s*$/gm, '');

    let content = '';
    let htmlContent = '';
    let retryCount = 0;
    const maxRetries = 2;

    while (retryCount <= maxRetries) {
      try {
        const result = await page.evaluate(
          ([markdownCode, htmlCode]: [string, string]) => {
            eval(markdownCode);
            eval(htmlCode);

            // @ts-expect-error - MarkdownPageExtractor is injected via eval
            const markdownExtractor = new MarkdownPageExtractor();
            const markdownResult = markdownExtractor.extractContent({
              includeInteractive: true,
              includeChanges: true,
            });

            // @ts-expect-error - HtmlPageExtractor is injected via eval
            const htmlExtractor = new HtmlPageExtractor();
            const htmlResult = htmlExtractor.extractIndexedHtml({
              maxElements: 1000,
              includeHidden: true,
            });

            return {
              content: markdownResult.content,
              html: htmlResult.html,
            };
          },
          [cleanMarkdownExtractorCode, cleanHtmlExtractorCode] as [string, string]
        );

        content = result.content;
        htmlContent = result.html;

        if (content?.trim()) {
          break;
        }
      } catch (evalError) {
        retryCount++;
        if (retryCount > maxRetries) {
          throw evalError;
        }
        await new Promise((resolve) => setTimeout(resolve, 500));
      }
    }

    return { content, htmlContent };
  }
}
