/**
 * Planner-Executor Agent Architecture
 * Adapted for browser extension
 *
 * Key differences
 * - Uses markdown extraction instead of accessibility tree (no CDP required)
 * - Uses Moondream API for visual element location
 * - Screenshots included in every planner + executor iteration
 * - Structured output via Zod schemas
 */

import { StateGraph, Annotation } from "@langchain/langgraph/web";
import { BaseMessage, HumanMessage, AIMessage, SystemMessage, ToolMessage } from "@langchain/core/messages";
import { z } from "zod";
import { EVENT_TYPES } from "./event-types.js";
import { getPageState } from "./pageState.js";

// Configuration
const MAX_PLANNER_ITERATIONS = 50;
const MAX_EXECUTOR_ITERATIONS = 3;
const SCREENSHOT_SIZE_PLANNER = { maxWidth: 1280, quality: 80 }; // Large for strategic view
const SCREENSHOT_SIZE_EXECUTOR = { maxWidth: 800, quality: 70 };  // Medium for execution

// Node names
const Nodes = {
  PLANNER: "planner",
  EXECUTOR: "executor",
  END: "__end__"
};

// ==================== STRUCTURED SCHEMAS ====================

/**
 * Planner output schema - structured strategic reasoning
 */
const PlannerOutputSchema = z.object({
  userTask: z.string()
    .describe("Restate the user's request clearly"),

  currentState: z.string()
    .describe("Current browser state from screenshot and page content - what do you see?"),

  executionHistory: z.string()
    .describe("Brief summary of what actions have been tried and their outcomes"),

  challengesIdentified: z.string()
    .describe("Any obstacles, errors, or issues blocking progress"),

  stepByStepReasoning: z.string()
    .describe("Think step-by-step: What should happen next? Why? What's the best approach?"),

  proposedActions: z.array(z.string()).max(5)
    .describe("1-5 high-level actions for executor to perform (empty array if taskComplete=true)"),

  taskComplete: z.boolean()
    .describe("Is the user's request fully satisfied? Set true only when done."),

  finalAnswer: z.string()
    .describe("If taskComplete=true, provide complete answer to user. Otherwise empty string.")
});

/**
 * Agent state - tracks execution across iterations
 */
const PlannerExecutorState = Annotation.Root({
  // Messages history
  messages: Annotation({
    reducer: (x, y) => x.concat(y),
    default: () => [],
  }),

  // User's original request
  userTask: Annotation({
    reducer: (x, y) => y,
    default: () => "",
  }),

  // Iteration counters
  plannerIterations: Annotation({
    reducer: (x, y) => y,
    default: () => 0,
  }),

  executorIterations: Annotation({
    reducer: (x, y) => y,
    default: () => 0,
  }),

  // Task completion
  taskComplete: Annotation({
    reducer: (x, y) => y,
    default: () => false,
  }),

  // Final answer to user
  finalAnswer: Annotation({
    reducer: (x, y) => y,
    default: () => "",
  }),

  // Current planner output
  currentPlan: Annotation({
    reducer: (x, y) => y,
    default: () => null,
  }),

  // Execution history for planner context
  executionHistory: Annotation({
    reducer: (x, y) => x.concat(y),
    default: () => [],
  }),

  // Token usage tracking
  tokenUsage: Annotation({
    reducer: (x, y) => ({
      input_tokens: (x?.input_tokens || 0) + (y?.input_tokens || 0),
      output_tokens: (x?.output_tokens || 0) + (y?.output_tokens || 0),
      total_tokens: (x?.total_tokens || 0) + (y?.total_tokens || 0),
    }),
    default: () => ({ input_tokens: 0, output_tokens: 0, total_tokens: 0 }),
  }),
});

// ==================== PROMPTS ====================

const PLANNER_SYSTEM_PROMPT = `You are a strategic planner for browser automation.

Your role is to:
1. Analyze the current situation (screenshot + page content)
2. Review what has been tried (execution history)
3. Think strategically about the best next steps
4. Propose 1-5 high-level actions for the executor

You see:
- Screenshot of current page (visual layout)
- Page content in markdown (text, links, buttons with indices)
- Execution history (what has been tried)

<executor-capabilities>
The executor has two interaction modes:
1. INDEX-BASED (primary): Uses [index] from page content
   - Fast and reliable when indices are clear
2. VISUAL (fallback): Uses visual descriptions
   - Used when indices fail or are unclear
   - Example: "Click the blue submit button at bottom of form"

When proposing actions:
- Prefer index-based actions when elements have clear indices
- Suggest visual descriptions when:
  * Indices are unclear or ambiguous
  * Previous index-based attempts failed
  * Elements are dynamic or in popups
</executor-capabilities>

Think carefully and propose clear actions. The executor will implement your plan.`;

const EXECUTOR_SYSTEM_PROMPT = `You are an executor agent that performs browser actions.

Your role is to:
1. Execute the proposed actions from the planner
2. Use available tools to interact with the browser
3. Call 'done' tool when all actions are completed

<tool-strategy>
PRIMARY TOOLS (Try first):
- click_by_index(index): Click element using index from page content [0:score]
- fill_by_index(index, value): Fill input field using index from page content

VISUAL FALLBACK TOOLS (Use when primary tools fail):
- visual_click(description): Click element by visual description
  Example: visual_click("blue submit button at bottom of form")
- visual_type(description, text): Type into field by visual description
  Example: visual_type("email input field", "user@example.com")
</tool-strategy>

<fallback-strategies>
CLICK ESCALATION STRATEGY:
1. First attempt: Use click_by_index(index) with index from page content
2. If "Element not found" or "Click failed": Use visual_click with descriptive text
3. Visual descriptions should include:
   - Color/appearance: "blue button", "red link"
   - Position: "top right corner", "below the header"
   - Text content: "containing 'Submit'", "labeled 'Search'"
   - Context: "in the login form", "next to the logo"

WHEN TO USE VISUAL FALLBACK:
- Error: "Element [index] not found" → Immediate visual_click
- Error: "Failed to click" → Retry with visual_click
- Situation: Index unclear in screenshot → Use visual_click directly
- Situation: Dynamic/popup elements → Prefer visual_click
- After 2 failed indexed clicks → Switch to visual approach

VISUAL DESCRIPTION BEST PRACTICES:
✓ "blue submit button at bottom of form"
✓ "search icon in top navigation bar"
✓ "first checkbox in the list"
✓ "X close button in modal corner"
✗ "element-123" (too technical)
✗ "button" (too vague)
</fallback-strategies>

Work through the actions systematically, then call 'done' when finished.`;

// ==================== AGENT CLASS ====================

export class PlannerExecutorAgent {
  constructor({ llm, langchainTools, options, memory }) {
    console.error('[PLANNER-EXECUTOR] *** CONSTRUCTOR CALLED ***');

    try {
      this.plannerLLM = llm; // For strategic planning with structured output
      this.executorLLM = llm; // For tool execution
      this.langchainTools = langchainTools || [];
      this.memory = memory;
      this.options = options || {};

      // Event emitter
      this.onEvent = options?.onEvent || (() => {});
      this.isAborted = options?.isAborted || (() => false);
      this.getPageContentTool = options?.getPageContentTool;

      // Config option to enable machine vision (screenshots) - decided at ai_agent.js level
      this.takeScreenshot = options?.takeScreenshot !== false; // Default to true
      console.log('[PLANNER-EXECUTOR] takeScreenshot config:', this.takeScreenshot);

      // Moondream API key for visual tools
      this.moondreamApiKey = options?.moondreamApiKey || process.env.MOONDREAM_API_KEY;

      console.log('[PLANNER-EXECUTOR] Tools available:', this.langchainTools.map(t => t.name));
      console.log('[PLANNER-EXECUTOR] Moondream API:', this.moondreamApiKey ? 'configured' : 'not configured');

      // State
      this.state = {};

      // Build workflow graph
      console.error('[PLANNER-EXECUTOR] About to call buildGraph()...');
      this.workflow = this.buildGraph();
      console.error('[PLANNER-EXECUTOR] buildGraph() completed, workflow:', typeof this.workflow);
    } catch (error) {
      console.error('[PLANNER-EXECUTOR] CONSTRUCTOR ERROR:', error);
      console.error('[PLANNER-EXECUTOR] Error message:', error.message);
      console.error('[PLANNER-EXECUTOR] Error stack:', error.stack);
      throw error;
    }
  }

  // ==================== PLANNER NODE ====================

  /**
   * Planner node - strategic reasoning with structured output
   */
  async plannerNode(state) {
    console.log(`[PLANNER] Iteration ${state.plannerIterations + 1}/${MAX_PLANNER_ITERATIONS}`);

    // Check iteration limit
    if (state.plannerIterations >= MAX_PLANNER_ITERATIONS) {
      return {
        taskComplete: true,
        finalAnswer: "Task incomplete - reached iteration limit"
      };
    }

    // Build execution history summary
    const historyText = state.executionHistory.length > 0
      ? state.executionHistory.join('\n\n')
      : "No actions taken yet.";

    // Get page state with screenshot
    const pageState = await getPageState({
      getPageContentTool: this.getPageContentTool,
      getTabs: null, // Planner doesn't need tabs info
      takeScreenshotCallback: this.takeScreenshotCallback,
      supportsVision: true, // ai_agent.js already decided if screenshots should be taken
      screenshotOptions: {
        ...SCREENSHOT_SIZE_PLANNER,
        reasoning: 'Planner strategic context'
      },
      includeIndexedNodesMarkdown: true,
      includeMarkdown: false,
      includeIndexedNodesHtml: false,
      includeScreenshot: this.takeScreenshot
    });

    const visionUsed = !!pageState.screenshot;

    // Build prompt for planner
    const userPrompt = `TASK: ${state.userTask}

EXECUTION METRICS:
- Planner iterations: ${state.plannerIterations}
- Executor iterations: ${state.executorIterations}

EXECUTION HISTORY:
${historyText}

<browser-state>
Current page:
- Title: ${pageState.pageTitle}
- URL: ${pageState.pageUrl}

Page content (with [index] for interactive elements):
${pageState.indexedNodesMarkdown}
</browser-state>

Analyze the screenshot and browser state. Plan the next strategic actions.`;

    // Build HumanMessage with text + screenshot
    const messageContent = [
      { type: "text", text: userPrompt }
    ];

    // Add screenshot if available
    if (visionUsed) {
      messageContent.push({
        type: "image_url",
        image_url: { url: pageState.screenshot }
      });
      console.log('[PLANNER] Including screenshot in context (vision enabled)');
    } else if (!this.supportsVision) {
      console.log('[PLANNER] Skipping screenshot (model does not support vision)');
    }

    // Get structured output from LLM
    const structuredLLM = this.plannerLLM.withStructuredOutput(PlannerOutputSchema);

    const messages = [
      new SystemMessage(PLANNER_SYSTEM_PROMPT),
      new HumanMessage({ content: messageContent })
    ];

    console.log('[PLANNER] Invoking LLM for strategic planning...');
    const plannerOutput = await structuredLLM.invoke(messages);

    console.log('[PLANNER] Plan:', {
      taskComplete: plannerOutput.taskComplete,
      actions: plannerOutput.proposedActions,
      reasoning: plannerOutput.stepByStepReasoning.substring(0, 200)
    });

    // Update state (don't store messages - only use for LLM invocation)
    return {
      plannerIterations: state.plannerIterations + 1,
      taskComplete: plannerOutput.taskComplete,
      finalAnswer: plannerOutput.finalAnswer,
      currentPlan: plannerOutput
    };
  }

  // ==================== EXECUTOR NODE ====================

  /**
   * Executor node - execute actions from planner
   */
  async executorNode(state) {
    console.log(`[EXECUTOR] Starting execution`);

    if (!state.currentPlan || !state.currentPlan.proposedActions) {
      console.error('[EXECUTOR] No plan to execute');
      return {};
    }

    const plan = state.currentPlan;
    const actions = plan.proposedActions;

    if (actions.length === 0) {
      console.log('[EXECUTOR] No actions to execute');
      return {};
    }

    // Get page state with screenshot
    const pageState = await getPageState({
      getPageContentTool: this.getPageContentTool,
      getTabs: null, // Executor doesn't need tabs info
      takeScreenshotCallback: this.takeScreenshotCallback,
      supportsVision: true, // ai_agent.js already decided if screenshots should be taken
      screenshotOptions: {
        ...SCREENSHOT_SIZE_EXECUTOR,
        reasoning: 'Executor action context'
      },
      includeIndexedNodesMarkdown: true,
      includeMarkdown: false,
      includeIndexedNodesHtml: false,
      includeScreenshot: this.takeScreenshot
    });

    const visionUsed = !!pageState.screenshot;

    // Build executor prompt
    let executorPrompt = `Execute the following actions:
${actions.map((a, i) => `${i + 1}. ${a}`).join('\n')}

<browser-state>
Current page:
- Title: ${pageState.pageTitle}
- URL: ${pageState.pageUrl}

Interactive elements (with [index] for click/fill):
${pageState.indexedNodesMarkdown}
</browser-state>

INSTRUCTIONS:
1. Try index-based tools first: click_by_index([index]) or fill_by_index([index], text)
2. If index fails or unclear, use visual tools: visual_click("description") or visual_type("description", text)
3. Call 'done' when all actions are complete

Work through the actions systematically.`;

    // Build HumanMessage with text + screenshot
    const messageContent = [
      { type: "text", text: executorPrompt }
    ];

    // Add screenshot if available
    if (visionUsed) {
      messageContent.push({
        type: "image_url",
        image_url: { url: pageState.screenshot }
      });
      console.log('[EXECUTOR] Including screenshot in context (vision enabled)');
    } else if (!this.supportsVision) {
      console.log('[EXECUTOR] Skipping screenshot (model does not support vision)');
    }

    // Bind tools to executor LLM
    const executorChain = this.executorLLM.bindTools(this.langchainTools);

    // Execute up to MAX_EXECUTOR_ITERATIONS
    const messages = [
      new SystemMessage(EXECUTOR_SYSTEM_PROMPT),
      new HumanMessage({ content: messageContent })
    ];

    const executionResults = [];
    let doneToolCalled = false;

    for (let i = 0; i < MAX_EXECUTOR_ITERATIONS && !doneToolCalled; i++) {
      console.log(`[EXECUTOR] Sub-iteration ${i + 1}/${MAX_EXECUTOR_ITERATIONS}`);

      // Get response with tool calls
      const response = await executorChain.invoke(messages);
      messages.push(response);

      // Execute tools
      if (response.tool_calls && response.tool_calls.length > 0) {
        for (const toolCall of response.tool_calls) {
          const toolName = toolCall.name;
          const tool = this.langchainTools.find(t => t.name === toolName);

          if (!tool) {
            const errorMsg = new ToolMessage({
              content: `Tool ${toolName} not found`,
              tool_call_id: toolCall.id,
              name: toolName
            });
            messages.push(errorMsg);
            continue;
          }

          try {
            console.log(`[EXECUTOR] Calling tool: ${toolName}`);
            const result = await tool.invoke(toolCall.args);

            const toolMsg = new ToolMessage({
              content: typeof result === 'string' ? result : JSON.stringify(result),
              tool_call_id: toolCall.id,
              name: toolName
            });
            messages.push(toolMsg);

            executionResults.push(`${toolName}: ${JSON.stringify(result)}`);

            // Check if done tool was called
            if (toolName === 'done') {
              doneToolCalled = true;
              console.log('[EXECUTOR] Done tool called, execution complete');
            }

          } catch (error) {
            console.error(`[EXECUTOR] Tool ${toolName} failed:`, error);
            const errorMsg = new ToolMessage({
              content: `Error: ${error.message}`,
              tool_call_id: toolCall.id,
              name: toolName
            });
            messages.push(errorMsg);
            executionResults.push(`${toolName}: ERROR - ${error.message}`);
          }
        }
      } else {
        // No tool calls, break
        break;
      }
    }

    // Build execution summary (store text only, not full messages with screenshots)
    const executionSummary = `Executed actions from plan:
Actions: ${actions.join('; ')}
Results: ${executionResults.join('; ')}
Done: ${doneToolCalled}`;

    console.log('[EXECUTOR] Execution summary:', executionSummary.substring(0, 200));

    // Don't return messages to state - only executionHistory text summary
    return {
      executorIterations: state.executorIterations + 1,
      executionHistory: [executionSummary]
    };
  }

  // ==================== ROUTING ====================

  /**
   * Route from planner - go to executor or end
   */
  plannerEdge(state) {
    const abortReason = this.isAborted();
    if (abortReason) {
      throw new Error(abortReason);
    }

    // If task complete, end
    if (state.taskComplete) {
      console.log('[PLANNER] Task marked complete, ending');
      return Nodes.END;
    }

    // If no actions proposed, something is wrong
    if (!state.currentPlan || !state.currentPlan.proposedActions || state.currentPlan.proposedActions.length === 0) {
      console.warn('[PLANNER] No actions proposed, ending');
      return Nodes.END;
    }

    // Go to executor
    return Nodes.EXECUTOR;
  }

  /**
   * Route from executor - go back to planner
   */
  executorEdge(state) {
    const abortReason = this.isAborted();
    if (abortReason) {
      throw new Error(abortReason);
    }

    // Always go back to planner for next planning iteration
    return Nodes.PLANNER;
  }

  // ==================== GRAPH ====================

  /**
   * Build LangGraph workflow
   */
  buildGraph() {
    const workflow = new StateGraph(PlannerExecutorState)
      .addNode(Nodes.PLANNER, this.plannerNode.bind(this))
      .addNode(Nodes.EXECUTOR, this.executorNode.bind(this))
      .addEdge("__start__", Nodes.PLANNER)
      .addConditionalEdges(Nodes.PLANNER, this.plannerEdge.bind(this), {
        [Nodes.EXECUTOR]: Nodes.EXECUTOR,
        [Nodes.END]: Nodes.END
      })
      .addConditionalEdges(Nodes.EXECUTOR, this.executorEdge.bind(this), {
        [Nodes.PLANNER]: Nodes.PLANNER,
        [Nodes.END]: Nodes.END
      });

    return workflow.compile({
      recursionLimit: MAX_PLANNER_ITERATIONS + MAX_EXECUTOR_ITERATIONS
    });
  }

  // ==================== RUN ====================

  /**
   * Run the agent
   */
  async run(prompt, options = {}) {
    console.log('[PLANNER-EXECUTOR] Starting agent with task:', prompt);

    const initialState = {
      userTask: prompt
      // Note: messages[] not used in state - only for LLM invocations
      // Screenshots included fresh in each request, not accumulated
    };

    const stream = await this.workflow.stream(initialState, {
      streamMode: "values"
    });

    for await (const state of stream) {
      this.state = state;
      this.onEvent(EVENT_TYPES.AI_UPDATE, { ...state });
    }

    console.log('[PLANNER-EXECUTOR] Agent completed');
    console.log('[PLANNER-EXECUTOR] Final state:', {
      taskComplete: this.state.taskComplete,
      plannerIterations: this.state.plannerIterations,
      executorIterations: this.state.executorIterations,
      finalAnswer: this.state.finalAnswer?.substring(0, 200)
    });

    return {
      completed: this.state.taskComplete || false,
      output: this.state.finalAnswer || "Task incomplete",
      iterations: this.state.plannerIterations || 0,
      tokenUsage: this.state.tokenUsage
    };
  }

  /**
   * Clean up incomplete tool calls from the message history
   * This is called when the agent is stopped to prevent "tool_call_id" errors
   * Note: PlannerExecutorAgent doesn't accumulate messages in state like ReactGraph,
   * so this is a no-op for compatibility
   */
  cleanupIncompleteToolCalls() {
    // PlannerExecutorAgent doesn't maintain a persistent message history
    // Each planner/executor invocation uses fresh context
    console.log('🤖 PlannerExecutorAgent: No message cleanup needed (stateless design)');
  }
}
