/**
 * Page Content Provider Interface
 *
 * This abstraction allows the agent to get page content from any source
 * without being coupled to specific implementations (playwright, extension, etc.)
 *
 * The agent doesn't need to know about pages, tabs, or browser instances.
 * It just needs a way to get content when needed.
 *
 * NOTE: PlaywrightPageContentProvider is in a separate file (PlaywrightPageContentProvider.ts)
 * to avoid bundling Node.js dependencies (fs, path, url) in the browser extension build.
 */

// Type definitions
export interface PageContentResult {
  content: string;
  htmlContent: string;
  url: string;
  title: string;
  tabId?: number;
  isSystemPage?: boolean;
  isError?: boolean;
  errorType?: string;
  errorMessage?: string;
}

export interface ContentExtractionOptions {
  includeInteractive?: boolean;
  includeChanges?: boolean;
  useIndexedElements?: boolean;
}

// Error titles that indicate page load failure
const ERROR_TITLES = [
  "This site can't be reached",
  "This page isn't working",
  'Server not found',
  'Unable to connect',
  'Connection refused',
  'ERR_CONNECTION_REFUSED',
  'ERR_CONNECTION_TIMED_OUT',
  'ERR_NAME_NOT_RESOLVED',
  'ERR_NETWORK_CHANGED',
  'ERR_INTERNET_DISCONNECTED',
  'ERR_SSL_PROTOCOL_ERROR',
  'ERR_CERT_COMMON_NAME_INVALID',
  'ERR_CERT_DATE_INVALID',
  '404',
  '500',
  '502',
  '503',
  'Bad Gateway',
  'Service Unavailable',
  'Internal Server Error',
  'Page not found',
  'Access Denied',
  '403 Forbidden',
] as const;

/**
 * Abstract base class for page content providers
 */
export abstract class PageContentProvider {
  /**
   * Get the current page content
   */
  abstract getPageContent(targetId?: string | number): Promise<PageContentResult>;

  /**
   * Get the current URL
   */
  async getCurrentUrl(targetId?: string | number): Promise<string> {
    const { url } = await this.getPageContent(targetId);
    return url;
  }

  /**
   * Get the current page title
   */
  async getPageTitle(targetId?: string | number): Promise<string> {
    const { title } = await this.getPageContent(targetId);
    return title;
  }
}

/**
 * Extension implementation of PageContentProvider
 */
export class ExtensionPageContentProvider extends PageContentProvider {
  async getPageContent(tabId?: number): Promise<PageContentResult> {
    try {
      const targetTabId = tabId ?? (await this.getCurrentActiveTab());
      const tab = await chrome.tabs.get(targetTabId);

      // Check for chrome-error:// pages (navigation failures) - handle specially before generic system page check
      if (tab.url?.startsWith('chrome-error://')) {
        console.log(`[ExtensionProvider] Chrome error page detected: ${tab.url}, title: ${tab.title}`);
        const errorMessage = tab.title
          ? `Page load failed: "${tab.title}". The site may be unreachable, the domain may not exist, or there is a network/DNS error.`
          : `Page load failed. The site may be unreachable, the domain may not exist, or there is a network/DNS error.`;
        return {
          content: `# Page Load Error\n\n${errorMessage}\n\nThis is a Chrome error page indicating the requested URL could not be loaded.`,
          htmlContent: '',
          url: tab.url || '',
          title: tab.title || 'Error',
          tabId: tab.id,
          isError: true,
          errorType: 'page_load_failed',
          errorMessage,
        };
      }

      // Check for system/extension pages
      if (this.isSystemPage(tab.url)) {
        console.log(`[ExtensionProvider] System/extension page detected: ${tab.url}`);
        return {
          content: `System page: This is a browser system or extension page where content extraction is not available.\n\nURL: ${tab.url}\nTitle: ${tab.title || 'System Page'}`,
          htmlContent: '',
          url: tab.url || '',
          title: tab.title || 'System Page',
          tabId: tab.id,
          isSystemPage: true,
        };
      }

      // Check for page load errors
      const pageLoadError = this.detectPageLoadError(tab);
      if (pageLoadError) {
        console.log(`[ExtensionProvider] Page load error detected: ${pageLoadError}`);
        return {
          content: `# Page Load Error\n\n${pageLoadError}\n\nURL: ${tab.url || 'unknown'}\nTitle: ${tab.title || 'Error'}`,
          htmlContent: '',
          url: tab.url || '',
          title: tab.title || 'Error',
          tabId: tab.id,
          isError: true,
          errorType: 'page_load_failed',
          errorMessage: pageLoadError,
        };
      }

      // Ensure content script is ready
      try {
        await this.ensureContentScriptReady(targetTabId);
      } catch (contentScriptError) {
        // Try injecting script manually before giving up
        try {
          console.log(`[ExtensionProvider] Content script ping failed, attempting manual injection for tab ${targetTabId}`);
          await chrome.scripting.executeScript({
            target: { tabId: targetTabId },
            files: ['content.js']
          });
          
          // Wait a bit for the script to initialize
          await new Promise(resolve => setTimeout(resolve, 500));
          
          // Try pinging one last time
          await this.ensureContentScriptReady(targetTabId, 2000);
          console.log(`[ExtensionProvider] Manual injection successful for tab ${targetTabId}`);
        } catch (injectionError) {
          console.error(`[ExtensionProvider] Manual injection failed:`, injectionError);
          // Continue to error handling below...
        }

        // Content script failed - re-fetch tab info to check if it's now an error page
        // Chrome error pages don't support content scripts, but the tab title/url may have updated
        console.log(`[ExtensionProvider] Content script failed, re-checking tab state...`);
        const updatedTab = await chrome.tabs.get(targetTabId);

        // Check if tab URL changed to chrome-error://
        if (updatedTab.url?.startsWith('chrome-error://')) {
          console.log(`[ExtensionProvider] Tab is now a chrome-error:// page: ${updatedTab.url}`);
          const errorMessage = updatedTab.title
            ? `Page load failed: "${updatedTab.title}". The site may be unreachable, the domain may not exist, or there is a network/DNS error.`
            : `Page load failed. The site may be unreachable, the domain may not exist, or there is a network/DNS error.`;
          return {
            content: `# Page Load Error\n\n${errorMessage}\n\nThis is a Chrome error page indicating the requested URL could not be loaded.`,
            htmlContent: '',
            url: updatedTab.url || '',
            title: updatedTab.title || 'Error',
            tabId: updatedTab.id,
            isError: true,
            errorType: 'page_load_failed',
            errorMessage,
          };
        }

        // Check for error titles that may have appeared
        const pageLoadError = this.detectPageLoadError(updatedTab);
        if (pageLoadError) {
          console.log(`[ExtensionProvider] Detected page load error after content script failure: ${pageLoadError}`);
          return {
            content: `# Page Load Error\n\n${pageLoadError}\n\nURL: ${updatedTab.url || 'unknown'}\nTitle: ${updatedTab.title || 'Error'}`,
            htmlContent: '',
            url: updatedTab.url || '',
            title: updatedTab.title || 'Error',
            tabId: updatedTab.id,
            isError: true,
            errorType: 'page_load_failed',
            errorMessage: pageLoadError,
          };
        }

        // Check if this is a restricted domain (Chrome Web Store, Google accounts, etc.)
        // These pages load fine but Chrome blocks content script injection for security
        if (this.isRestrictedDomain(updatedTab.url)) {
          console.log(`[ExtensionProvider] Restricted domain detected: ${updatedTab.url}`);
          return {
            content: `# Restricted Page (Content Not Accessible)\n\nThis page is open and loaded, but Chrome blocks extensions from reading its content for security reasons.\n\nURL: ${updatedTab.url}\nTitle: ${updatedTab.title || 'Restricted Page'}\n\nNote: The page IS open in the browser. You can navigate away or use other tabs, but cannot read or interact with this page's content.`,
            htmlContent: '',
            url: updatedTab.url || '',
            title: updatedTab.title || 'Restricted Page',
            tabId: updatedTab.id,
            isSystemPage: true, // Treat as system page - loaded but not accessible
          };
        }

        // Content script failed - check if page is actually loaded (has valid title, not an error page)
        const isPageLoaded = updatedTab.status === 'complete' && 
          updatedTab.title && 
          !ERROR_TITLES.some(err => updatedTab.title?.toLowerCase().includes(err.toLowerCase()));
        
        if (isPageLoaded) {
          // Page IS loaded but content script can't access it (SPAs like Telegram, WhatsApp Web)
          console.log(`[ExtensionProvider] Content script blocked on loaded page: ${updatedTab.url} (title: ${updatedTab.title})`);
          const errorMessage = `The page "${updatedTab.title}" is loaded but the extension cannot access its content. This is common with apps like Telegram, WhatsApp Web, and other SPAs that use strict security policies or Shadow DOM.`;
          return {
            content: `# Content Script Blocked\n\n${errorMessage}\n\nURL: ${updatedTab.url}\nTitle: ${updatedTab.title}\n\nThe page IS open and functional - you can interact with it manually or try refreshing.`,
            htmlContent: '',
            url: updatedTab.url || '',
            title: updatedTab.title || 'Unknown',
            tabId: updatedTab.id,
            isError: true,
            errorType: 'content_script_blocked',
            errorMessage,
          };
        }
        
        // Page genuinely failed to load
        console.log(`[ExtensionProvider] Page failed to load: ${updatedTab.url}`);
        const genericErrorMessage = `The page at "${updatedTab.url}" could not be loaded. This may indicate a network error, DNS failure, or server issue.`;
        return {
          content: `# Page Load Error\n\n${genericErrorMessage}\n\nURL: ${updatedTab.url || 'unknown'}\nTitle: ${updatedTab.title || 'Error'}`,
          htmlContent: '',
          url: updatedTab.url || '',
          title: updatedTab.title || 'Error',
          tabId: updatedTab.id,
          isError: true,
          errorType: 'page_load_failed',
          errorMessage: genericErrorMessage,
        };
      }

      // Extract content with timeout
      const response = await this.sendContentExtractionMessage(targetTabId);

      if (response?.success && response.result) {
        let content = '';
        let htmlContent = '';
        
        if (typeof response.result === 'string') {
          content = response.result;
        } else {
          content = response.result.content || '';
          htmlContent = response.result.htmlContent || '';
        }

        if (!content?.trim()) {
          throw new Error('Page content extraction returned empty result');
        }

        const formattedContent = tab.title ? `# ${tab.title}\n\n${content}` : content;

        return {
          content: formattedContent,
          htmlContent,
          url: tab.url || '',
          title: tab.title || '',
          tabId: tab.id,
        };
      }

      if (response?.error) {
        throw new Error(`Page content extraction failed: ${response.error}`);
      }

      throw new Error('Failed to extract page content - no response');
    } catch (error) {
      console.error('[ExtensionProvider] Content extraction error:', error);
      const errorMessage = error instanceof Error ? error.message : String(error);
      throw new Error(`Page content extraction failed: ${errorMessage}`);
    }
  }

  private isSystemPage(url?: string): boolean {
    if (!url) return false;
    return (
      url.startsWith('chrome-extension://') ||
      url.startsWith('chrome://') ||
      url.startsWith('chrome-error://') ||
      url.startsWith('about:') ||
      url.startsWith('edge://') ||
      url.startsWith('brave://')
    );
  }

  /**
   * Check if URL is a restricted domain where Chrome blocks content script injection
   * These pages load normally but extensions cannot access their content
   *
   * Source: chromium/src/extensions/common/extension_urls.cc
   * - kChromeWebstoreBaseURL = "https://chrome.google.com/webstore"
   * - kNewChromeWebstoreBaseURL = "https://chromewebstore.google.com/"
   */
  private isRestrictedDomain(url?: string): boolean {
    if (!url) return false;
    try {
      const urlObj = new URL(url);
      const hostname = urlObj.hostname.toLowerCase();
      // Chrome Web Store domains - verified from Chromium source
      return hostname === 'chrome.google.com' || hostname === 'chromewebstore.google.com';
    } catch {
      return false;
    }
  }

  private async sendContentExtractionMessage(
    tabId: number
  ): Promise<{ success?: boolean; result?: { content?: string; htmlContent?: string }; error?: string }> {
    const CONTENT_EXTRACTION_TIMEOUT = 15000;
    console.log(
      `[ExtensionProvider] Requesting page content from tab ${tabId} (timeout: ${CONTENT_EXTRACTION_TIMEOUT}ms)...`
    );
    const startTime = Date.now();

    let timeoutId: ReturnType<typeof setTimeout> | undefined;

    const response = await Promise.race([
      chrome.tabs
        .sendMessage(tabId, {
          type: 'getPageContent',
          data: {
            includeInteractive: true,
            includeChanges: true,
            useIndexedElements: true,
          },
        })
        .finally(() => {
          if (timeoutId) clearTimeout(timeoutId);
        }),
      new Promise<never>((_, reject) => {
        timeoutId = setTimeout(() => {
          console.error(
            `[ExtensionProvider] Content extraction timeout after ${CONTENT_EXTRACTION_TIMEOUT}ms for tab ${tabId}`
          );
          reject(
            new Error(
              `Content extraction timeout after ${CONTENT_EXTRACTION_TIMEOUT}ms - content script may be unresponsive`
            )
          );
        }, CONTENT_EXTRACTION_TIMEOUT);
      }),
    ]);

    console.log(`[ExtensionProvider] Content extraction completed in ${Date.now() - startTime}ms`);
    return response;
  }

  private async getCurrentActiveTab(): Promise<number> {
    const isWebTab = (tab: chrome.tabs.Tab): boolean => {
      if (!tab?.url) return false;
      return !this.isSystemPage(tab.url);
    };

    // Try active tab in current window
    let tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    let webTab = tabs.find(isWebTab);

    // Try current window
    if (!webTab) {
      console.log('[ExtensionProvider] Active tab is extension/system page, looking for web tab...');
      const windowTabs = await chrome.tabs.query({ currentWindow: true });
      const sortedWindowTabs = windowTabs.sort(
        (a, b) => (b.lastAccessed || 0) - (a.lastAccessed || 0)
      );
      webTab = sortedWindowTabs.find(isWebTab);
    }

    // Try last focused window
    if (!webTab) {
      console.log('[ExtensionProvider] No web tab in current window, trying last focused window...');
      const focusedTabs = await chrome.tabs.query({ lastFocusedWindow: true });
      const sortedFocusedTabs = focusedTabs.sort(
        (a, b) => (b.lastAccessed || 0) - (a.lastAccessed || 0)
      );
      webTab = sortedFocusedTabs.find(isWebTab);
    }

    // Try all windows
    if (!webTab) {
      console.log('[ExtensionProvider] No web tab in focused window, trying all windows...');
      const allTabs = await chrome.tabs.query({});
      const sortedAllTabs = allTabs.sort(
        (a, b) => (b.lastAccessed || 0) - (a.lastAccessed || 0)
      );
      webTab = sortedAllTabs.find(isWebTab);
    }

    if (!webTab?.id) {
      // No web page found - fall back to ANY active tab so agent can see current state
      console.log('[ExtensionProvider] No web tab found, falling back to active tab (may be extension page)...');
      const activeTabs = await chrome.tabs.query({ active: true, currentWindow: true });
      if (activeTabs[0]?.id) {
        console.log(`[ExtensionProvider] Using active tab: ${activeTabs[0].id} - ${activeTabs[0].title || activeTabs[0].url}`);
        return activeTabs[0].id;
      }
      // Still nothing - this should be extremely rare
      throw new Error('No tabs available in the browser');
    }

    console.log(`[ExtensionProvider] Found web tab: ${webTab.id} - ${webTab.title || webTab.url}`);
    return webTab.id;
  }

  private async ensureContentScriptReady(tabId: number, timeoutMs: number = 15000): Promise<boolean> {
    const RETRY_DELAY = 300;
    const INITIAL_PING_DURATION = 2000; // Try pinging for 2s before assuming we need to inject
    const startTime = Date.now();

    console.log(`[ExtensionProvider] Requesting page content from tab ${tabId} (timeout: ${timeoutMs}ms)...`);

    // Helper to ping the content script
    const ping = async (): Promise<boolean> => {
      try {
        const response = await chrome.tabs.sendMessage(tabId, { type: 'ping' });
        return response?.status === 'pong';
      } catch {
        return false;
      }
    };

    // 1. Initial Ping Phase
    while (Date.now() - startTime < INITIAL_PING_DURATION) {
      if (await ping()) return true;
      await new Promise((resolve) => setTimeout(resolve, RETRY_DELAY));
    }

    // 2. Injection Phase
    // If we're here, the script isn't responding. Attempt manual injection.
    console.log(
      `[ExtensionProvider] Content script not responding after ${INITIAL_PING_DURATION}ms, attempting manual injection...`
    );
    try {
      // Check if we can access the tab (avoid injecting into restricted pages if possible to detect early,
      // but executeScript will just fail if it's restricted, which we catch below)
      await chrome.scripting.executeScript({
        target: { tabId },
        files: ['content.js'],
      });
      console.log(`[ExtensionProvider] Manual injection successful for tab ${tabId}`);
      
      // Give it a moment to initialize
      await new Promise((resolve) => setTimeout(resolve, 500));
    } catch (injectionError) {
      console.warn(`[ExtensionProvider] Manual injection failed (page might be restricted):`, injectionError);
      // Continue to final ping phase anyway, just in case it was a race condition
    }

    // 3. Final Ping Phase
    // Keep pinging until total timeout expires
    while (Date.now() - startTime < timeoutMs) {
      if (await ping()) {
        console.log(`[ExtensionProvider] Content script ready after manual injection/wait`);
        return true;
      }
      await new Promise((resolve) => setTimeout(resolve, RETRY_DELAY));
    }

    console.log(
      `[ExtensionProvider] Content script not responding after ${timeoutMs}ms. It might be a restricted page or injection failed.`
    );
    throw new Error(
      'Content script not responding. The page might be restricted (chrome://, about:blank) or still loading.'
    );
  }

  /**
   * Detect page load errors from tab state
   * Returns error message if page failed to load, null otherwise
   */
  private detectPageLoadError(tab: chrome.tabs.Tab): string | null {
    if (tab.status === 'loading') {
      return null;
    }

    const title = tab.title || '';
    const url = tab.url || '';

    // Check title for error indicators
    for (const errorTitle of ERROR_TITLES) {
      if (title.toLowerCase().includes(errorTitle.toLowerCase())) {
        return `The page failed to load: "${title}". This may be due to a network error, server issue, or the URL may be incorrect.`;
      }
    }

    // Check for Chrome error page URLs
    if (url.startsWith('chrome-error://')) {
      return `The page failed to load (Chrome error page). The URL may be unreachable or the server may be down.`;
    }

    // Check for empty/blank pages
    if (tab.status === 'complete' && (!url || url === 'about:blank' || url === '')) {
      return `The page appears to be blank or empty. Navigation may have failed.`;
    }

    // Check for data: URLs with errors
    if (url.startsWith('data:text/html')) {
      if (title.includes('Error') || title.includes('error')) {
        return `The page shows an error: "${title}"`;
      }
    }

    return null;
  }
}

