import { StateGraph, Annotation } from "@langchain/langgraph/web";
import { BaseMessage, HumanMessage, AIMessage, SystemMessage, ToolMessage } from "@langchain/core/messages";
import { ChatPromptTemplate, MessagesPlaceholder } from "@langchain/core/prompts";
import { z } from "zod";
import { EVENT_TYPES } from "./event-types.js";
import { CaptchaDetector } from "./improvements/captcha-detector.js";
import { toYAML, formatThinkingYAML, formatReflectionYAML, formatToolCallYAML } from "./yaml-serializer.js";

const Nodes = {
  REFLECT: "reflect",
  EXECUTE: "execute",
  THINK: "think",
  END: "__end__"
};

/**
 * Execute-Reflect Agent Architecture
 * Based on the LangGraph reflection pattern from docs/reflection.md
 */


const DEFAULT_MAX_ITERATIONS = 1204;
const ABSOLUTE_MAX_ITERATIONS = 1024; // Hard limit to prevent infinite loops

// Define the state structure
const GraphState = Annotation.Root({
  // Core messaging state
  messages: Annotation({
    reducer: (x, y) => x.concat(y),
    default: () => [],
  }),

  // Current page content (updated before each execution)
  pageContent: Annotation({
    reducer: (x, y) => y,
    default: () => "",
  }),

  // Previous page content for comparison
  pageContentOld: Annotation({
    reducer: (x, y) => y,
    default: () => "",
  }),

  // Number of iterations completed
  iterations: Annotation({
    reducer: (x, y) => y,
    default: () => 0,
  }),

  // Maximum iterations allowed
  maxIterations: Annotation({
    reducer: (x, y) => y,
    default: () => DEFAULT_MAX_ITERATIONS,
  }),

  // Whether the task is complete
  taskComplete: Annotation({
    reducer: (x, y) => y,
    default: () => false,
  }),

  // Number of steps executed since last reflection
  stepsSinceReflection: Annotation({
    reducer: (x, y) => y,
    default: () => 0,
  }),

  // Flag for CAPTCHA blocking
  captchaBlocked: Annotation({
    reducer: (x, y) => y,
    default: () => false,
  }),

  // Token usage tracking - cumulative across all LLM calls
  tokenUsage: Annotation({
    reducer: (x, y) => ({
      input_tokens: (x?.input_tokens || 0) + (y?.input_tokens || 0),
      output_tokens: (x?.output_tokens || 0) + (y?.output_tokens || 0),
      total_tokens: (x?.total_tokens || 0) + (y?.total_tokens || 0),
    }),
    default: () => ({ input_tokens: 0, output_tokens: 0, total_tokens: 0 }),
  }),
});

// Execution prompt with Chain-of-Thought reasoning
const EXECUTION_PROMPT = ChatPromptTemplate.fromMessages([
  [
    "system",
    `You are a browser automation agent that uses Chain-of-Thought reasoning.

ALWAYS follow this format:

1. First, think step-by-step:
   - Analyze the current page state
   - Identify what the user wants to achieve
   - Decide what action to take next and why
   - Consider potential issues or edge cases

2. Then take action by calling the appropriate tool

IMPORTANT GUIDELINES:
- Always provide your reasoning BEFORE calling tools
- After receiving user's response to your question, take ACTION immediately
- Don't ask multiple follow-up questions - use reasonable defaults
- Handle popups/cookies first if they block the task
- Choose elements with highest scores when multiple options exist
- Wait 2-3 seconds after typing for dropdowns to appear`
  ],
  new MessagesPlaceholder("messages"),
]);


// Define schema for structured reflection output
const ReflectionSchema = z.object({
  taskComplete: z.boolean().describe("Whether the task has been completed successfully"),
  completionPercentage: z.number().min(0).max(100).describe("Estimated percentage of task completion (0-100)"),
  progress: z.string().describe("Summary of progress made in this iteration"),
  issues: z.array(z.string()).describe("List of any errors or issues encountered"),
  nextSteps: z.array(z.string()).describe("Specific actions to take in the next iteration"),
  critique: z.string().describe("Overall critique and analysis of the execution"),
  requestMoreIterations: z.boolean().describe("Whether to increase max iterations to continue working (only set when approaching limit)"),
  additionalIterationsNeeded: z.number().describe("Number of additional iterations needed to complete the task (0 if not requesting more)")
});

// Reflection prompt - SIMPLIFIED for faster analysis
const REFLECTION_PROMPT = ChatPromptTemplate.fromMessages([
  [
    "system",
    `Analyze the browser agent's execution and provide feedback.

CONTEXT:
- Current iteration: {iterations}/{maxIterations}
- Current page URL: {pageUrl}
- Current page title: {pageTitle}

The messages below show the full conversation history including:
- User's original request
- Agent's thinking process
- All tool calls made and their results
- Any errors encountered

EVALUATE:
1. Is the task complete? (Be skeptical - verify actual completion)
2. What progress was made? (0-100% estimate)
3. Any errors or issues?
4. What are the next steps needed?

Current page content:
{pageContent}

Return structured analysis based on the full conversation history.`
  ],
  new MessagesPlaceholder("messages"),
]);

const ThinkingSchema = z.object({
  confidence: z.number().min(0).max(1).describe("Confidence level in the planned action (0-1)"),
  reasoning: z.string().describe("Brief reasoning about the next action"),
  risks: z.array(z.string()).describe("Potential risks or issues"),
  proceed: z.boolean().describe("Whether to proceed with the action")
});

const THINKING_PROMPT = ChatPromptTemplate.fromMessages([
  ["system",
    `You are analyzing a browser automation action before execution.
CURRENT STATE:
- Page URL: {pageUrl}
- Page Title: {pageTitle}
- User Request: {userRequest}

ANALYZE THE NEXT STEP:
1. What action should be taken next?
2. What could go wrong?
3. What's your confidence level (0-1)?
4. Should we proceed?`],
]);

export class ExecuteReflectAgent {
  constructor({ getPageContentTool, llm, langchainTools, options, isAborted, memory }) {
    this.getPageContentTool = getPageContentTool; // The only way to get page content
    this.llm = llm;
    this.langchainTools = langchainTools;
    this.options = options;
    this.isAborted = isAborted || (() => false);
    this.memory = memory;
    // Set up unified event emitter
    this.onEvent = options?.onEvent || (() => { });


    this.captchaDetector = new CaptchaDetector(); // CAPTCHA/blocking detection

    // Validate that we have the required tool
    if (!this.getPageContentTool) {
      throw new Error('ExecuteReflectAgent requires getPageContentTool');
    }


    this.executionChain = EXECUTION_PROMPT.pipe(
      this.llm.bindTools(this.langchainTools || [])
    );

    this.reflectionChain = REFLECTION_PROMPT.pipe(
      this.llm.withStructuredOutput(ReflectionSchema, { strict: false })
    );

    this.thinkingChain = THINKING_PROMPT.pipe(
      this.llm.withStructuredOutput(ThinkingSchema, { strict: false })
    );
  }

  /**
   * Update the LLM and rebind all chains
   * @param {Object} newLLM - New LangChain LLM instance
   */
  setLLM(newLLM) {
    this.llm = newLLM;
    // Rebind chains with new LLM
    this.executionChain = EXECUTION_PROMPT.pipe(
      this.llm.bindTools(this.langchainTools || [])
    );
    this.reflectionChain = REFLECTION_PROMPT.pipe(
      this.llm.withStructuredOutput(ReflectionSchema, { strict: false })
    );
    this.thinkingChain = THINKING_PROMPT.pipe(
      this.llm.withStructuredOutput(ThinkingSchema, { strict: false })
    );
  }

  /**
   * Emit an event through the unified event system
   */
  #emitEvent(type, data) {
    const event = {
      type,
      tabId: this.tabId,
      data,
      timestamp: Date.now()
    };
    this.onEvent(event);
  }


  /**
   * Capture current page content and update state
   */
  async getPageState(state) {
    // Use the page content tool (agent doesn't know about implementation)
    const result = await this.getPageContentTool.execute();
    const pageContent = result.content || '';
    const htmlContent = result.htmlContent || '';
    const pageUrl = result.url || '';
    const pageTitle = result.title || '';

    // Check for CAPTCHA/blocking
    let captchaDetected = false;

    if (pageContent) {
      // Detect CAPTCHA
      const detection = await this.captchaDetector.detect(pageContent, pageUrl, pageTitle);

      if (detection.hasCaptcha) {
        captchaDetected = true;
        console.log(`[CAPTCHA] Detected ${detection.type} (confidence: ${detection.confidence}%), Indicators:`, detection.indicators.slice(0, 3).join(', '));
      }
    }

    return {
      pageContentOld: state.pageContent || "",
      pageContent: pageContent || "",
      htmlContent: htmlContent || "",
      pageUrl: pageUrl,
      pageTitle: pageTitle,
      captchaDetected
    };
  }

  /**
   * Compute the difference between old and new page content
   */
  computePageDiff(oldContent, newContent) {
    if (!oldContent || !newContent) {
      return "No previous content to compare";
    }

    if (oldContent === newContent) {
      return "No changes detected in page content";
    }

    // Simple diff - in production, you might want a more sophisticated diff
    const oldLines = oldContent.split('\n');
    const newLines = newContent.split('\n');

    const changes = [];
    const maxLines = Math.max(oldLines.length, newLines.length);

    for (let i = 0; i < maxLines; i++) {
      if (oldLines[i] !== newLines[i]) {
        if (i < oldLines.length && i < newLines.length) {
          changes.push(`Line ${i + 1} changed`);
        } else if (i >= oldLines.length) {
          changes.push(`Line ${i + 1} added`);
        } else {
          changes.push(`Line ${i + 1} removed`);
        }
      }
    }

    return changes.length > 0
      ? `${changes.length} changes detected:\n${changes.slice(0, 10).join('\n')}${changes.length > 10 ? '\n...' : ''}`
      : "Minor formatting changes only";
  }

  /**
   * Think node - analyzes context and plans next actions
   */
  async thinkNode(state) {
    // Check if we should stop
    if (this.isAborted()) {
      // Note: Stopped message AI_UPDATE is already emitted by background.ts UI:STOP handler
      // with proper messages[] format. No duplicate emission needed here.
      return {
        messages: [new AIMessage("Operation cancelled by user")],
        taskComplete: true
      };
    }

    const pageState = await this.getPageState(state);

    console.log('[THINK] Querying LLM...');
    const response = await this.thinkingChain.invoke({
      pageContent: pageState.pageContent,
      pageUrl: pageState.pageUrl,
      pageTitle: pageState.pageTitle,
      userRequest: this.userRequest,
      iterations: state.iterations,
    }, {
      timeout: 60000 // 60 second timeout for thinking
    });

    // Extract token usage from response
    const tokenUsage = {
      input_tokens: response.usage_metadata?.input_tokens || 0,
      output_tokens: response.usage_metadata?.output_tokens || 0,
      total_tokens: response.usage_metadata?.total_tokens || 0
    };
    if (tokenUsage.total_tokens === 0 && (tokenUsage.input_tokens > 0 || tokenUsage.output_tokens > 0)) {
      tokenUsage.total_tokens = tokenUsage.input_tokens + tokenUsage.output_tokens;
    }
    console.log(`[THINK] Token usage: ${tokenUsage.input_tokens} in, ${tokenUsage.output_tokens} out, ${tokenUsage.total_tokens} total`);

    return {
      pageContentOld: state.pageContent,
      pageContent: pageState.pageContent,
      thinkingOutput: response,
      lastThinkingIteration: state.iterations,
      messages: response ? [
        new SystemMessage("Thoughts:\n" + formatThinkingYAML(response))
      ] : [],
      captchaBlocked: pageState.captchaDetected,
      tokenUsage
    };
  }

  /**
   * Execute node - performs browser automation tasks
   */
  async executeNode(state) {
    // Check if we should stop
    if (this.isAborted()) {
      // Note: Stopped message AI_UPDATE is already emitted by background.ts UI:STOP handler
      // with proper messages[] format. No duplicate emission needed here.
      return {
        messages: [new AIMessage("Operation cancelled by user")],
        taskComplete: true
      };
    }

    let messagesToUse = state.messages;

    // Log execution iteration
    console.log(`▶️\texecuteNode() ${state.iterations + 1}`);

    const response = await this.executionChain.invoke({
      // add page content as a role:user message
      messages: [...messagesToUse,
      new HumanMessage("Page indexed content is enclosed in tripple backtricks. What step to do next?\n```\n" + state.pageContent + "\n```")],
    }, {
      timeout: 60000 // 60 second timeout for execution
    });

    // Extract token usage from execution response
    const tokenUsage = {
      input_tokens: response.usage_metadata?.input_tokens || 0,
      output_tokens: response.usage_metadata?.output_tokens || 0,
      total_tokens: response.usage_metadata?.total_tokens || 0
    };
    if (tokenUsage.total_tokens === 0 && (tokenUsage.input_tokens > 0 || tokenUsage.output_tokens > 0)) {
      tokenUsage.total_tokens = tokenUsage.input_tokens + tokenUsage.output_tokens;
    }
    console.log(`[EXECUTE] Token usage: ${tokenUsage.input_tokens} in, ${tokenUsage.output_tokens} out, ${tokenUsage.total_tokens} total`);

    // Handle tool calls
    const toolCalls = response.tool_calls || [];
    const toolMessages = [];

    for (const toolCall of toolCalls) {
      console.log(`[TOOL] ${toolCall.name}\n${formatToolCallYAML(toolCall.name, toolCall.args)}`);
      if (this.isAborted()) {
        // Emit agent stopped event when we actually stop
        this.#emitEvent(EVENT_TYPES.AGENT_STOPPED, {
          completed: false,
          reason: "user_stopped",
          iteration: state.iterations
        });

        return {
          messages: [new AIMessage("Operation cancelled by user")],
          taskComplete: true
        };
      }

      const tool = this.langchainTools.find(t => t.name === toolCall.name);
      if (!tool) {
        console.error(`Tool ${toolCall.name} not found`);
        continue;
      }
      try {
        const startTime = Date.now();
        const toolResult = await tool.invoke(toolCall.args);
        const executionTime = Date.now() - startTime;

        // Log tool result concisely
        const resultStr = typeof toolResult === 'string' ? toolResult : toYAML(toolResult);
        console.log(`[TOOL] Result (${executionTime}ms): ${resultStr.substring(0, 100)}...`);

        toolMessages.push(new ToolMessage({
          content: typeof toolResult === 'string' ? toolResult : toYAML(toolResult),
          tool_call_id: toolCall.id,
          name: toolCall.name
        }));

      } catch (error) {
        console.error(`Error executing tool ${toolCall.name}:`, error);

        toolMessages.push(new ToolMessage({
          content: `Tool execution failed: ${error.message}`,
          tool_call_id: toolCall.id,
          name: toolCall.name
        }));
      }
    }

    // Capture page content AFTER execution for reflection
    const postExecutionPageUpdate = await this.getPageState(state);

    return {
      pageContentOld: state.pageContent, // Previous content before execution
      pageContent: postExecutionPageUpdate.pageContent, // Current content after execution
      pageUrl: postExecutionPageUpdate.pageUrl, // URL from page content tool
      pageTitle: postExecutionPageUpdate.pageTitle, // Title from page content tool
      messages: [response, ...toolMessages],
      iterations: state.iterations + 1,
      stepsSinceReflection: state.stepsSinceReflection + 1,
      tokenUsage
    };
  }

  /**
   * Reflect node - critiques the execution and provides feedback
   */
  async reflectNode(state) {
    // Check if CAPTCHA blocked the task
    if (state.captchaBlocked) {
      console.log('[REFLECTION] Task blocked by CAPTCHA - marking as failed');

      // Return immediate failure reflection
      return {
        messages: [new SystemMessage(
          "Task failed due to CAPTCHA/blocking. Manual intervention required or use alternative approach."
        )],
        taskComplete: true, // Mark as complete (failed)
        stepsSinceReflection: 0,
        maxIterations: state.maxIterations
      };
    }

    const pageDiff = this.computePageDiff(state.pageContentOld, state.pageContent);
    let messagesForReflection = state.messages;

    // Extract page URL and title from current page content
    const pageUrl = state.pageContent?.match(/URL: (.*?)[\n\r]/)?.[1] || 'unknown';
    const pageTitle = state.pageContent?.match(/Title: (.*?)[\n\r]/)?.[1] || 'unknown';

    const reflection = await this.reflectionChain.invoke({
      messages: messagesForReflection,
      pageContent: state.pageContent,
      pageUrl: pageUrl,
      pageTitle: pageTitle,
      iterations: state.iterations,
      maxIterations: state.maxIterations,
    }, {
      // No timeout for reflection
    });

    // Extract token usage from reflection response
    const tokenUsage = {
      input_tokens: reflection.usage_metadata?.input_tokens || 0,
      output_tokens: reflection.usage_metadata?.output_tokens || 0,
      total_tokens: reflection.usage_metadata?.total_tokens || 0
    };
    if (tokenUsage.total_tokens === 0 && (tokenUsage.input_tokens > 0 || tokenUsage.output_tokens > 0)) {
      tokenUsage.total_tokens = tokenUsage.input_tokens + tokenUsage.output_tokens;
    }
    console.log(`[REFLECT] Token usage: ${tokenUsage.input_tokens} in, ${tokenUsage.output_tokens} out, ${tokenUsage.total_tokens} total`);

    // Handle iteration limit increase if requested
    let updatedMaxIterations = state.maxIterations;
    if (reflection.requestMoreIterations && reflection.additionalIterationsNeeded > 0) {
      const proposedIterations = state.maxIterations + reflection.additionalIterationsNeeded;

      // Apply hard limit to prevent infinite loops
      if (proposedIterations <= ABSOLUTE_MAX_ITERATIONS) {
        updatedMaxIterations = proposedIterations;
        console.log(`[REFLECT] Increasing iteration limit from ${state.maxIterations} to ${updatedMaxIterations}`);
      } else {
        console.log(`[REFLECT] Cannot increase iterations beyond absolute limit of ${ABSOLUTE_MAX_ITERATIONS}`);
      }
    }

    return {
      messages: [new SystemMessage({
        content: "Reflection:\n" + formatReflectionYAML(reflection)
      })],
      taskComplete: reflection.taskComplete,
      stepsSinceReflection: 0, // Reset step counter after reflection
      maxIterations: updatedMaxIterations,
      tokenUsage
    };
  }

  /**
   * Determine if we should continue after thinking
   */
  thinkEdge(state) {
    // If task is already complete or we hit max iterations, end
    if (state.taskComplete || state.iterations >= state.maxIterations) {
      console.log(`[GRAPH] Task complete or max iterations reached after thinking`);
      return "__end__";
    }

    // If CAPTCHA blocked, go to reflection for handling
    if (state.captchaBlocked) {
      console.log(`[GRAPH] CAPTCHA blocking detected, moving to reflection`);
      return "reflect";
    }

    // Otherwise, proceed to execution
    return "execute";
  }

  executeEdge(state) {
    // If we've reached max iterations, do a final reflection before ending
    if (state.iterations >= state.maxIterations) {
      console.log(`[GRAPH] Max iterations reached (${state.iterations}/${state.maxIterations})`);
      // If we haven't reflected recently, do one final reflection
      if (state.stepsSinceReflection > 0) {
        return "reflect";
      }
      return "__end__";
    }

    if (state.taskComplete) {
      return "__end__";
    }

    const reflectionThreshold = Math.min(3, Math.floor(state.maxIterations * 0.3));
    if (state.stepsSinceReflection >= reflectionThreshold) {
      console.log(`[GRAPH] Reflection threshold reached (${state.stepsSinceReflection}/${reflectionThreshold}), reflecting...`);
      return "reflect";
    }

    // Continue executing if we haven't reached the reflection threshold
    console.log(`[GRAPH] Continuing execution (${state.stepsSinceReflection}/${reflectionThreshold})`);
    return Nodes.EXECUTE;
  }

  /**
   * Determine next step after reflection
   */
  reflectEdge(state) {
    // After reflection, check if we should end
    if (state.taskComplete) {
      return "__end__";
    }
    // Check if we've reached the (possibly updated) max iterations
    if (state.iterations >= state.maxIterations) {
      return "__end__";
    }
    // Return to thinking for next iteration
    return "think";
  }

  /**
   * Build and compile the graph
   */
  buildGraph() {
    const workflow = new StateGraph(GraphState)
      .addNode(Nodes.THINK, this.thinkNode.bind(this))
      .addNode(Nodes.EXECUTE, this.executeNode.bind(this))
      .addNode(Nodes.REFLECT, this.reflectNode.bind(this))
      .addEdge("__start__", Nodes.THINK)  // Start with thinking
      .addConditionalEdges(Nodes.THINK, this.thinkEdge.bind(this), {
        execute: Nodes.EXECUTE,
        reflect: Nodes.REFLECT,
        "__end__": Nodes.END
      })
      .addConditionalEdges(Nodes.EXECUTE, this.executeEdge.bind(this), {
        execute: Nodes.EXECUTE,
        reflect: Nodes.REFLECT,
        "__end__": Nodes.END
      })
      .addConditionalEdges(Nodes.REFLECT, this.reflectEdge.bind(this), {
        think: Nodes.THINK,
        "__end__": Nodes.END
      });

    const compiled = workflow.compile({
      recursionLimit: ABSOLUTE_MAX_ITERATIONS  // Increased from default 25 to handle complex tasks
    });
    return compiled;
  }

  /**
   * Get current domain from URL
   */
  getCurrentDomain() {
    try {
      // Try to get from tab URL if available
      if (this.tabId && typeof chrome !== 'undefined' && chrome.tabs) {
        // This would need to be async, but for now return a placeholder
        return 'unknown';
      }
      // Try to get from window.location if in browser context
      if (typeof window !== 'undefined' && window.location) {
        return window.location.hostname;
      }
      return 'unknown';
    } catch (error) {
      return 'unknown';
    }
  }

  /**
   * Extract action history from messages
   */
  extractActionsFromMessages(messages) {
    const actions = [];

    for (const msg of messages) {
      // Look for tool calls in AIMessage objects
      if (msg.tool_calls && Array.isArray(msg.tool_calls)) {
        for (const toolCall of msg.tool_calls) {
          actions.push({
            tool: toolCall.name,
            args: toolCall.args || {},
            result: '', // Will be populated from ToolMessage
            success: true
          });
        }
      }

      // Look for tool results in ToolMessage objects
      if (msg.constructor.name === 'ToolMessage' || msg.name) {
        const lastAction = actions[actions.length - 1];
        if (lastAction && lastAction.tool === msg.name) {
          lastAction.result = msg.content || '';
          // Check if the result indicates failure
          if (msg.content && (msg.content.includes('Error') || msg.content.includes('Failed'))) {
            lastAction.success = false;
          }
        }
      }
    }

    return actions;
  }

  /**
   * Run the execute-reflect agent
   */
  async run(userRequest, options = {}) {
    // Store userRequest as class field for easy access
    this.userRequest = userRequest;

    const graph = this.buildGraph();

    const initialState = {
      messages: [new HumanMessage(userRequest)],
      maxIterations: options.maxIterations || DEFAULT_MAX_ITERATIONS,
    };

    // Stream execution with increased recursion limit
    const stream = await graph.stream(initialState, {
      recursionLimit: RECURSION_LIMIT
    });

    let finalState = initialState;
    let executionError = null;

    try {
      for await (const event of stream) {
        if (this.isAborted()) {
          // Break execution but let ai_agent.js emit AGENT_STOPPED
          break;
        }

        // Update final state - properly merge nested objects
        for (const [key, value] of Object.entries(event)) {
          if (value !== undefined) {
            // Merge the state from each node
            finalState = { ...finalState, ...value };
          }
        }
      }
    } catch (error) {
      console.error('[AGENT] Unhandled error during agent execution:', error);
      executionError = error;
    }


    // Format the result like other agents
    const lastMessage = finalState.messages?.[finalState.messages.length - 1];
    let output = "Task execution completed.";
    let reason = "completed";

    if (executionError) {
      output = `Agent stopped due to an error: ${executionError.message}`;
      reason = "error";
    } else if (this.isAborted()) {
      output = "Operation cancelled by user.";
      reason = "user_stopped";
    } else if (finalState.taskComplete) {
      output = "Task completed successfully.";
      reason = "completed";
    } else if (finalState.iterations >= finalState.maxIterations) {
      output = `Reached maximum iterations (${finalState.maxIterations}). Task may not be fully complete.`;
      reason = "max_iterations";
    }

    // Include the last reflection if available
    if (finalState.reflect?.messages?.length > 0) {
      const lastReflection = finalState.reflect.messages[finalState.reflect.messages.length - 1];
      if (lastReflection?.content) {
        output += `\n\nFinal reflection: ${lastReflection.content}`;
      }
    }

    // Emit final token usage statistics
    if (finalState.tokenUsage) {
      this.#emitEvent(EVENT_TYPES.TOKEN_USAGE, {
        input_tokens: finalState.tokenUsage.input_tokens || 0,
        output_tokens: finalState.tokenUsage.output_tokens || 0,
        total_tokens: finalState.tokenUsage.total_tokens || 0,
        iterations: finalState.iterations || 0
      });
      console.log(`[TOKEN USAGE] Total: ${finalState.tokenUsage.total_tokens} tokens (${finalState.tokenUsage.input_tokens} in, ${finalState.tokenUsage.output_tokens} out) over ${finalState.iterations} iterations`);
    }

    // Note: AGENT_STOPPED event is emitted by ai_agent.js after workflow completes
    // This ensures consistent event emission from a single source of truth

    return {
      completed: finalState.taskComplete || false,
      output: output,
      reason: reason,
      iterations: finalState.iterations || 0,
      tokenUsage: finalState.tokenUsage
    };
  }

  /**
   * Clean up incomplete tool calls from the message history
   * This is called when the agent is stopped to prevent "tool_call_id" errors
   * Note: ExecuteReflectGraph doesn't accumulate messages in state like ReactGraph,
   * so this is a no-op for compatibility
   */
  cleanupIncompleteToolCalls() {
    // ExecuteReflectGraph doesn't maintain a persistent message history
    // Each execution/reflection cycle uses fresh context
    console.log('🤖 ExecuteReflectGraph: No message cleanup needed (stateless design)');
  }
}
