// @ts-nocheck

// AgentStatusCallbackHandler.js
// Custom callback handler for agent status updates

import { BaseCallbackHandler } from "@langchain/core/callbacks/base";
import { EVENT_TYPES } from "./event-types.js";

/**
 * Custom callback handler that extracts and reports agent thoughts and tool usage
 * Based on OpenAI o3 recommendations for proper AgentExecutor callback integration
 */
export class AgentStatusCallbackHandler extends BaseCallbackHandler {
  constructor(tabId, onEvent) {
    super();
    this.tabId = tabId;
    this.onEvent = onEvent || (() => {});
    this.name = "AgentStatusCallbackHandler";
  }
  
  /**
   * Emit an event through the unified event system
   */
  emitEvent(type, data) {
    const event = {
      type,
      tabId: this.tabId,
      data,
      timestamp: Date.now()
    };
    this.onEvent(event);
  }

  /**
   * Called when the agent performs an action (tool call)
   * The action.log contains the free-form reasoning from the agent
   */
  async handleAgentAction(action) {
    // Extract thought from the agent's reasoning log for internal processing
    if (action.log) {
      console.log("[Agent Reasoning]", action.log);
    }
  }

  /**
   * Called when a tool starts execution
   */
  async handleToolStart(tool, input) {
    // No-op - tool execution is tracked via messages in AI_UPDATE
  }

  /**
   * Called when a tool completes successfully
   */
  async handleToolEnd(output, tool) {
    // No-op - tool results are tracked via messages in AI_UPDATE
  }

  /**
   * Called when a tool encounters an error
   */
  async handleToolError(error, tool) {
    console.error(`[TOOL ERROR] ${tool.name}:`, error.message);
  }

  /**
   * Called when the agent completes its execution
   */
  async handleAgentEnd(action) {
    // Extract final answer from the agent
    if (action.returnValues && action.returnValues.output) {
      await this.sendThinkingUpdate(this.tabId, {
        type: "final_answer",
        content: action.returnValues.output
      });
    }
  }

  /**
   * Called when LLM starts generating
   */
  async handleLLMStart(llm, prompts) {
    await this.sendThinkingUpdate(this.tabId, {
      type: "reasoning",
      content: "Thinking..."
    });
  }

  /**
   * Called when LLM completes generation
   */
  async handleLLMEnd(output) {
    // The actual content is handled in handleAgentAction
  }

  /**
   * Called when there's an LLM error
   */
  async handleLLMError(error) {
    console.error("❌ [CALLBACK] handleLLMError:", error);
  }
}